/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.jobmanagement.view.action;

import org.eclipse.jface.action.IAction;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.ui.IViewActionDelegate;
import org.eclipse.ui.IViewPart;

import com.clustercontrol.jobmanagement.bean.JobConstant;
import com.clustercontrol.jobmanagement.composite.JobListComposite;
import com.clustercontrol.jobmanagement.composite.JobTreeComposite;
import com.clustercontrol.jobmanagement.util.JobEditStateUtil;
import com.clustercontrol.jobmanagement.util.JobTreeItemUtil;
import com.clustercontrol.jobmanagement.view.JobListView;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.jobmanagement.JobTreeItem;

/**
 * ジョブ[一覧]ビューの「削除」のクライアント側アクションクラス<BR>
 * 
 * @version 2.0.0
 * @since 1.0.0
 */
public class DeleteJobAction implements IViewActionDelegate {
	/** アクションID */
	public static final String ID = "com.clustercontrol.jobmanagement.view.action.DeleteJobAction";
	/** ビュー */
	private IViewPart m_view;

	/**
	 * ビューを保持します。ワークベンチにロードされた際に呼ばれます。
	 * 
	 * @param view ビューのインスタンス
	 * 
	 * @see org.eclipse.ui.IViewActionDelegate#init(org.eclipse.ui.IViewPart)
	 */
	@Override
	public void init(IViewPart view) {
		m_view = view;
	}

	/**
	 * ジョブ[一覧]ビューの「削除」が押された場合に、ジョブを削除します。
	 * <p>
	 * <ol>
	 * <li>ジョブ[一覧]ビューから選択されているジョブツリーアイテムを取得します。</li>
	 * <li>ジョブ[一覧]ビューから選択されているジョブツリーアイテムの親を取得します。</li>
	 * <li>削除の確認ダイアログを表示します。</li>
	 * <li>親のジョブツリーアイテムから、選択さえたジョブツリーアイテムを削除します。</li>
	 * <li>ジョブ[一覧]ビューを更新します。</li>
	 * </ol>
	 * 
	 * @see org.eclipse.ui.IActionDelegate#run(org.eclipse.jface.action.IAction)
	 * @see com.clustercontrol.jobmanagement.composite.JobTreeComposite
	 * @see com.clustercontrol.jobmanagement.composite.JobListComposite
	 */
	@Override
	public void run(IAction action) {
		JobTreeItem item = null;
		JobTreeItem parent = null;

		if (m_view instanceof JobListView) {
			JobListView view = (JobListView) m_view
					.getAdapter(JobListView.class);
			JobTreeComposite tree = view.getJobTreeComposite();
			JobListComposite list = view.getJobListComposite();
			if (tree.getTree().isFocusControl()) {
				item = tree.getSelectItem();
				parent = item.getParent();
			} else if (list.getTable().isFocusControl()) {
				item = list.getSelectJobTreeItem();
				if (item instanceof JobTreeItem) {
					parent = item.getParent();
				}
			} else {
				return;
			}

			if (parent instanceof JobTreeItem && item instanceof JobTreeItem) {
				// 確認ダイアログを生成
				String message = Messages.getString("job") + "["
						+ item.getData().getId() + "]"
						+ Messages.getString("message.job.2");
				if (MessageDialog.openQuestion(
						null,
						Messages.getString("confirmed"),
						message)) {
					JobTreeItemUtil.removeChildren(parent, item);

					if (item.getData().getType() == JobConstant.TYPE_JOBUNIT) {
						// ジョブユニットの削除
						JobEditStateUtil.removeEditedJobunit(item);
						if (JobEditStateUtil.getLockedJobunitBackup(item.getData()) != null) {
							// マネージャから取得してきたジョブユニット
							JobEditStateUtil.addDeletedJobunit(item);
						}
					} else {
						// ジョブユニット以外の削除はジョブユニットの編集にあたる
						JobEditStateUtil.addEditedJobunit(item);
					}

					tree.refresh(parent);
					tree.getTreeViewer().setSelection(
							new StructuredSelection(parent), true);
				}
			}
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.IActionDelegate#selectionChanged(org.eclipse.jface.action.IAction, org.eclipse.jface.viewers.ISelection)
	 */
	@Override
	public void selectionChanged(IAction action, ISelection selection) {

	}
}