/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.collectiverun.factory;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.sql.Timestamp;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Locale;

import javax.persistence.EntityExistsException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.CollectiveRunNotFound;
import com.clustercontrol.fault.FacilityNotFound;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.bean.EndStatusConstant;
import com.clustercontrol.collectiverun.bean.CollectiveRunInfo;
import com.clustercontrol.collectiverun.bean.CollectiveRunStatusConstant;
import com.clustercontrol.collectiverun.bean.CommandTypeConstant;
import com.clustercontrol.collectiverun.bean.ParameterTypeConstant;
import com.clustercontrol.collectiverun.message.CollectiveRunInstructionInfo;
import com.clustercontrol.collectiverun.model.CrunCmdMstEntity;
import com.clustercontrol.collectiverun.model.CrunCmdParamMstEntity;
import com.clustercontrol.collectiverun.model.CrunEndMstEntity;
import com.clustercontrol.collectiverun.model.CrunParamSelectMstEntity;
import com.clustercontrol.collectiverun.model.CrunSessionDetailEntity;
import com.clustercontrol.collectiverun.model.CrunSessionEntity;
import com.clustercontrol.collectiverun.model.CrunSessionParamEntity;
import com.clustercontrol.collectiverun.util.QueryUtil;
import com.clustercontrol.commons.util.JpaTransactionManager;
import com.clustercontrol.jobmanagement.bean.RunStatusConstant;
import com.clustercontrol.jobmanagement.bean.RunResultInfo;
import com.clustercontrol.plugin.impl.AsyncWorkerPlugin;
import com.clustercontrol.repository.bean.NodeInfo;
import com.clustercontrol.repository.bean.ScopeInfo;
import com.clustercontrol.repository.session.RepositoryControllerBean;
import com.clustercontrol.util.Messages;
import com.clustercontrol.util.XMLUtil;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * ジョブ操作クラス
 *
 * @version 2.1.1
 * @since 1.0.0
 */
public class OperationRun {
	private static Log m_log = LogFactory.getLog( OperationRun.class );
	private static String SHELL_COMMAND = "sh";

	/**
	 * コマンド実行
	 * 
	 * @param sessionId
	 * @param property
	 * @param locale
	 * @throws CollectiveRunNotFound
	 * @throws IOException
	 */
	public void run(String sessionId) throws CollectiveRunNotFound, IOException {
		m_log.debug("run() : sessionId=" + sessionId);

		//セッションIDからセッション(実行履歴)を取得
		CrunSessionEntity session = QueryUtil.getCrunSessionPK(sessionId);

		//実行コマンド取得
		String command = getCommand(session);

		//ステータスが待機中の場合
		if(session.getStatus().intValue() == CollectiveRunStatusConstant.TYPE_WAITING){
			//ステータスを実行中にする
			session.setStatus(Integer.valueOf(CollectiveRunStatusConstant.TYPE_RUNNING));
			//開始日時を設定
			session.setStartDate(new Timestamp(new Date().getTime()));
		}

		//ノードへ実行
		startNode(session, command);
	}

	/**
	 * ノードへの実行処理
	 * 
	 * @param sessionId
	 * @param facilityId
	 * @param command
	 * @return
	 * @throws CollectiveRunNotFound
	 */
	private void startNode(CrunSessionEntity session, String command) throws CollectiveRunNotFound {
		m_log.debug("startNode() start: sessionId=" + session.getSessionId());

		//セッションID取得
		String sessionId = session.getSessionId();

		//コマンドIDからコマンドマスタを取得
		CrunCmdMstEntity cmd = QueryUtil.getCrunCmdMstPK(session.getCommandId());

		//コマンドマスタからコマンドタイプを取得
		int commandType = cmd.getCommandType().intValue();

		String password = null;
		try {
			CrunSessionParamEntity param = QueryUtil.getCrunSessionParamPK(sessionId, "PASSWORD");
			password = param.getParamValue();
		} catch (CollectiveRunNotFound e) {
			// なにもしない
		}

		//セッションディテール(実行履歴詳細)を取得
		Collection<CrunSessionDetailEntity> collection = session.getCrunSessionDetailEntities();
		if(collection != null){
			for (CrunSessionDetailEntity sessionDetail : collection) {

				//実行情報を作成
				CollectiveRunInstructionInfo info = new CollectiveRunInstructionInfo();
				info.setSessionId(sessionId);
				info.setFacilityId(sessionDetail.getId().getFacilityId());
				if(commandType == CommandTypeConstant.TYPE_SHELL){
					info.setCommand(SHELL_COMMAND);
					info.setInputFile(command);
				}
				else{
					info.setCommand(command);
					info.setInputFile("");
				}
				info.setCommandType(com.clustercontrol.jobmanagement.bean.CommandTypeConstant.NORMAL);

				info.setPassword(password);

				try {
					//Queueに送信
					AsyncWorkerPlugin.addTask(CollectiveRunExecuteTaskFactory.class.getSimpleName(), info, true);
				} catch (Exception e) {
					AplLogger apllog = new AplLogger("CR","cr");
					String[] args = {info.getSessionId(), info.getFacilityId()};
					apllog.put("SYS","003", args);
					m_log.warn("startNode() CollectiveRunInstructionInfo send error  : "
							+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
				}
			}
		}

		m_log.debug("startNode() end: sessionId=" + session.getSessionId());
	}

	/**
	 * ノード終了処理
	 * 
	 * @param info
	 * @throws CollectiveRunNotFound
	 */
	public void endNode(RunResultInfo info) throws CollectiveRunNotFound {
		m_log.info("endNode() start: sessionId=" + info.getSessionId() + ", facilityId=" + info.getFacilityId() + ", status=" + info.getStatus());

		//セッションIDからセッション(実行履歴)を取得
		CrunSessionEntity session =
				QueryUtil.getCrunSessionPK(info.getSessionId());
		// Ingnore Invalid XML Chars
		info.setMessage(XMLUtil.ignoreInvalidString(info.getMessage()));
		info.setErrorMessage(XMLUtil.ignoreInvalidString(info.getErrorMessage()));

		//コマンドタイプチェック
		if(info.getCommandType() == com.clustercontrol.jobmanagement.bean.CommandTypeConstant.NORMAL){
			//コマンドタイプが通常の場合

			//セッションディテール(実行履歴詳細)を取得
			CrunSessionDetailEntity sessionDetail
			= QueryUtil.getCrunSessionDetailPK(info.getSessionId(), info.getFacilityId());

			//ステータスで分岐
			if(info.getStatus() == RunStatusConstant.START){
				//開始の場合

				if(sessionDetail.getStartDate() == null){
					//ステータスを実行中にする
					sessionDetail.setStatus(Integer.valueOf(CollectiveRunStatusConstant.TYPE_RUNNING));

					//開始日時を設定
					sessionDetail.setStartDate(new Timestamp(info.getTime()));

					//実行中ノード数をインクリメント
					session.setRunningCnt(Integer.valueOf(session.getRunningCnt().intValue() + 1));
					//待機中ノード数をデクリメント
					session.setWaitCnt(Integer.valueOf(session.getWaitCnt().intValue() - 1));
				}
				else{
					AplLogger apllog = new AplLogger("CR","cr");
					String[] args = {info.getSessionId(), info.getFacilityId()};
					apllog.put("SYS","004", args);
					m_log.info("endNode() this messsage is already received. drop message. " +
							" SessionId=" + info.getSessionId() +
							", FacilityId=" + info.getFacilityId());
				}
				sessionDetail = null;
			}
			else{
				//ステータスが実行中の場合
				if(sessionDetail.getStatus().intValue() == CollectiveRunStatusConstant.TYPE_RUNNING){
					//実行中ノード数をデクリメント
					session.setRunningCnt(Integer.valueOf(session.getRunningCnt().intValue() - 1));
				}

				if(info.getStatus() == RunStatusConstant.END){
					//終了の場合

					//ステータスが実行中の場合
					if(sessionDetail.getStatus().intValue() == CollectiveRunStatusConstant.TYPE_RUNNING){
						//ステータスに終了を設定
						sessionDetail.setStatus(Integer.valueOf(CollectiveRunStatusConstant.TYPE_END));
						//終了日時を設定
						sessionDetail.setEndDate(new Timestamp(info.getTime()));
						//終了値を設定
						sessionDetail.setEndValue(Integer.valueOf(info.getEndValue()));
						//メッセージを設定
						sessionDetail.setMessage(info.getMessage());
						//エラーメッセージを設定
						sessionDetail.setErrorMessage(info.getErrorMessage());

						//終了状態を設定
						setEndStatus(session, sessionDetail);
					} else {
						m_log.warn("detailStatus : from " + sessionDetail.getStatus() + " to END");
					}
				}
				else if(info.getStatus() == RunStatusConstant.ERROR){
					//失敗の場合

					//ステータスが実行中の場合
					if(sessionDetail.getStatus().intValue() == CollectiveRunStatusConstant.TYPE_RUNNING){
						//ステータスに終了を設定
						sessionDetail.setStatus(Integer.valueOf(CollectiveRunStatusConstant.TYPE_ERROR));
						//メッセージを設定
						sessionDetail.setMessage(info.getMessage());
						//エラーメッセージを設定
						sessionDetail.setErrorMessage(info.getErrorMessage());

						//失敗ノード数をインクリメント
						session.setEndAbnomalCnt(Integer.valueOf(session.getEndAbnomalCnt().intValue() + 1));
					} else {
						m_log.warn("detailStatus : from " + sessionDetail.getStatus() + " to ERROR");
					}
				}
				sessionDetail = null;


				//終了チェック
				if(checkAllNodeEnd(session)){
					//全ノード終了の場合

					//ノード数設定
					setNodeCount(session);

					//ステータスが実行中の場合
					if(session.getStatus().intValue() == CollectiveRunStatusConstant.TYPE_RUNNING){
						//ステータスに終了を設定
						session.setStatus(Integer.valueOf(CollectiveRunStatusConstant.TYPE_END));
						//終了日時を設定
						session.setEndDate(new Timestamp(info.getTime()));
					}
				}
			}
		}

		m_log.debug("endNode() end: sessionId=" + info.getSessionId() + ", facilityId=" + info.getFacilityId() + ", status=" + info.getStatus());
	}

	/**
	 * ノード数設定処理
	 * 
	 * @param sessionId
	 * @param facilityId
	 */
	private void setNodeCount(CrunSessionEntity session) {
		m_log.debug("setNodeCount() : sessionId=" + session.getSessionId());

		int normal = 0;
		int warning = 0;
		int abnormal = 0;
		int running = 0;
		int waiting = 0;

		//セッションディテール(実行履歴詳細)を取得
		Collection<CrunSessionDetailEntity> collection = session.getCrunSessionDetailEntities();
		if(collection != null){
			for(CrunSessionDetailEntity sessionDetail : collection) {

				//ステータスをチェック
				if(sessionDetail.getStatus().intValue() == CollectiveRunStatusConstant.TYPE_RUNNING){
					//実行中
					running++;
				}
				else if(sessionDetail.getStatus().intValue() == CollectiveRunStatusConstant.TYPE_WAITING){
					//待機中
					waiting++;
				}
				else if(sessionDetail.getStatus().intValue() == CollectiveRunStatusConstant.TYPE_ERROR){
					//実行失敗
					abnormal++;
				}

				//終了状態をチェック
				if(sessionDetail.getEndStatus() != null){
					if(sessionDetail.getEndStatus().intValue() == EndStatusConstant.TYPE_NORMAL){
						//実行中
						normal++;
					}
					else if(sessionDetail.getEndStatus().intValue() == EndStatusConstant.TYPE_WARNING){
						//待機中
						warning++;
					}
					else if(sessionDetail.getEndStatus().intValue() == EndStatusConstant.TYPE_ABNORMAL){
						//実行失敗
						abnormal++;
					}
				}
			}
		}

		//実行中の数を設定
		session.setRunningCnt(Integer.valueOf(running));
		//待機中の数を設定
		session.setWaitCnt(Integer.valueOf(waiting));
		//正常の数を設定
		session.setEndNomalCnt(Integer.valueOf(normal));
		//警告の数を設定
		session.setEndWarningCnt(Integer.valueOf(warning));
		//異常の数を設定
		session.setEndAbnomalCnt(Integer.valueOf(abnormal));
	}

	/**
	 * 終了状態設定処理
	 * 
	 * @param sessionId
	 * @param facilityId
	 */
	private void setEndStatus(CrunSessionEntity session, CrunSessionDetailEntity sessionDetail) {
		m_log.debug("setEndStatus() : sessionId=" + session.getSessionId() + ", facilityId=" + sessionDetail.getId().getFacilityId());

		//終了状態（正常）を取得
		CrunEndMstEntity endInfoNormal = null;
		try {
			endInfoNormal = QueryUtil.getCrunEndMstPK(session.getCommandId(), EndStatusConstant.TYPE_NORMAL);
		} catch (CollectiveRunNotFound e) {
		}

		//終了状態（警告）を取得
		CrunEndMstEntity endInfoWarning = null;
		try {
			endInfoWarning = QueryUtil.getCrunEndMstPK(session.getCommandId(), EndStatusConstant.TYPE_WARNING);
		} catch (CollectiveRunNotFound e) {
		}

		Integer endValue = sessionDetail.getEndValue();
		if(endValue != null){
			if(endInfoNormal != null &&
					endValue.compareTo(endInfoNormal.getEndValueFrom()) >= 0 &&
					endValue.compareTo(endInfoNormal.getEndValueTo()) <= 0){
				//終了状態（正常）の範囲内ならば、正常とする
				sessionDetail.setEndStatus(Integer.valueOf(EndStatusConstant.TYPE_NORMAL));
				//成功ノード数をインクリメント
				session.setEndNomalCnt(Integer.valueOf(session.getEndNomalCnt().intValue() + 1));
			}
			else if(endInfoWarning != null &&
					endValue.compareTo(endInfoWarning.getEndValueFrom()) >= 0 &&
					endValue.compareTo(endInfoWarning.getEndValueTo()) <= 0){
				//終了状態（警告）の範囲内ならば、警告とする
				sessionDetail.setEndStatus(Integer.valueOf(EndStatusConstant.TYPE_WARNING));
				//警告ノード数をインクリメント
				session.setEndWarningCnt(Integer.valueOf(session.getEndWarningCnt().intValue() + 1));
			}
			else{
				//終了状態（異常）の範囲内ならば、異常とする
				sessionDetail.setEndStatus(Integer.valueOf(EndStatusConstant.TYPE_ABNORMAL));
				//失敗ノード数をインクリメント
				session.setEndAbnomalCnt(Integer.valueOf(session.getEndAbnomalCnt().intValue() + 1));
			}
		}
	}

	/**
	 * 全ノード終了チェック処理
	 * 
	 * @param session
	 * @return
	 */
	private boolean checkAllNodeEnd(CrunSessionEntity session) {
		m_log.debug("checkAllNodeEnd() start: sessionId=" + session.getSessionId());

		//終了フラグをfalseにする
		boolean end = false;

		//ステータスが実行中または待機中が存在するかチェック
		boolean runNode = false;
		if(session.getRunningCnt().intValue() > 0 || session.getWaitCnt().intValue() > 0){
			runNode = true;
		}

		//全ノード終了の場合
		if(!runNode){
			//終了フラグをtrueにする
			end = true;
		}

		m_log.debug("checkAllNodeEnd() end: sessionId=" + session.getSessionId());
		return end;
	}

	/**
	 * コマンド取得処理
	 * 
	 * @param sessionId
	 * @return
	 * @throws CollectiveRunNotFound
	 * @throws IOException
	 */
	private String getCommand(CrunSessionEntity session) throws CollectiveRunNotFound, IOException {
		m_log.debug("getCommand() : sessionId=" + session.getSessionId());
		String command = null;

		//コマンドIDからコマンドマスタを取得
		CrunCmdMstEntity cmd = QueryUtil.getCrunCmdMstPK(session.getCommandId());

		//------------------------------
		//コマンドの引数を作成する
		//------------------------------
		StringBuffer argument = new StringBuffer();
		//コマンドマスタからコマンドパラメータを取得
		Collection<CrunCmdParamMstEntity> cmdPramList = cmd.getCrunCmdParamMstEntities();
		if(cmdPramList != null){
			for (CrunCmdParamMstEntity cmdParam : cmdPramList) {
				//セッションパラメータ(実行履歴パラメータ)を取得
				CrunSessionParamEntity sessionParam
				= QueryUtil.getCrunSessionParamPK(session.getSessionId(), cmdParam.getId().getParamId());

				if(sessionParam.getParamValue() != null &&
						sessionParam.getParamValue().length() > 0){
					if(cmdParam.getParamPrefix() != null &&
							cmdParam.getParamPrefix().length() > 0){
						//Prefixがnullまたは空白文字の場合、Prefixは付けない
						argument.append(cmdParam.getParamPrefix());
						argument.append(" ");
					}
					//パラメータ値がnullまたは空白文字の場合、パラメータ値はつけない
					argument.append(sessionParam.getParamValue());
					argument.append(" ");
				}
			}
		}

		if(cmd.getCommandType().intValue() == CommandTypeConstant.TYPE_SHELL){
			//コマンドタイプがシェルの場合、ファイル内容の前に'set - 'と引数を追加する
			command = "set - " + argument.toString() + "\n" + readShellFile(cmd.getCommand());
		}
		else{
			//コマンドタイプがシェルの場合、コマンドの後に引数を追加する
			command = cmd.getCommand() + " " + argument.toString();
		}

		m_log.debug("getCommand() : command=" + command);
		return command;
	}

	/**
	 * シェルファイル読み込み処理
	 * 
	 * @param fileName
	 * @return ファイル文字列
	 * @throws IOException
	 */
	private String readShellFile(String fileName) throws IOException {
		m_log.debug("readShellFile() : fileName=" + fileName);

		StringBuffer filePath = new StringBuffer();
		filePath.append(fileName);

		FileInputStream fileInputStream = null;
		InputStreamReader inputStreamReader = null;
		BufferedReader bufferedReader=null;
		String lineData;
		StringBuffer fileData = new StringBuffer();
		try {
			fileInputStream = new FileInputStream(filePath.toString());
			inputStreamReader = new InputStreamReader(fileInputStream);
			bufferedReader = new BufferedReader(inputStreamReader);
			//一行ずつ全て読み込むまで繰り返す
			while ((lineData = bufferedReader.readLine())!=null){
				fileData.append(lineData);
				//改行コードを追加
				fileData.append("\n");
			}
		} catch (FileNotFoundException e) {
			AplLogger apllog = new AplLogger("CR","cr");
			String[] args = {fileName};
			apllog.put("SYS","005", args);
			m_log.info("readShellFile() : fileName=" + fileName + " is not found : " + e.getMessage());
			throw e;
		} catch (IOException e) {
			AplLogger apllog = new AplLogger("CR","cr");
			String[] args = {fileName};
			apllog.put("SYS","006", args);
			m_log.info("readShellFile() : fileName=" + fileName + " IO error : " + e.getMessage());
			throw e;
		} finally {
			try{
				if(bufferedReader!=null){
					bufferedReader.close();
				}
				if(inputStreamReader!=null){
					inputStreamReader.close();
				}
				if(fileInputStream!=null){
					fileInputStream.close();
				}
			}catch(IOException e){
			}
		}

		return fileData.toString();
	}

	/**
	 * 一括制御の実行履歴情報を作成します。
	 * 
	 * @param typeId
	 * @param info
	 * @throws EntityExistsException
	 * @throws FacilityNotFound
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * 
	 */
	public String createSession(String typeId, CollectiveRunInfo info)
			throws FacilityNotFound, HinemosUnknown, EntityExistsException, InvalidRole {
		m_log.debug("createSession() : typeId=" + typeId);

		JpaTransactionManager jtm = new JpaTransactionManager();

		//リポジトリセッションBean取得
		RepositoryControllerBean repository;
		repository = new RepositoryControllerBean();

		//ファシリティID（スコープ）取得
		String facilityId = null;
		if(info.getFacilityId() instanceof String && info.getFacilityId().length() > 0){
			facilityId = info.getFacilityId();
		}

		//名前取得
		String commandId = null;
		if(info.getName() instanceof String && info.getName().length() > 0){
			String nameString = info.getName();
			//名前からコマンドIDを取得する
			commandId = getCommandId(typeId, nameString);
		}

		//コマンドIDからコマンドマスタを取得
		CrunCmdMstEntity cmd = null;
		try {
			cmd = QueryUtil.getCrunCmdMstPK(commandId);
		} catch (CollectiveRunNotFound e) {
			FacilityNotFound e1 = new FacilityNotFound("CrunCmdMstEntity.findByPrimaryKey, "
					+ "commandId = " + commandId);
			throw e1;
		}

		//セッションを作成
		boolean createFlg = false;
		String sessionId = null;
		Date createTime = new Date();
		CrunSessionEntity session = null;
		for(int i = 0; !createFlg; i++){
			try {
				SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMddHHmmss");
				sessionId = dateFormat.format(createTime);

				DecimalFormat format = new DecimalFormat("-000");
				sessionId = sessionId + format.format(i);
				String scopeText = repository.getFacilityPath(facilityId, null);

				//セッション(実行履歴)を作成
				session = new CrunSessionEntity(sessionId);
				// 重複チェック
				jtm.checkEntityExists(CrunSessionEntity.class, session.getSessionId());
				session.setStartDate(null);
				session.setEndDate(null);
				session.setTypeId(cmd.getCrunTypeMstEntity() == null ? null : cmd.getCrunTypeMstEntity().getTypeId());
				session.setCommandId(cmd.getCommandId());
				session.setFacilityId(facilityId);
				session.setScopeText(scopeText);
				session.setNodeCnt(0);
				session.setWaitCnt(0);
				session.setRunningCnt(0);
				session.setEndNomalCnt(0);
				session.setEndWarningCnt(0);
				session.setEndAbnomalCnt(0);
				session.setStatus(CollectiveRunStatusConstant.TYPE_WAITING);
				createFlg = true;
			} catch (EntityExistsException e) {
				//プライマリーキー重複
				//再度セッション(実行履歴)の作成を行う
			}
		}

		//対象ファシリティIDからセッションディテール(実行履歴詳細)を作成
		ArrayList<String> nodeIdList = repository.getExecTargetFacilityIdList(session.getFacilityId(), null);

		try {
			if(nodeIdList != null & nodeIdList.size() > 0){
				for(int i = 0; i < nodeIdList.size(); i++){
					String facilityName = null;
					if (repository.isNode(facilityId)) {
						NodeInfo nodeInfo = repository.getNode(facilityId);
						facilityName = nodeInfo.getFacilityName();
					} else {
						ScopeInfo scopeInfo = repository.getScope(facilityId);
						facilityName = scopeInfo.getFacilityName();
					}

					//セッションディテール(実行履歴詳細)を作成
					CrunSessionDetailEntity sessionDetail
					= new CrunSessionDetailEntity(session, nodeIdList.get(i));
					// 重複チェック
					jtm.checkEntityExists(CrunSessionDetailEntity.class, sessionDetail.getId());
					sessionDetail.setStartDate(null);
					sessionDetail.setEndDate(null);
					sessionDetail.setFacilityName(facilityName);
					sessionDetail.setMessage(null);
					sessionDetail.setErrorMessage(null);
					sessionDetail.setStatus(Integer.valueOf(CollectiveRunStatusConstant.TYPE_WAITING));
					sessionDetail.setEndStatus(null);
					sessionDetail.setEndValue(null);
				}
				//ノード数設定
				session.setNodeCnt(Integer.valueOf(nodeIdList.size()));
				//待機中ノード数設定
				session.setWaitCnt(Integer.valueOf(nodeIdList.size()));
			}

			//コマンドマスタからコマンドパラメータを取得
			Collection<CrunCmdParamMstEntity> cmdPramList = cmd.getCrunCmdParamMstEntities();
			if(cmdPramList != null){
				for(CrunCmdParamMstEntity cmdParam : cmdPramList){
					int paramType = cmdParam.getCrunParamMstEntity().getParamType().intValue();

					//パラメータ値取得
					String value = null;
					String paramId = cmdParam.getId().getParamId();
					if(info.getParamValue(paramId) instanceof String && info.getParamValue(paramId).length() > 0){
						value = info.getParamValue(paramId);
					}

					//パラメータタイプが選択の場合、表示文字列からパラメータ値を取得する
					if(paramType == ParameterTypeConstant.TYPE_SELECT){
						value = getParamSelectValue(paramId, value, Locale.getDefault());
					}

					//セッションパラメータ(実行履歴パラメータ)を作成
					// インスタンス生成
					CrunSessionParamEntity sessionParam = new CrunSessionParamEntity(session, paramId);
					// 重複チェック
					jtm.checkEntityExists(CrunSessionParamEntity.class, sessionParam.getId());
					sessionParam.setParamValue(value);
				}
			}
		} catch (EntityExistsException e) {
			m_log.info("createSession() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage());
			throw e;
		}

		return sessionId;
	}

	/**
	 * コマンドID取得
	 * 
	 * 表示文字列からコマンドIDを取得する
	 * 
	 * @param typeId 種別ID
	 * @param name 表示文字列
	 * @return コマンドID
	 */
	public static String getCommandId(String typeId, String name) {
		m_log.debug("getCommandId() : typeId=" + typeId + ", name=" + name);

		String commandId = null;

		//種別IDからコマンドマスタを取得
		List<CrunCmdMstEntity> cmdList = QueryUtil.getCrunCmdMstByTypeId(typeId);

		for (CrunCmdMstEntity cmd : cmdList) {
			//表示項目IDからリソースを取得
			String nameValue = Messages.getString(cmd.getNameId(), Locale.getDefault());
			//リソースが一致したらコマンドIDを返す
			if(nameValue.equals(name)){
				commandId = cmd.getCommandId();
				break;
			}
		}

		return commandId;
	}

	/**
	 * パラメータ選択値取得
	 * 
	 * 表示文字列からパラメータ選択値を取得する
	 * 
	 * @param paramId パラメータID
	 * @param name 表示文字列
	 * @param locale ロケール
	 * @return パラメータ選択値
	 */
	private String getParamSelectValue(String paramId, String name, Locale locale) {
		m_log.debug("getParamSelectValue() : paramId=" + paramId + ", name=" + name);

		String value = null;

		//パラメータIDからパラメータ選択マスタを取得
		List<CrunParamSelectMstEntity> paramSelectList
			= QueryUtil.getCrunParamSelectMstByParamId(paramId);

		for (CrunParamSelectMstEntity paramSelect : paramSelectList) {
			//表示項目IDからリソースを取得
			String nameValue = Messages.getString(paramSelect.getNameId(), locale);
			//リソースが一致したらコマンドIDを返す
			if(nameValue.equals(name)){
				value = paramSelect.getParamValue();
				break;
			}
		}

		return value;
	}

	/**
	 * パラメータ選択値取得
	 * 
	 * 表示文字列からパラメータ選択値を取得する
	 * 
	 * @param paramId パラメータID
	 * @param name 表示文字列
	 * @param locale ロケール
	 * @return パラメータ選択値
	 */
	public void checkEndSession() {
		m_log.debug("checkEndSession()");

		//実行中のセッションを取得
		List<CrunSessionEntity> collection
		= QueryUtil.getCrunSessionByStatus(CollectiveRunStatusConstant.TYPE_RUNNING);

		for (CrunSessionEntity session : collection) {

			//ノード数設定
			setNodeCount(session);

			//終了チェック
			if(checkAllNodeEnd(session)){
				//全ノード終了の場合

				//ステータスが実行中の場合
				if(session.getStatus().intValue() == CollectiveRunStatusConstant.TYPE_RUNNING){
					//ステータスに終了を設定
					session.setStatus(Integer.valueOf(CollectiveRunStatusConstant.TYPE_END));
					//終了日時を設定
					session.setEndDate(new Timestamp((new Date()).getTime()));
				}
			}
		}
	}
}
