/*
Copyright (C) 2014 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.factory;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import javax.persistence.EntityExistsException;
import javax.persistence.NoResultException;
import javax.persistence.Query;

import com.clustercontrol.accesscontrol.bean.RoleIdConstant;
import com.clustercontrol.cloud.CloudManagerFault;
import com.clustercontrol.cloud.ErrorCode;
import com.clustercontrol.cloud.ICloudOption;
import com.clustercontrol.cloud.MessagesHolder;
import com.clustercontrol.cloud.SessionService;
import com.clustercontrol.cloud.bean.CloudAccountResource;
import com.clustercontrol.cloud.bean.CloudService;
import com.clustercontrol.cloud.bean.CloudUser;
import com.clustercontrol.cloud.bean.CreateAccountResourceRequest;
import com.clustercontrol.cloud.bean.CreateCloudUserRequest;
import com.clustercontrol.cloud.bean.ModifyAccountBillingDetailRequest;
import com.clustercontrol.cloud.bean.ModifyAccountResourceRequest;
import com.clustercontrol.cloud.dao.CloudAccountResourceDao;
import com.clustercontrol.cloud.dao.CloudUserDao;
import com.clustercontrol.cloud.persistence.EntityManagerEx;
import com.clustercontrol.cloud.persistence.Transactional;
import com.clustercontrol.cloud.registry.ObjectRegistryService;
import com.clustercontrol.cloud.util.AccountResourceUtil;
import com.clustercontrol.cloud.util.CloudCredential;
import com.clustercontrol.cloud.util.RepositoryControllerBeanWrapper;
import com.clustercontrol.fault.InvalidRole;

@Transactional
public class AccountResourceOperator implements IAccountResourceOperator {
	@Override
	public CloudAccountResource createCloudAccountResource(CreateAccountResourceRequest request) throws CloudManagerFault, InvalidRole {
		CloudAccountResourceDao dao = new CloudAccountResourceDao();
		dao.setAccountResourceId(request.getAccountResourceId());
		dao.setAccountResourceName(request.getAccountResourceName());
		dao.setCloudServiceId(request.getCloudServiceId());
		dao.setDescription(request.getDescription());
		dao.setRetentionPeriod(31);
		Calendar beginning = Calendar.getInstance();
		beginning.setTime(new Date());
		beginning.set(Calendar.HOUR_OF_DAY, 0);
		beginning.set(Calendar.MINUTE, 0);
		beginning.set(Calendar.SECOND, 0);
		beginning.set(Calendar.MILLISECOND, 0);
		dao.setBillingLastDate(new Timestamp(beginning.getTime().getTime()));

		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		try {
			em.persist(dao);
		}
		catch (EntityExistsException e) {
			throw ErrorCode.ACCOUNTRESOURCE_ALREADY_EXIST.cloudManagerFault(dao.getAccountResourceId());
		}

		ICloudServiceOperator serviceOperator = ObjectRegistryService.registry().get(ICloudServiceOperator.class);
		CloudService service = serviceOperator.findCloudService(request.getCloudServiceId());
		
		// アカウントを作成するより先にツリー構造を用意する。アカウントを作成した際のオブジェクト権限の追加が失敗するので。
		AccountResourceUtil.buildAccountResourceHierarchy(
			RepositoryControllerBeanWrapper.bean(),
			dao.getAccountResourceId(),
			dao.getAccountResourceName(),
			service.getCloudTypeId(),
			dao.getDescription(),
			dao.getCloudServiceId(),
			request.getAccount().getRoleId(),
			new MessagesHolder("com.clustercontrol.cloud.messages"));

		// アカウントの作成を行う。
		CreateCloudUserRequest userRequest = new CreateCloudUserRequest();
		userRequest.setCloudUserId(request.getAccount().getCloudUserId());
		userRequest.setCloudUserName(request.getAccount().getCloudUserName());
		userRequest.setAccountResourceId(dao.getAccountResourceId());
		userRequest.setAccessKey(request.getAccount().getAccessKey());
		userRequest.setSecretKey(request.getAccount().getSecretKey());
		userRequest.setDescription(request.getAccount().getDescription());
		userRequest.setRoleId(request.getAccount().getRoleId());

		ICloudUserOperator userOperator = ObjectRegistryService.registry().get(ICloudUserOperator.class);
		userOperator.createCloudAccount(userRequest);
		CloudUserDao userDao = em.find(CloudUserDao.class, userRequest.getCloudUserId());
		assert userDao != null;

		// 管理アカウントを設定。JPQL で更新しないと、CloudUserDao のコミットが失敗する。setAdminUser では失敗する。
		Query query = em.createQuery("UPDATE CloudAccountResourceDao a SET a.account = :user WHERE a.accountResourceId = '" + request.getAccountResourceId() + "'");
		query.setParameter("user", userDao);
		query.executeUpdate();
		
		return new CloudAccountResource(dao);
	}

	@Override
	public CloudAccountResource modifyCloudAccountResource(ModifyAccountResourceRequest request) throws CloudManagerFault {
		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		CloudAccountResourceDao dao = em.find(CloudAccountResourceDao.class, request.getAccountResourceId());
		if (dao == null) {
			throw ErrorCode.ACCOUNTRESOURCE_INVALID_ACCOUNTRESOURCE_NOT_FOUND.cloudManagerFault(request.getAccountResourceId());
		}
		
		if (request.getAccountResourceName() != null) {
			dao.setAccountResourceName(request.getAccountResourceName());
		}
		if (request.getDescription() != null) {
			dao.setDescription(request.getDescription());
		}
		
		return new CloudAccountResource(dao);
	}

	@Override
	public CloudAccountResource modifyAccountBillingDetail(ModifyAccountBillingDetailRequest request) throws CloudManagerFault {
		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		CloudAccountResourceDao dao = em.find(CloudAccountResourceDao.class, request.getAccountResourceId());
		if (dao == null) {
			throw ErrorCode.ACCOUNTRESOURCE_INVALID_ACCOUNTRESOURCE_NOT_FOUND.cloudManagerFault(request.getAccountResourceId());
		}
		
		dao.setBillingDetailCollectorFlg(request.isBillingDetailCollectorFlg());
		dao.setRetentionPeriod(request.getRetentionPeriod());
		
		return new CloudAccountResource(dao);
	}

	@Override
	public void removeCloudAccountResource(String accountResourceId) throws CloudManagerFault, InvalidRole {
		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		CloudAccountResourceDao dao = em.find(CloudAccountResourceDao.class, accountResourceId);
		if (dao == null) {
			throw ErrorCode.ACCOUNTRESOURCE_INVALID_ACCOUNTRESOURCE_NOT_FOUND.cloudManagerFault(accountResourceId);
		}
		
		try (RemovedEventNotifier<CloudAccountResource> notifier = new RemovedEventNotifier<>(null, CloudAccountResource.class, new CloudAccountResource(dao))) {
			em.remove(dao);
			AccountResourceUtil.removeAccountResourceHierarchy(RepositoryControllerBeanWrapper.bean(), accountResourceId);
			notifier.setCompleted();
		}
	}

	@Override
	@Transactional(Transactional.TransactionType.Supported)
	public CloudAccountResource findCloudAccountResource(String accountResourceId) throws CloudManagerFault {
		try {
			EntityManagerEx em = SessionService.current().getEntityManagerEx();
			return new CloudAccountResource(em.find(CloudAccountResourceDao.class, accountResourceId));
		}
		catch (NoResultException e) {
			throw ErrorCode.ACCOUNTRESOURCE_INVALID_ACCOUNTRESOURCE_NOT_FOUND.cloudManagerFault(accountResourceId);
		}
	}

	@Override
	@Transactional(Transactional.TransactionType.Supported)
	public CloudAccountResource findCloudAccountResourceByCurrentUser(String accountResourceId) throws CloudManagerFault {
		return findCloudAccountResourceByUser(
				SessionService.current().getHinemosCredential().getUserId(),
				accountResourceId
				);
	}
	
	@Override
	@Transactional(Transactional.TransactionType.Supported)
	public List<CloudAccountResource> findAllCloudAccountResource() throws CloudManagerFault {
		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		List<CloudAccountResource> cars = new ArrayList<>(); 
		for (CloudAccountResourceDao dao: em.findAll(CloudAccountResourceDao.class)) {
			cars.add(new CloudAccountResource(dao));
		}
		return cars;
	}
	
	@Override
	@Transactional(Transactional.TransactionType.Supported)
	public List<CloudAccountResource> findCloudAccountResourcesByCurrentUser() throws CloudManagerFault {
		return findCloudAccountResourcesByUser(
				SessionService.current().getHinemosCredential().getUserId()
				);
	}

	@Override
	@Transactional(Transactional.TransactionType.Supported)
	public List<String> getCloudServicesForBilling(String accountResourceId) throws CloudManagerFault {
		IAccountResourceOperator operator = ObjectRegistryService.registry().get(IAccountResourceOperator.class);
		CloudAccountResource accountResource = operator.findCloudAccountResource(accountResourceId);
	
		ICloudServiceOperator serviceOperator = ObjectRegistryService.registry().get(ICloudServiceOperator.class);
		CloudService cloudService = serviceOperator.findCloudService(accountResource.getCloudServiceId());
	
		ICloudUserOperator userOperator = ObjectRegistryService.registry().get(ICloudUserOperator.class);
		CloudUser cloudUser = userOperator.findCloudUser(accountResource.getAccountId());
	
		ICloudOption cloudOption = ObjectRegistryService.registry().get(ICloudOption.class, cloudService.getCloudTypeId());
		cloudOption.setCredential(new CloudCredential(cloudUser));
		
		return cloudOption.getCloudServicesForBilling();
	}

	@Override
	public CloudAccountResource findCloudAccountResourceByUser(String userId, String accountResourceId) throws CloudManagerFault {
		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		Query query = em.createQuery("SELECT DISTINCT u.accountResource FROM CloudUserDao u, UserEntity h, RoleEntity r WHERE (u.accountResource.accountResourceId = :accountResourceId" +
				" AND h.userId = :userId AND h MEMBER OF r.userEntities AND r.roleId = u.roleId) OR (h.userId = :userId AND h MEMBER OF r.userEntities AND r.roleId = :ADMINISTRATORS)");
		
		query.setParameter("accountResourceId", accountResourceId);
		query.setParameter("userId", SessionService.current().getHinemosCredential().getUserId());
		query.setParameter("ADMINISTRATORS", RoleIdConstant.ADMINISTRATORS);
		
//		EntityManagerEx em = SessionContextService.context().getEntityManagerEx();
//		Query query = em.createQuery("SELECT DISTINCT u.accountResource FROM CloudUserDao u, UserEntity h, RoleEntity r WHERE u.accountResource.accountResourceId = :accountResourceId" +
//				" AND h.userId = :userId AND h MEMBER OF r.userEntities AND r.roleId = u.roleId");
//		query.setParameter("accountResourceId", accountResourceId);
//		query.setParameter("userId", userId);

		try {
			return new CloudAccountResource((CloudAccountResourceDao)query.getSingleResult());
		}
		catch (NoResultException e) {
			throw ErrorCode.ACCOUNTRESOURCE_INVALID_ACCOUNTRESOURCE_NOT_FOUND.cloudManagerFault(accountResourceId);
		}
	}

	@Override
	public List<CloudAccountResource> findCloudAccountResourcesByUser(String userId) throws CloudManagerFault {
		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		Query query = em.createQuery("SELECT DISTINCT u.accountResource FROM CloudUserDao u, UserEntity h, RoleEntity r WHERE (" +
				"h.userId = :userId AND h MEMBER OF r.userEntities AND r.roleId = u.roleId) OR (h.userId = :userId AND h MEMBER OF r.userEntities AND r.roleId = :ADMINISTRATORS)");
		
		query.setParameter("userId", SessionService.current().getHinemosCredential().getUserId());
		query.setParameter("ADMINISTRATORS", RoleIdConstant.ADMINISTRATORS);

//		EntityManagerEx em = SessionContextService.context().getEntityManagerEx();
//		Query query = em.createQuery("SELECT DISTINCT u.accountResource FROM CloudUserDao u, UserEntity h, RoleEntity r WHERE h.userId = :userId AND h MEMBER OF r.userEntities AND r.roleId = u.roleId");
//		query.setParameter("userId", userId);

		List<?> daos = query.getResultList();
		List<CloudAccountResource> cars = new ArrayList<>(); 
		for (Object dao: daos) {
			cars.add(new CloudAccountResource((CloudAccountResourceDao)dao));
		}
		return cars;
	}
}
