/*
Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.persistence;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;

import com.clustercontrol.cloud.InternalManagerError;
import com.clustercontrol.cloud.SessionService;
import com.clustercontrol.cloud.SessionService.ISession;


public class PersistenceUtil {
	public static class TransactionScope implements AutoCloseable {
		private boolean completed = false;
		private boolean transactional = false;
		private boolean requiredNew = false;
		
		public TransactionScope() {
			this(Transactional.TransactionType.Required);
		}

		public TransactionScope(Transactional.TransactionType transactionType) {
			ISession session = SessionService.current();
			switch (transactionType) {
			case Required:
				transactional = !session.isTransaction();
				break;
			case RequiredNew:
				transactional = true;
				requiredNew = true;
				break;
			default:
				break;
			}

			if (requiredNew) {
				SessionService.offer();
				session = SessionService.current();
			}
			
			if (transactional) session.beginTransaction();
		}
		
		public void complete() {
			completed = true;
		}
		
		@Override
		public void close() {
			if (!completed) {
				SessionService.current().rollbackTransaction();
			}
			else {
				if (transactional) SessionService.current().commitTransaction();
			}
			
			if (requiredNew) SessionService.poll();
		}
	}
	
	private static class TransactionalInvoker2 implements InvocationHandler {
		private Object implementor;
		
		public TransactionalInvoker2(Object implementor) {
			this.implementor = implementor;
		}
		
		@Override
		public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
			boolean requiredNew = false;
			try {
				ISession session = SessionService.current();
				boolean transactional = false;
				
				Transactional ct = implementor.getClass().getAnnotation(Transactional.class);
				if (ct != null) {
					switch (ct.value()) {
					case Required:
						transactional = !session.isTransaction();
						break;
					case RequiredNew:
						transactional = true;
						requiredNew = true;
						break;
					default:
						break;
					}
				}

				Method m = implementor.getClass().getMethod(method.getName(), method.getParameterTypes());
				Transactional mt = m.getAnnotation(Transactional.class);
				if (mt != null) {
					transactional =  mt.value() == Transactional.TransactionType.Required;
					switch (mt.value()) {
					case Required:
						transactional = !session.isTransaction();
						requiredNew = false;
						break;
					case RequiredNew:
						transactional = true;
						requiredNew = true;
						break;
					default:
						break;
					}
				}

				if (requiredNew) {
					SessionService.offer();
					session = SessionService.current();
				}
				
				if (transactional) {
					session.beginTransaction();

					Object result = null;
					try {
						result = method.invoke(implementor, args);
					}
					catch (Exception e) {
						session.rollbackTransaction();
						throw e;
					}

					session.commitTransaction();

					return result;
				}
				else {
					return method.invoke(implementor, args);
				}
			}
			catch (InvocationTargetException e) {
				throw e.getTargetException();
			}
			catch (IllegalAccessException | IllegalArgumentException e) {
				throw e;
			}
			catch (RuntimeException e) {
				throw e;
			}
			catch (Exception e) {
				throw new InternalManagerError(e);
			}
			finally {
				if (requiredNew) SessionService.poll();
			}
		}
	}
	
	public static <T> T decorateTransactional(Class<T> interfaceClass, T implementer) {
		return 	interfaceClass.cast(Proxy.newProxyInstance(interfaceClass.getClassLoader(), new Class<?>[]{interfaceClass}, new TransactionalInvoker2(implementer)));
	}
}
