/*
Copyright (C) 2014 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
*/
package com.clustercontrol.ws.cloudn;

import java.util.ArrayList;
import java.util.List;

import javax.jws.WebService;

import org.eclipse.jdt.annotation.NonNull;

import com.clustercontrol.accesscontrol.bean.PrivilegeConstant.SystemPrivilegeMode;
import com.clustercontrol.cloud.CloudManagerFault;
import com.clustercontrol.cloud.ICloudContext;
import com.clustercontrol.cloud.SessionService;
import com.clustercontrol.cloud.bean.Tag;
import com.clustercontrol.cloud.cloudn.factory.CloudnAsynchJobTrackingOperator;
import com.clustercontrol.cloud.cloudn.rest.CloudnCompute;
import com.clustercontrol.cloud.cloudn.rest.ListNetworksResponse;
import com.clustercontrol.cloud.cloudn.rest.ListSecurityGroupsResponse;
import com.clustercontrol.cloud.cloudn.rest.ResetPasswordForVirtualMachineResponse;
import com.clustercontrol.cloud.cloudn.rest.api.RestfulApplicationException;
import com.clustercontrol.cloud.cloudn.util.CloudnConstants;
import com.clustercontrol.cloud.cloudn.util.CloudnUtil;
import com.clustercontrol.cloud.factory.CloudRoleConstants;
import com.clustercontrol.cloud.util.Tuple;
import com.clustercontrol.cloud.validation.AuthorizingValidator_role_region;
import com.clustercontrol.cloud.validation.annotation.CustomMethodValidation;
import com.clustercontrol.cloud.validation.annotation.ElementId;
import com.clustercontrol.cloud.validation.annotation.Identity;
import com.clustercontrol.cloud.validation.annotation.NotNull;
import com.clustercontrol.cloud.validation.annotation.Size;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.InvalidUserPass;
import com.clustercontrol.ws.cloud.IWebServiceBase;
import com.clustercontrol.ws.cloud.security.HinemosAccessRight;

/**
 * クラウド管理オプション用のエンドポイント
 *
 */
@WebService(serviceName = "CloudnOptionEndpointService", portName = "CloudnOptionEndpointPort", targetNamespace = "http://cloudn.ws.clustercontrol.com", endpointInterface="com.clustercontrol.ws.cloudn.CloudnOptionEndpoint")
public class CloudnOptionEndpointImpl implements CloudnOptionEndpoint, IWebServiceBase, CloudnConstants, CloudRoleConstants {
	protected @NonNull CloudnCompute getCloudnEndpoint() throws CloudManagerFault {
		ICloudContext context = SessionService.current().get(ICloudContext.class);
		return CloudnUtil.getCloudnEndpoint(
				CloudnCompute.class,
				context.getAccessDestionation().getCloudUser().getAccessKey(),
				context.getAccessDestionation().getCloudUser().getSecretKey(),
				context.getCurrentRegion().getEndpoint(EP_TYPE_COMPUTE_FLAT).getLocation());
	}

	@Override
	@HinemosAccessRight(roleName=CLOUDMANAGEMENT, right=SystemPrivilegeMode.READ)
	@CustomMethodValidation(AuthorizingValidator_role_region.class)
	public List<Network> getNetworks(
			@ElementId("roleId") @Identity String roleId,
			@ElementId("region") @NotNull @Size(max=64) String region
			) throws CloudManagerFault, InvalidUserPass, InvalidRole {
		try {
			List<Network> networks = new ArrayList<>(); 
			ListNetworksResponse res = getCloudnEndpoint().listNetworks();
			for (com.clustercontrol.cloud.cloudn.rest.Network net: res.network) {
				Network network = new Network();
				network.setName(net.name);
				network.setNetworkId(net.id);
				network.setVpcId(net.vpcid);
				network.setAvailabilityZone(net.zonename);
				network.setCidrBlock(net.cidr);
				networks.add(network);
			}
			return networks;
		}
		catch (RestfulApplicationException e) {
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}
	}

	@Override
	public void start() {
	}

	@Override
	public void stop() {
	}

	@Override
	@HinemosAccessRight(roleName=CLOUDMANAGEMENT, right=SystemPrivilegeMode.READ)
	@CustomMethodValidation(AuthorizingValidator_role_region.class)
	public List<SecurityGroup> getSecurityGroups(
			@ElementId("roleId") @Identity String roleId,
			@ElementId("region") @NotNull @Size(max=64) String region) throws CloudManagerFault, InvalidUserPass, InvalidRole 
	{
		try {
			List<SecurityGroup> securityGroups = new ArrayList<>(); 
			ListSecurityGroupsResponse res = getCloudnEndpoint().listSecurityGroups(null);
			for (com.clustercontrol.cloud.cloudn.rest.SecurityGroup restSecurityGroup: res.securitygroup) {
				SecurityGroup securityGroup = new SecurityGroup();
				securityGroup.setName(restSecurityGroup.name);
				securityGroup.setId(restSecurityGroup.id);
				securityGroup.setDescription(restSecurityGroup.description);
				if(restSecurityGroup.tags != null && !restSecurityGroup.tags.isEmpty()){
					securityGroup.setTags(new ArrayList<Tag>());
					for(com.clustercontrol.cloud.cloudn.rest.Tag tag: restSecurityGroup.tags){
						securityGroup.getTags().add(new Tag(tag.key, tag.value));
					}
				}
				securityGroups.add(securityGroup);
			}
			return securityGroups;
		}
		catch (RestfulApplicationException e) {
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}
	}
	
	@Override
	@HinemosAccessRight(roleName=CLOUDMANAGEMENT, right={SystemPrivilegeMode.READ, SystemPrivilegeMode.EXEC})
	@CustomMethodValidation(AuthorizingValidator_role_region.class)
	public void resetPassword(
			@ElementId("roleId") @Identity String roleId,
			@ElementId("region") @NotNull @Size(max=64) String region,
			@ElementId("instanceId") @NotNull String instanceId) throws CloudManagerFault, InvalidUserPass, InvalidRole
	{
		try {
			ResetPasswordForVirtualMachineResponse response = getCloudnEndpoint().resetPasswordForVirtualMachine(instanceId);
			CloudnAsynchJobTrackingOperator.registJobId(Tuple.build("getPassword", instanceId, getAccountResourceId()), response.jobid);
		}
		catch (RestfulApplicationException e) {
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}
	}

	@Override
	@HinemosAccessRight(roleName=CLOUDMANAGEMENT, right=SystemPrivilegeMode.READ)
	@CustomMethodValidation(AuthorizingValidator_role_region.class)
	public PasswordResult getPassword(
			@ElementId("roleId") @Identity String roleId,
			@ElementId("region") @NotNull @Size(max=64) String region,
			@ElementId("instanceId") @NotNull String instanceId) throws CloudManagerFault, InvalidUserPass, InvalidRole
	{
		try {
			String jobId = CloudnAsynchJobTrackingOperator.findJobId(Tuple.build("getPassword", instanceId, getAccountResourceId()));
			return CloudnAsynchJobTrackingOperator.getJobResult(getCloudnEndpoint(), jobId, PasswordResult.class);
		}
		catch (RestfulApplicationException e) {
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}
	}

	@Override
	@HinemosAccessRight(roleName=CLOUDMANAGEMENT, right=SystemPrivilegeMode.READ)
	@CustomMethodValidation(AuthorizingValidator_role_region.class)
	public AttachStorageResult getAttachStorageResult(
			@ElementId("roleId") @Identity String roleId,
			@ElementId("region") @NotNull @Size(max=64) String region,
			@ElementId("storageId") @NotNull String storageId,
			@ElementId("instanceId") @NotNull String instanceId) throws CloudManagerFault, InvalidUserPass, InvalidRole
	{
		try {
			String jobId = CloudnAsynchJobTrackingOperator.findJobId(Tuple.build("getAttachStorageResult", storageId, instanceId, getAccountResourceId()));
			return CloudnAsynchJobTrackingOperator.getJobResult(getCloudnEndpoint(), jobId, AttachStorageResult.class);
		}
		catch (RestfulApplicationException e) {
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}
	}
	
	@Override
	@HinemosAccessRight(roleName=CLOUDMANAGEMENT, right=SystemPrivilegeMode.READ)
	@CustomMethodValidation(AuthorizingValidator_role_region.class)
	public DeleteInstanceResult getDeleteInstanceResult(
			@ElementId("roleId") @Identity String roleId,
			@ElementId("region") @NotNull @Size(max=64) String region,
			@ElementId("instanceId") @NotNull String instanceId) throws CloudManagerFault, InvalidUserPass, InvalidRole
	{
		try {
			String jobId = CloudnAsynchJobTrackingOperator.findJobId(Tuple.build("getDeleteInstanceResult", instanceId, getAccountResourceId()));
			return CloudnAsynchJobTrackingOperator.getJobResult(getCloudnEndpoint(), jobId, DeleteInstanceResult.class);
		}
		catch (RestfulApplicationException e) {
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}
	}
	
	
	private String getAccountResourceId(){
		return SessionService.current().get(ICloudContext.class).getAccessDestionation().getCloudAccountResource().getAccountResourceId();
	}
}