/*
Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;

import com.clustercontrol.cloud.InternalManagerError;
import com.clustercontrol.cloud.SessionService;
import com.clustercontrol.cloud.persistence.TransactionException;
import com.clustercontrol.fault.FacilityNotFound;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.InvalidSetting;
import com.clustercontrol.fault.NotifyDuplicate;
import com.clustercontrol.fault.NotifyNotFound;
import com.clustercontrol.fault.UsedFacility;
import com.clustercontrol.notify.bean.NotifyInfo;
import com.clustercontrol.notify.bean.NotifyRelationInfo;
import com.clustercontrol.notify.bean.OutputBasicInfo;
import com.clustercontrol.notify.session.NotifyControllerBean;

public class NotifyControllerBeanWrapper extends NotifyControllerBean {

	private static ThreadLocal<NotifyControllerBeanWrapper> instance  = new ThreadLocal<NotifyControllerBeanWrapper>() {
		protected NotifyControllerBeanWrapper initialValue()
		{
			return null;
		}
	};

	public NotifyControllerBeanWrapper() {
		super();
	}

	@Override
	public NotifyInfo getNotify(final String notifyId) throws NotifyNotFound, InvalidRole, HinemosUnknown {
		try {
			return execute(new Callable<NotifyInfo>() {
				@Override
				public NotifyInfo call() throws Exception {
					return new NotifyControllerBean().getNotify(notifyId);
				}
			});
		}
		catch (InterruptedException e) { 
			throw new InternalManagerError(e);
		}
		catch (ExecutionException e) {
			// めんどくさいが、以下のように ExecutionException を含めて例外をラップしないと、
			// ExecutionException 分の例外が途切れる
			if (e.getCause() instanceof InvalidRole) {
				throw new InvalidRole(e.getCause().getMessage(), e);
			}
			else if (e.getCause() instanceof NotifyNotFound) {
				throw new NotifyNotFound(e.getCause().getMessage(), e);
			}
			else if (e.getCause() instanceof HinemosUnknown) {
				throw new HinemosUnknown(e.getCause().getMessage(), e);
			}
			else {
				throw new InternalManagerError(e);
			}
		} catch (Exception e) {
			throw new InternalManagerError(e);
		}
	}

	@Override
	public boolean addNotify(final NotifyInfo info) throws NotifyDuplicate, InvalidSetting, InvalidRole, HinemosUnknown {
		try {
			boolean result = execute(new Callable<Boolean>() {
				@Override
				public Boolean call() throws Exception {
					return new NotifyControllerBean().addNotify(info);
				}
			});
			
			SessionService.current().addRollbackAction(
					new SessionService.RolebackAction() {
						@Override
						public void rollback() throws TransactionException {
							try {
								execute(new Callable<Object>() {
									@Override
									public Object call() throws Exception {
										new NotifyControllerBean().deleteNotify(info.getNotifyId());
										return null;
									}
								});
							}
							catch (Exception e) {
								throw new TransactionException(e);
							}
						}
					});
			return result;
		}
		catch (InterruptedException e) { 
			throw new InternalManagerError(e);
		}
		catch (ExecutionException e) {
			// めんどくさいが、以下のように ExecutionException を含めて例外をラップしないと、
			// ExecutionException 分の例外が途切れる
			if (e.getCause() instanceof InvalidRole) {
				throw new InvalidRole(e.getCause().getMessage(), e);
			}
			else if (e.getCause() instanceof NotifyDuplicate) {
				throw new NotifyDuplicate(e.getCause().getMessage(), e);
			}
			else if (e.getCause() instanceof InvalidSetting) {
				throw new InvalidSetting(e.getCause().getMessage(), e);
			}
			else if (e.getCause() instanceof HinemosUnknown) {
				throw new HinemosUnknown(e.getCause().getMessage(), e);
			}
			else {
				throw new InternalManagerError(e);
			}
		} catch (Exception e) {
			throw new InternalManagerError(e);
		}
	}
	
	@Override
	public boolean deleteNotify(final String notifyId) throws NotifyNotFound, InvalidRole, HinemosUnknown {
		try {
			ITaskResult<Boolean> result = execute(new Callable<ITaskResult<Boolean>>() {
				@Override
				public ITaskResult<Boolean> call() throws Exception {
					NotifyControllerBean bean = new NotifyControllerBean();
					final NotifyInfo info = bean.getNotify(notifyId);
					final boolean result = bean.deleteNotify(notifyId);
					
					ITaskResult<Boolean> rolebackAction = new ITaskResult<Boolean>() {
						@Override
						public void rollback() throws TransactionException {
							try {
								execute(new Callable<Object>() {
									@Override
									public Object call() throws Exception {
										new NotifyControllerBean().addNotify(info);
										return null;
									}
								});
							}
							catch (Exception e) {
								throw new TransactionException(e);
							}
						}

						@Override
						public Boolean getResult() throws Exception {
							return result;
						}
					};
					
					return rolebackAction;
				}
			});
			
			SessionService.current().addRollbackAction(result);
			return result.getResult();
		}
		catch (InterruptedException e) {
			throw new InternalManagerError(e);
		}
		catch (ExecutionException e) {
			// めんどくさいが、以下のように ExecutionException を含めて例外をラップしないと、
			// ExecutionException 分の例外が途切れる
			if (e.getCause() instanceof InvalidRole) {
				throw new InvalidRole(e.getCause().getMessage(), e);
			}
			else if (e.getCause() instanceof NotifyNotFound) {
				throw new NotifyNotFound(e.getCause().getMessage(), e);
			}
			else if (e.getCause() instanceof HinemosUnknown) {
				throw new HinemosUnknown(e.getCause().getMessage(), e);
			}
			else {
				throw new InternalManagerError(e);
			}
		} catch (Exception e) {
			throw new InternalManagerError(e);
		}
	}
	
	private  <T> T execute(Callable<T> task) throws Exception {
		return HinemosUtil.submit(task).get();
	}
	
	public static NotifyControllerBeanWrapper bean() {
		NotifyControllerBeanWrapper bean = instance.get();
		if (bean == null) {
			bean = new NotifyControllerBeanWrapper();
			instance.set(bean);
		}
		return bean;
	}

	@Override
	public boolean modifyNotify(NotifyInfo info) throws NotifyDuplicate,
			InvalidRole, HinemosUnknown, InvalidSetting {
		throw new UnsupportedOperationException();
	}

	@Override
	public ArrayList<NotifyInfo> getNotifyList() throws NotifyNotFound,
			InvalidRole, HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public ArrayList<NotifyInfo> getNotifyListByOwnerRole(String ownerRoleId)
			throws InvalidRole, HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public HashMap<String, NotifyInfo> getNotifyMap() throws InvalidRole,
			HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public ArrayList<NotifyRelationInfo> getNotifyRelation(final String notifyGroupId) throws InvalidRole, HinemosUnknown {
		try {
			return execute(new Callable<ArrayList<NotifyRelationInfo>>() {
				@Override
				public ArrayList<NotifyRelationInfo> call() throws Exception {
					return new NotifyControllerBean().getNotifyRelation(notifyGroupId);
				}
			});
		}
		catch (InterruptedException e) { 
			throw new InternalManagerError(e);
		}
		catch (ExecutionException e) {
			// めんどくさいが、以下のように ExecutionException を含めて例外をラップしないと、
			// ExecutionException 分の例外が途切れる
			if (e.getCause() instanceof InvalidRole) {
				throw new InvalidRole(e.getCause().getMessage(), e);
			}
			else if (e.getCause() instanceof HinemosUnknown) {
				throw new HinemosUnknown(e.getCause().getMessage(), e);
			}
			else {
				throw new InternalManagerError(e);
			}
		} catch (Exception e) {
			throw new InternalManagerError(e);
		}
	}

	@Override
	public boolean modifyNotifyRelation(final Collection<NotifyRelationInfo> info, final String notifyGroupId) throws NotifyNotFound, InvalidRole, HinemosUnknown {
		try {
			ITaskResult<Boolean> result = execute(new Callable<ITaskResult<Boolean>>() {
				@Override
				public ITaskResult<Boolean> call() throws Exception {
					// NotifyControllerBean の関数設計がおかしいので、複雑がが以下の処理でロールバックを実現する。
					NotifyControllerBean bean = new NotifyControllerBean();
					final List<NotifyRelationInfo> relations = bean.getNotifyRelation(notifyGroupId);
					final boolean result = bean.modifyNotifyRelation(info, notifyGroupId);
					
					ITaskResult<Boolean> rolebackAction = new ITaskResult<Boolean>() {
						@Override
						public void rollback() throws TransactionException {
							try {
								execute(new Callable<Object>() {
									@Override
									public Object call() throws Exception {
										NotifyControllerBean bean = new NotifyControllerBean();
										bean.modifyNotifyRelation(relations, notifyGroupId);
										return null;
									}
								});
							}
							catch (Exception e) {
								throw new TransactionException(e);
							}
						}
						@Override
						public Boolean getResult() throws Exception {
							return result;
						}
					};
					return rolebackAction;
				}
			});
			
			SessionService.current().addRollbackAction(result);
			return result.getResult();
		}
		catch (InterruptedException e) {
			throw new InternalManagerError(e);
		}
		catch (ExecutionException e) {
			// めんどくさいが、以下のように ExecutionException を含めて例外をラップしないと、
			// ExecutionException 分の例外が途切れる
			if (e.getCause() instanceof InvalidRole) {
				throw new InvalidRole(e.getCause().getMessage(), e);
			}
			else if (e.getCause() instanceof HinemosUnknown) {
				throw new HinemosUnknown(e.getCause().getMessage(), e);
			}
			else {
				throw new InternalManagerError(e);
			}
		} catch (Exception e) {
			throw new InternalManagerError(e);
		}
	}

	@Override
	public boolean deleteNotifyRelation(String notifyGroupId) throws InvalidRole, HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public boolean addNotifyRelation(final Collection<NotifyRelationInfo> info) throws InvalidRole, HinemosUnknown {
		try {
			ITaskResult<Boolean> result = execute(new Callable<ITaskResult<Boolean>>() {
				@Override
				public ITaskResult<Boolean> call() throws Exception {
					// NotifyControllerBean の関数設計がおかしいので、複雑がが以下の処理でロールバックを実現する。
					NotifyControllerBean bean = new NotifyControllerBean();
					Map<String, List<NotifyRelationInfo>> map = new HashMap<>();
					final List<NotifyRelationInfo> relations = new ArrayList<>(info);
					Iterator<NotifyRelationInfo> iter = relations.iterator();
					while (iter.hasNext()) {
						NotifyRelationInfo relation = iter.next();
						List<NotifyRelationInfo> rs = map.get(relation.getNotifyGroupId());
						if (rs == null) {
							rs = bean.getNotifyRelation(relation.getNotifyGroupId());
							if (rs == null) {
								rs = Collections.emptyList();
							}
							map.put(relation.getNotifyGroupId(), rs);
						}
						for (NotifyRelationInfo r: rs) {
							if (relation.getNotifyId().equals(r.getNotifyId())) {
								iter.remove();
								break;
							}
						}
					}
					final boolean result = bean.addNotifyRelation(relations);
					
					ITaskResult<Boolean> rolebackAction = new ITaskResult<Boolean>() {
						@Override
						public void rollback() throws TransactionException {
							try {
								execute(new Callable<Object>() {
									@Override
									public Object call() throws Exception {
										NotifyControllerBean bean = new NotifyControllerBean();
										for (NotifyRelationInfo relation: relations) {
											List<NotifyRelationInfo> rs = new ArrayList<>(bean.getNotifyRelation(relation.getNotifyGroupId()));
											Iterator<NotifyRelationInfo> iter = rs.iterator();
											while (iter.hasNext()) {
												NotifyRelationInfo r = iter.next();
												if (r.getNotifyId().equals(relation.getNotifyId())) {
													iter.remove();
													// 効率わるいけど。。。。
													bean.deleteNotifyRelation(relation.getNotifyGroupId());
													bean.addNotifyRelation(rs);
													break;
												}
											}
										}
										return null;
									}
								});
							}
							catch (Exception e) {
								throw new TransactionException(e);
							}
						}
						@Override
						public Boolean getResult() throws Exception {
							return result;
						}
					};
					return rolebackAction;
				}
			});
			
			SessionService.current().addRollbackAction(result);
			return result.getResult();
		}
		catch (InterruptedException e) {
			throw new InternalManagerError(e);
		}
		catch (ExecutionException e) {
			// めんどくさいが、以下のように ExecutionException を含めて例外をラップしないと、
			// ExecutionException 分の例外が途切れる
			if (e.getCause() instanceof InvalidRole) {
				throw new InvalidRole(e.getCause().getMessage(), e);
			}
			else if (e.getCause() instanceof HinemosUnknown) {
				throw new HinemosUnknown(e.getCause().getMessage(), e);
			}
			else {
				throw new InternalManagerError(e);
			}
		} catch (Exception e) {
			throw new InternalManagerError(e);
		}
	}

	@Override
	public ArrayList<String> checkNotifyId(String notifyId) throws InvalidRole,
			HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public void setNotifyStatus(String notifyId, boolean validFlag)
			throws HinemosUnknown, NotifyNotFound, NotifyDuplicate, InvalidRole {
		throw new UnsupportedOperationException();
	}

	@Override
	public void isUseFacilityId(String facilityId) throws UsedFacility {
		throw new UnsupportedOperationException();
	}

	@Override
	public void insertEventLog(OutputBasicInfo output, int confirmState)
			throws InvalidRole, HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public void sendAfterConvertHostname(String ipAddress, int port,
			String facility, String severity, String facilityId,
			String message, String timeStamp) throws InvalidRole,
			HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public void sendMail(String[] address, OutputBasicInfo outputBasicInfo)
			throws InvalidRole, HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public void notify(String pluginId, String monitorId, String facilityId,
			String subKey, long generationDate, int priority,
			String application, String messageId, String message,
			String messageOrg, ArrayList<String> notifyIdList, String srcId)
			throws FacilityNotFound, HinemosUnknown, NotifyNotFound,
			InvalidRole {
		throw new UnsupportedOperationException();
	}

	@Override
	public void notify(final OutputBasicInfo notifyInfo, final String notifyGroupId) {
		try {
			execute(new Callable<Object>() {
				@Override
				public Object call() throws Exception {
					new NotifyControllerBean().notify(notifyInfo, notifyGroupId);
					return null;
				}
			});
		}
		catch (Exception e) {
			throw new InternalManagerError(e);
		}
	}

	@Override
	protected void notify(OutputBasicInfo notifyInfo, List<String> notifyIdList) {
		throw new UnsupportedOperationException();
	}
}