/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.collectiverun.session;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;

import javax.persistence.EntityExistsException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.CollectiveRunNotFound;
import com.clustercontrol.fault.FacilityNotFound;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.InvalidSetting;
import com.clustercontrol.fault.UsedFacility;
import com.clustercontrol.bean.PluginConstant;
import com.clustercontrol.bean.StatusConstant;
import com.clustercontrol.commons.bean.ViewListInfo;
import com.clustercontrol.collectiverun.bean.CommandInfo;
import com.clustercontrol.collectiverun.bean.CommandParameterInfo;
import com.clustercontrol.collectiverun.bean.CommandTypeInfo;
import com.clustercontrol.collectiverun.bean.DetailInfo;
import com.clustercontrol.collectiverun.bean.HistoryFilterInfo;
import com.clustercontrol.collectiverun.bean.ItemConstant;
import com.clustercontrol.collectiverun.bean.ItemInfo;
import com.clustercontrol.collectiverun.bean.ParameterInfo;
import com.clustercontrol.collectiverun.bean.ParameterSelectInfo;
import com.clustercontrol.collectiverun.bean.CollectiveRunInfo;
import com.clustercontrol.collectiverun.bean.SessionInfo;
import com.clustercontrol.collectiverun.bean.SessionParameterInfo;
import com.clustercontrol.collectiverun.bean.TreeItem;
import com.clustercontrol.collectiverun.bean.TypeInfo;
import com.clustercontrol.collectiverun.factory.DeleteMaster;
import com.clustercontrol.collectiverun.factory.ModifyMaster;
import com.clustercontrol.collectiverun.factory.OperationRun;
import com.clustercontrol.collectiverun.factory.SelectMaster;
import com.clustercontrol.collectiverun.factory.SelectParameter;
import com.clustercontrol.collectiverun.factory.SelectSession;
import com.clustercontrol.collectiverun.model.CrunCmdMstEntity;
import com.clustercontrol.collectiverun.model.CrunCmdParamMstEntity;
import com.clustercontrol.collectiverun.model.CrunParamMstEntity;
import com.clustercontrol.collectiverun.model.CrunParamSelectMstEntity;
import com.clustercontrol.collectiverun.model.CrunSessionDetailEntity;
import com.clustercontrol.collectiverun.model.CrunSessionEntity;
import com.clustercontrol.collectiverun.model.CrunSessionParamEntity;
import com.clustercontrol.collectiverun.model.CrunTypeMstEntity;
import com.clustercontrol.collectiverun.util.CollectiveRunValidator;
import com.clustercontrol.collectiverun.util.QueryUtil;
import com.clustercontrol.commons.session.CheckFacility;
import com.clustercontrol.commons.util.JpaTransactionManager;
import com.clustercontrol.jobmanagement.bean.RunResultInfo;
import com.clustercontrol.util.apllog.AplLogger;


/**
 *
 * <!-- begin-user-doc -->
 * 一括制御の管理を行う Session Bean です。<BR>
 * クライアントからの Entity Bean へのアクセスは、Session Bean を介して行います。
 * <!-- end-user-doc -->
 */
public class CollectiveRunControllerBean implements CheckFacility {
	private static Log m_log = LogFactory.getLog( CollectiveRunControllerBean.class );

	/**
	 * 
	 * 一括制御[履歴]ビューで選択されている履歴に対応する種別IDを取得します。
	 * 
	 * @param sessionId 選択されている履歴のセッションID
	 * @return 種別ID
	 * @throws FacilityNotFound
	 * @throws HinemosUnknown
	 */
	public String getParameterTypeId(String sessionId) throws FacilityNotFound, HinemosUnknown {
		JpaTransactionManager jtm = null;
		SelectParameter select = new SelectParameter();
		String parameterTypeId = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			parameterTypeId = select.getTypeId(sessionId);
			jtm.commit();
		} catch (FacilityNotFound e) {
			jtm.rollback();
			throw e;
		} catch (Exception e) {
			m_log.warn("getParameterTypeId() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return parameterTypeId;
	}

	/**
	 * 一括制御[履歴]ビューに表示する履歴のリストを取得します。
	 * 
	 * 
	 * @param histories 表示履歴数
	 * 
	 * @return 一括制御[履歴]情報
	 * 
	 * @see com.clustercontrol.bean.ViewListInfo
	 * @see com.clustercontrol.collectiverun.factory.SelectSession#getHistoryList(Locale, int)
	 * 
	 * @throws HinemosUnknown
	 */
	public ViewListInfo getHistoryList(int histories) throws HinemosUnknown {
		JpaTransactionManager jtm = null;
		ViewListInfo historyList = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			SelectSession select = new SelectSession();
			historyList = select.getHistoryList(histories);
			jtm.commit();
		} catch (Exception e) {
			m_log.warn("getHistoryList() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return historyList;
	}

	/**
	 * 一括制御[履歴]ビューに表示する履歴のリストを取得します。
	 * 
	 * 
	 * @param filter フィルタ
	 * @param histories 表示履歴数
	 * 
	 * @return 一括制御[履歴]情報
	 * 
	 * @see com.clustercontrol.bean.ViewListInfo
	 * @see com.clustercontrol.collectiverun.factory.SelectSession#getHistoryList(Property, Locale, int)
	 * 
	 * @throws HinemosUnknown
	 */
	public ViewListInfo getHistoryList(HistoryFilterInfo filter, int histories) throws HinemosUnknown {
		JpaTransactionManager jtm = null;
		ViewListInfo historyList = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			SelectSession select = new SelectSession();
			historyList = select.getHistoryList(filter, histories);
			jtm.commit();
		} catch (Exception e) {
			m_log.warn("getHistoryList() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return historyList;
	}

	/**
	 * 一括制御[詳細]ビューに表示する情報を取得します。<BR>
	 * 
	 * 
	 * @param sessionId セッションID
	 * @return 一括制御[詳細]情報
	 * 
	 * @see com.clustercontrol.collectiverun.factory.SelectSession#getDetailList(String)
	 * 
	 * @throws HinemosUnknown
	 */
	public ArrayList<DetailInfo> getDetailList(String sessionId) throws HinemosUnknown {
		JpaTransactionManager jtm = null;
		ArrayList<DetailInfo> detailList = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			SelectSession select = new SelectSession();
			detailList = select.getDetailList(sessionId);
			jtm.commit();
		} catch (Exception e) {
			m_log.warn("getDetailList() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return detailList;
	}

	/**
	 * 一括制御を実行する際に呼び出される、インタフェースメソッドです。
	 * セッションIDを生成し、一括制御を実行します。
	 * 
	 * @param typeId 種別ID
	 * @param info 実行情報
	 * @throws CollectiveRunNotFound
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws FacilityNotFound
	 * @throws InvalidSetting 
	 */
	public String run(String typeId, CollectiveRunInfo info) throws CollectiveRunNotFound, HinemosUnknown, FacilityNotFound, InvalidRole, InvalidSetting {
		JpaTransactionManager jtm = null;

		CollectiveRunValidator.validateCollectiveRunInfo(info);
		
		//実行履歴データ作成
		String sessionId = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			try {
				sessionId = createSession(typeId, info);
			} catch (FacilityNotFound e) {
				AplLogger apllog = new AplLogger("CR","cr");
				apllog.put("SYS","002");
				throw e;
			} catch (InvalidRole e) {
				AplLogger apllog = new AplLogger("CR","cr");
				apllog.put("SYS","002");
				throw e;
			} catch (HinemosUnknown e) {
				AplLogger apllog = new AplLogger("CR","cr");
				apllog.put("SYS","002");
				throw e;
			} catch (Exception e) {
				AplLogger apllog = new AplLogger("CR","cr");
				apllog.put("SYS","002");
				m_log.warn("run() : "
						+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
				throw new HinemosUnknown(e.getMessage(), e);
			}

			try {
				run(sessionId);
			} catch (HinemosUnknown e) {
				AplLogger apllog = new AplLogger("CR","cr");
				apllog.put("SYS","001");
				throw e;
			} catch (CollectiveRunNotFound e) {
				AplLogger apllog = new AplLogger("CR","cr");
				apllog.put("SYS","001");
				throw e;
			} catch (Exception e) {
				AplLogger apllog = new AplLogger("CR","cr");
				apllog.put("SYS","001");
				m_log.warn("run() : "
						+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
				throw new HinemosUnknown(e.getMessage(), e);
			}
			jtm.commit();
		} catch (CollectiveRunNotFound e) {
			jtm.rollback();
			throw e;
		} catch (HinemosUnknown e) {
			jtm.rollback();
			throw e;
		} catch (FacilityNotFound e) {
			jtm.rollback();
			throw e;
		} catch (Exception e) {
			m_log.warn("run() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return sessionId;
	}

	/**
	 * 一括制御の実行履歴情報を作成します。<BR>
	 * 返り値としてセッションIDを返します。
	 * 
	 * @param typeId 種別ID
	 * @param info 実行情報
	 * 
	 * @return セッションID
	 * 
	 * @throws FacilityNotFound
	 * @throws InvalidRole
	 * @throws HinemosUnknown
	 */
	public String createSession(String typeId, CollectiveRunInfo info)
			throws FacilityNotFound, InvalidRole, HinemosUnknown {
		JpaTransactionManager jtm = null;
		OperationRun operation = new OperationRun();
		String ret = null;

		//実行履歴データ作成
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			ret = operation.createSession(typeId, info);
			jtm.commit();
		} catch (EntityExistsException e) {
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (FacilityNotFound e) {
			jtm.rollback();
			throw e;
		} catch (InvalidRole e) {
			jtm.rollback();
			throw e;
		} catch (Exception e) {
			m_log.warn("createSession() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}

		return ret;
	}

	/**
	 * 一括制御を実行する際に呼び出されるインタフェースメソッドです。
	 * 
	 * @param sessionId セッションID
	 * @throws HinemosUnknown
	 * @throws CollectiveRunNotFound
	 */
	public void run(String sessionId) throws HinemosUnknown, CollectiveRunNotFound {
		JpaTransactionManager jtm = null;
		OperationRun operation = new OperationRun();

		try{
			jtm = new JpaTransactionManager();
			jtm.begin();
			operation.run(sessionId);
			jtm.commit();
		} catch (CollectiveRunNotFound e) {
			jtm.rollback();
			throw e;
		} catch (IOException e) {
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch(Exception e){
			m_log.warn("run() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
	}

	/**
	 * 一括制御の終了処理を行います。<BR>
	 * 終了状態を、実行履歴・実行履歴詳細情報に設定します。
	 * 
	 * @param info 実行結果情報
	 * 
	 * @see com.clustercontrol.collectiverun.factory.OperationRun#endNode(RunResultInfo)
	 * @see com.clustercontrol.jobmanagement.message.RunResultInfo
	 * 
	 * @throws HinemosUnknown
	 * @throws CollectiveRunNotFound
	 */
	public void endNode(RunResultInfo info) throws HinemosUnknown, CollectiveRunNotFound {
		JpaTransactionManager jtm = null;
		OperationRun operation = new OperationRun();

		try{
			jtm = new JpaTransactionManager();
			jtm.begin();
			operation.endNode(info);
			jtm.commit();
		}catch(CollectiveRunNotFound e){
			jtm.rollback();
			throw e;
		}catch(Exception e){
			m_log.warn("endNode() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
	}

	/**
	 * Quartzからのコールバックメソッド。<BR>
	 * セッションの終了状態をチェックします。
	 * 
	 * @param info
	 * @throws HinemosUnknown
	 */
	public void checkEndSession() throws HinemosUnknown {
		JpaTransactionManager jtm = null;
		OperationRun operation = new OperationRun();

		try{
			jtm = new JpaTransactionManager();
			jtm.begin();
			operation.checkEndSession();
			jtm.commit();
		} catch (Exception e) {
			m_log.warn("checkEndSession() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
	}

	/**
	 * 一括制御コマンドの情報を取得します。<BR>
	 * コマンド情報はツリー構造で表現されます。
	 * 
	 * 
	 * @return ツリー構造で表現された一括制御コマンドの情報
	 * @throws HinemosUnknown
	 * 
	 */
	public TreeItem getTree() throws HinemosUnknown {
		JpaTransactionManager jtm = null;
		SelectMaster select = new SelectMaster();
		TreeItem treeItem = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			treeItem = select.getTree();
			jtm.commit();
		} catch (Exception e) {
			m_log.warn("getTree() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return treeItem;
	}

	/**
	 * 一括制御のマスタ情報を更新します。
	 * 
	 * @param info
	 * @param parentInfo
	 * 
	 * @throws HinemosUnknown
	 */
	public void updateMaster(ItemInfo info, ItemInfo parentInfo) throws HinemosUnknown {
		JpaTransactionManager jtm = null;
		ModifyMaster modify = new ModifyMaster();

		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			modify.update(info, parentInfo);
			jtm.commit();
		} catch (EntityExistsException e) {
			outputErrorUpdateMaster(info, parentInfo, e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (Exception e) {
			outputErrorUpdateMaster(info, parentInfo, e);
			m_log.warn("updateMaster() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
	}

	/**
	 * 一括制御マスタ情報の更新時にエラーを出力します。
	 * 
	 * @param info
	 * @param parentInfo
	 * @param e
	 */
	private void outputErrorUpdateMaster(ItemInfo info, ItemInfo parentInfo, Exception e) {

		if(info == null || parentInfo == null){
			return;
		}

		String id = "";
		String parentId = "";
		if(info.getType() == ItemConstant.TYPE_TYPE){
			id = ((TypeInfo)info).getId();
		}
		else if(info.getType() == ItemConstant.TYPE_COMMAND){
			id = ((CommandInfo)info).getId();
			parentId = ((TypeInfo)parentInfo).getId();
		}
		else if(info.getType() == ItemConstant.TYPE_COMMAND_PARAMETER){
			id = ((CommandParameterInfo)info).getId();
			parentId = ((CommandInfo)parentInfo).getId();
		}
		else if(info.getType() == ItemConstant.TYPE_PARAMETER){
			id = ((ParameterInfo)info).getId();
		}
		else if(info.getType() == ItemConstant.TYPE_SELECT_ITEM){
			id = String.valueOf(((ParameterSelectInfo)info).getOrderNo());
			parentId = ((ParameterSelectInfo)info).getId();
		}

		AplLogger apllog = new AplLogger("CR","cr");
		String[] args = { String.valueOf(info.getType()), id, parentId };
		apllog.put("SYS", "007", args);
		m_log.debug("updateMaster() : " + e.getMessage());
	}

	/**
	 * 一括制御マスタ情報を削除します。
	 * 
	 * @param info
	 * @param parentInfo
	 * @throws HinemosUnknown
	 * @throws CollectiveRunNotFound
	 */
	public void deleteMaster(ItemInfo info, ItemInfo parentInfo) throws HinemosUnknown, CollectiveRunNotFound {
		JpaTransactionManager jtm = null;
		DeleteMaster delete = new DeleteMaster();

		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			delete.delete(info, parentInfo);
			jtm.commit();
		} catch (CollectiveRunNotFound e) {
			outputErrorDeleteMaster(info, parentInfo, e);
			jtm.rollback();
			throw e;
		} catch (Exception e) {
			outputErrorDeleteMaster(info, parentInfo, e);
			m_log.warn("deleteMaster() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
	}

	/**
	 * 一括制御マスタ情報の削除時にエラーを出力します。
	 * 
	 * @param info
	 * @param parentInfo
	 * @param e
	 */
	private void outputErrorDeleteMaster(ItemInfo info, ItemInfo parentInfo, Exception e) {

		if(info == null || parentInfo == null){
			return;
		}

		String id = "";
		String parentId = "";
		if(info.getType() == ItemConstant.TYPE_TYPE){
			id = ((TypeInfo)info).getId();
		}
		else if(info.getType() == ItemConstant.TYPE_COMMAND){
			id = ((CommandInfo)info).getId();
			parentId = ((TypeInfo)parentInfo).getId();
		}
		else if(info.getType() == ItemConstant.TYPE_COMMAND_PARAMETER){
			id = ((CommandParameterInfo)info).getId();
			parentId = ((CommandInfo)parentInfo).getId();
		}
		else if(info.getType() == ItemConstant.TYPE_PARAMETER){
			id = ((ParameterInfo)info).getId();
		}
		else if(info.getType() == ItemConstant.TYPE_SELECT_ITEM){
			id = String.valueOf(((ParameterSelectInfo)info).getOrderNo());
			parentId = ((ParameterSelectInfo)info).getId();
		}

		AplLogger apllog = new AplLogger("CR","cr");
		String[] args = { String.valueOf(info.getType()), id, parentId };
		apllog.put("SYS", "008", args);
		m_log.debug("outputErrorDeleteMaster() : " + e.getMessage());
	}

	/**
	 * 一括制御マスタ情報を更新します。<BR>
	 * 表示順を上下させる際に使用します。
	 * 
	 * @param info1
	 * @param info2
	 * @param parentInfo
	 * 
	 * @throws HinemosUnknown
	 */
	public void changeOrder(ItemInfo info1, ItemInfo info2, ItemInfo parentInfo) throws HinemosUnknown {
		JpaTransactionManager jtm = null;
		ModifyMaster modify = new ModifyMaster();

		try{
			jtm = new JpaTransactionManager();
			jtm.begin();
			modify.changeOrder(info1, info2, parentInfo);
			jtm.commit();
		} catch (EntityExistsException e) {
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch(Exception e) {
			m_log.warn("changeOrder() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
	}

	/**
	 * 一括制御[パラメータ]ビューの種別に表示する文字列のリスト取得します。
	 * 
	 * 
	 * @return 種別文字列
	 * @throws HinemosUnknown
	 */
	public ArrayList<String> getTypeStringList() throws HinemosUnknown {
		JpaTransactionManager jtm = null;
		ArrayList<String> typeStringList = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			typeStringList = SelectMaster.getTypeStringList();
			jtm.commit();
		} catch (Exception e) {
			m_log.warn("getTypeStringList() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return typeStringList;
	}

	/**
	 * 種別のハッシュマップ（key : 種別ID, value : 種別文字列）を取得します。
	 * 
	 * 
	 * @return 種別IDと種別文字列のハッシュマップ
	 * 
	 * @throws HinemosUnknown
	 */
	public HashMap<String, String> getTypeHashMap() throws HinemosUnknown {
		JpaTransactionManager jtm = null;
		HashMap<String, String> typeHashMap = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			typeHashMap = SelectMaster.getTypeHashMap();
			jtm.commit();
		} catch (Exception e) {
			m_log.warn("getTypeHashMap() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return typeHashMap;
	}

	/**
	 * ファシリティIDが使用されているかチェックします。
	 * <P>
	 * 使用されていたら、UsedFacility がスローされる。
	 * 
	 * 
	 * @param facilityId ファシリティ
	 * @throws UsedFacility
	 * 
	 * @see com.clustercontrol.commons.session.CheckFacility#isUseFacilityId(java.lang.String)
	 * @see com.clustercontrol.bean.PluginConstant;
	 */
	@Override
	public void isUseFacilityId(String facilityId) throws UsedFacility {
		JpaTransactionManager jtm = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();

			// ファシリティIDが使用されているセッションを取得する。
			List<CrunSessionEntity> collectionSession
			= QueryUtil.getCrunSessionByFaciiltyId(facilityId);

			Iterator<CrunSessionEntity> itrCrunSessionEntity = collectionSession.iterator();
			while(itrCrunSessionEntity.hasNext()){
				//セッションを取得
				CrunSessionEntity session = itrCrunSessionEntity.next();
				//セッションの実行状態が終了の場合、UsedFacilityをスロー
				if(session.getStatus().intValue() != StatusConstant.TYPE_END){
					throw new UsedFacility(PluginConstant.TYPE_COLLECTIVERUN);
				}
			}

			// ファシリティIDが使用されているセッションディテールを取得する。
			List<CrunSessionDetailEntity> collectionSessionDetail
			= QueryUtil.getCrunSessionDetailByFacilityId(facilityId);
			Iterator<CrunSessionDetailEntity> itrCrunSessionDetailEntity = collectionSessionDetail.iterator();
			while(itrCrunSessionDetailEntity.hasNext()){
				//セッションディテールを取得
				CrunSessionDetailEntity sessionDetail = itrCrunSessionDetailEntity.next();
				//セッションディテールの実行状態が終了の場合、UsedFacilityをスロー
				if(sessionDetail.getStatus().intValue() != StatusConstant.TYPE_END){
					throw new UsedFacility(PluginConstant.TYPE_COLLECTIVERUN);
				}
			}
			jtm.commit();
		} catch (Exception e) {
			m_log.warn("isUseFacilityId() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			try {
				jtm.rollback();
			} catch (Exception e1) {
			}
		} finally {
			try {
				jtm.close();
			} catch (Exception e1) {
			}
		}
	}

	/**
	 * コマンドIDを取得します。
	 * 
	 * 
	 * @param name 名前
	 * @return コマンドID
	 * @throws HinemosUnknown
	 */
	public String getCommandId(String typeId, String name) throws HinemosUnknown {
		JpaTransactionManager jtm = null;
		String ret = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();

			//名前からコマンドIDを取得
			ret = OperationRun.getCommandId(typeId, name);

			jtm.commit();
		} catch (Exception e) {
			m_log.warn("getCommandId() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return ret;
	}

	/**
	 * コマンドマスタを取得します。
	 * 
	 * 
	 * @param id コマンドID
	 * @return コマンド種別ID
	 * @throws CollectiveRunNotFound
	 * @throws HinemosUnknown
	 */
	public CommandInfo getCommandMst(String id) throws CollectiveRunNotFound, HinemosUnknown{
		JpaTransactionManager jtm = null;
		CommandInfo mst = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();

			//コマンドIDからコマンドマスタを取得
			CrunCmdMstEntity cmd = QueryUtil.getCrunCmdMstPK(id);

			//コマンド情報をコマンドマスタとして生成
			mst = new CommandInfo();
			mst.setId(cmd.getCommandId());
			mst.setCommand(cmd.getCommand());
			mst.setName(cmd.getNameId());
			mst.setCommandTypeId(cmd.getCrunTypeMstEntity().getTypeId());

			jtm.commit();
		} catch (CollectiveRunNotFound e) {
			jtm.rollback();
			throw e;
		} catch (Exception e) {
			m_log.warn("getCommandMst() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return mst;
	}
	/**
	 * セッション情報を取得します。
	 * 
	 * 
	 * @param id セッションID
	 * @return セッション情報
	 * @throws CollectiveRunNotFound
	 * @throws HinemosUnknown
	 */
	public SessionInfo getSessionInfo(String id) throws CollectiveRunNotFound, HinemosUnknown{
		JpaTransactionManager jtm = null;
		SessionInfo info = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();

			//セッションIDからセッション(実行履歴)を取得
			CrunSessionEntity session = QueryUtil.getCrunSessionPK(id);

			//セッション情報を生成
			info = new SessionInfo();
			info.setId(session.getSessionId());
			if(session.getStartDate() != null){
				info.setStartDate(session.getStartDate().getTime());
			}
			if(session.getEndDate() != null){
				info.setEndDate(session.getEndDate().getTime());
			}
			info.setTypeId(session.getTypeId());
			info.setCommandId(session.getCommandId());
			info.setFacilityId(session.getFacilityId());
			info.setScopeText(session.getScopeText());
			info.setNodeCount(session.getNodeCnt());
			info.setWaitCount(session.getWaitCnt());
			info.setRunnningCount(session.getRunningCnt());
			info.setEndNormalCount(session.getEndNomalCnt());
			info.setEndWarningCount(session.getEndWarningCnt());
			info.setEndAbnormalCount(session.getEndAbnomalCnt());
			info.setStatus(session.getStatus());

			for(CrunSessionParamEntity sessionParam : session.getCrunSessionParamEntities()){
				SessionParameterInfo paramInfo = new SessionParameterInfo();
				paramInfo.setId(sessionParam.getId().getParamId());
				paramInfo.setSessionId(sessionParam.getId().getSessionId());
				paramInfo.setValue(sessionParam.getParamValue());
				info.addParam(paramInfo);
			}
			jtm.commit();
		} catch (CollectiveRunNotFound e) {
			jtm.rollback();
			throw e;
		} catch (Exception e) {
			m_log.warn("getSessionInfo() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return info;
	}
	/**
	 * パラメータIDからパラメータ種別を取得します。
	 * 
	 * 
	 * @param id セッションID
	 * @return セッション情報
	 * @throws CollectiveRunNotFound
	 * @throws HinemosUnknown
	 */
	public Integer getParameterType(String id) throws CollectiveRunNotFound, HinemosUnknown{
		JpaTransactionManager jtm = null;
		Integer ret = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();

			CrunParamMstEntity param = QueryUtil.getCrunParamMstPK(id);
			ret = param.getParamType();
			jtm.commit();
		} catch (CollectiveRunNotFound e) {
			jtm.rollback();
			throw e;
		} catch (Exception e) {
			m_log.warn("getParameterType() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return ret;
	}
	/**
	 * コマンド種別IDからコマンド種別マスタを取得します。
	 * 
	 * 
	 * @param id コマンド種別ID
	 * @return コマンド種別情報
	 * @throws CollectiveRunNotFound
	 * @throws HinemosUnknown
	 */
	public CommandTypeInfo getCommandTypeMst(String id) throws CollectiveRunNotFound, HinemosUnknown{
		JpaTransactionManager jtm = null;
		CommandTypeInfo ret = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();

			CrunTypeMstEntity type = QueryUtil.getCrunTypeMstPK(id);
			ret = new CommandTypeInfo();
			ret.setId(type.getTypeId());
			ret.setNameId(type.getNameId());
			ret.setOrderNo(type.getOrderNo());

			for(CrunCmdMstEntity entity : type.getCrunCmdMstEntities()){
				//コマンド情報をコマンドマスタとして生成
				CommandInfo cmdMst = new CommandInfo();
				cmdMst.setId(entity.getCommandId());
				cmdMst.setCommand(entity.getCommand());
				cmdMst.setName(entity.getNameId());
				cmdMst.setCommandTypeId(entity.getCrunTypeMstEntity().getTypeId());
				cmdMst.setOrderNo(entity.getOrderNo().intValue());
				ret.addCmdMst(cmdMst);
			}
			jtm.commit();
		} catch (CollectiveRunNotFound e) {
			jtm.rollback();
			throw e;
		} catch (Exception e) {
			m_log.warn("getCommandTypeMst() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return ret;
	}
	/**
	 * コマンドIDからコマンドパラメータマスタを取得します。
	 * 
	 * 
	 * @param commandId コマンドID
	 * @return コマンドパラメータマスタ
	 * @throws HinemosUnknown
	 */
	public ArrayList<CommandParameterInfo> getCommandParameterMstList(String commandId) throws HinemosUnknown{
		JpaTransactionManager jtm = null;
		ArrayList<CommandParameterInfo> ret = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();

			List<CrunCmdParamMstEntity> cmdParamList
			= QueryUtil.getCrunCmdParamMstByCommandId(commandId);

			ret = new ArrayList<CommandParameterInfo>();

			for(CrunCmdParamMstEntity cmdParamMstEntity : cmdParamList){
				CommandParameterInfo mst = new CommandParameterInfo();
				mst.setId(cmdParamMstEntity.getId().getParamId());

				CrunParamMstEntity paramMstEntity = cmdParamMstEntity.getCrunParamMstEntity();
				ParameterInfo paramMst = new ParameterInfo();
				paramMst.setId(paramMstEntity.getParamId());
				paramMst.setParamType(paramMstEntity.getParamType());
				paramMst.setName(paramMstEntity.getNameId());

				Collection<CrunParamSelectMstEntity> selectMsts = paramMstEntity.getCrunParamSelectMstEntities();
				for(CrunParamSelectMstEntity paramSelectMst : selectMsts){
					ParameterSelectInfo selectMst = new ParameterSelectInfo();
					selectMst.setId(paramSelectMst.getId().getParamId());
					selectMst.setName(paramSelectMst.getNameId());
					selectMst.setOrderNo(paramSelectMst.getId().getOrderNo());
					selectMst.setValue(paramSelectMst.getParamValue());
					paramMst.addSelectMst(selectMst);
				}

				mst.setParamMst(paramMst);
				ret.add(mst);
			}

			jtm.commit();
		} catch (Exception e) {
			m_log.warn("getCommandParameterMstList() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return ret;
	}

	/**
	 * パラメータIDからパラメータ選択マスタを取得します。
	 * 
	 * 
	 * @param commandId コマンドID
	 * @return パラメータ選択マスタ
	 * @throws HinemosUnknown
	 */
	public ArrayList<ParameterSelectInfo> getParameterSelectMstList(String paramId) throws HinemosUnknown{
		JpaTransactionManager jtm = null;
		ArrayList<ParameterSelectInfo> ret = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();

			List<CrunParamSelectMstEntity> paramSelectList
			= QueryUtil.getCrunParamSelectMstByParamId(paramId);

			ret = new ArrayList<ParameterSelectInfo>();

			for(CrunParamSelectMstEntity paramSelectMstEntity : paramSelectList){
				ParameterSelectInfo info = new ParameterSelectInfo();
				info.setId(paramSelectMstEntity.getId().getParamId());
				info.setName(paramSelectMstEntity.getNameId());
				info.setOrderNo(paramSelectMstEntity.getId().getOrderNo());
				info.setValue(paramSelectMstEntity.getParamValue());
				ret.add(info);
			}

			jtm.commit();
		} catch (Exception e) {
			m_log.warn("getParameterSelectMstList() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return ret;
	}
}
