/*

Copyright (C) 2011 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.logfile.session;

import java.util.ArrayList;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.MonitorNotFound;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.calendar.bean.CalendarInfo;
import com.clustercontrol.calendar.session.CalendarControllerBean;
import com.clustercontrol.commons.util.JpaTransactionManager;
import com.clustercontrol.logfile.factory.SelectMonitorLogfile;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.repository.session.RepositoryControllerBean;

/**
 * ログファイル監視の管理を行う Session Bean <BR>
 * クライアントからの Entity Bean へのアクセスは、Session Bean を介して行います。
 * 
 */
public class MonitorLogfileControllerBean {

	/** ログ出力のインスタンス。 */
	private static Log m_log = LogFactory.getLog( MonitorLogfileControllerBean.class );


	/**
	 * ログファイル監視一覧リストを返します。
	 * 
	 * 
	 * @return Objectの2次元配列
	 * @throws MonitorNotFound
	 * @throws InvalidRole
	 * @throws HinemosUnknown
	 */
	public ArrayList<MonitorInfo> getLogfileList() throws MonitorNotFound, InvalidRole, HinemosUnknown{

		JpaTransactionManager jtm = null;
		SelectMonitorLogfile logfile = new SelectMonitorLogfile();
		ArrayList<MonitorInfo> list = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			list = logfile.getMonitorList(HinemosModuleConstant.MONITOR_LOGFILE);
			jtm.commit();
		} catch (InvalidRole e) {
			jtm.rollback();
			throw e;
		} catch (MonitorNotFound e) {
			jtm.rollback();
			throw e;
		} catch (Exception e) {
			m_log.warn("getLogfileList() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return list;
	}

	private static Object logfileLock = new Object();
	private static ArrayList<MonitorInfo> logfileCache = null;
	
	public static void clearLogfileCache () {
		m_log.info("clearLogfileCache()");
		synchronized (logfileLock) {
			logfileCache = null;
		}
	}
	
	/**
	 * 
	 * <注意！> このメソッドはAgentユーザ以外で呼び出さないこと！
	 * <注意！> キャッシュの都合上、Agentユーザ以外から呼び出すと、正常に動作しません。
	 * 
	 * facilityIDごとのログファイル監視一覧リストを返します。
	 * withCalendarをtrueにするとMonitorInfoのcalendarDTOに値が入ります。
	 * 
	 * 
	 * @return Objectの2次元配列
	 * @throws HinemosUnknown
	 * @throws MonitorNotFound
	 * 
	 */
	public ArrayList<MonitorInfo> getLogfileListForFacilityId (String facilityId, boolean withCalendar)
			throws MonitorNotFound, HinemosUnknown {
		ArrayList<MonitorInfo> ret = new ArrayList<MonitorInfo>();
		JpaTransactionManager jtm = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			
			ArrayList<MonitorInfo> monitorList = null;
			synchronized(logfileLock) {
				if (logfileCache == null) {
					m_log.info("getLogfileListForFacilityId : create logfileCache");
					logfileCache = getLogfileList();
				}
				// Cacheがrefreshされるタイミングによっては、
				// 下のfor文でNullPointerExceptionが発生するので、
				// monitorListに値を移しておく。
				monitorList = logfileCache;
			}
			
			for (MonitorInfo monitorInfo : monitorList) {
				String scope = monitorInfo.getFacilityId();
				ArrayList<String> facilityIdList
				= new RepositoryControllerBean().getExecTargetFacilityIdList(scope, monitorInfo.getOwnerRoleId());
				if (facilityIdList != null && facilityIdList.contains(facilityId)) {
					if (withCalendar) {
						String calendarId = monitorInfo.getCalendarId();
						try {
							CalendarInfo calendar = new CalendarControllerBean().getCalendarFull(calendarId);
							monitorInfo.setCalendar(calendar);
						} catch (Exception e) {
							m_log.warn("getLogfileList() : "
									+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
							throw new HinemosUnknown(e.getMessage(), e);
						}
					}
					ret.add(monitorInfo);
				}
			}
			jtm.commit();
		} catch (MonitorNotFound e) {
			jtm.rollback();
			throw e;
		} catch (HinemosUnknown e) {
			jtm.rollback();
			throw e;
		} catch (Exception e) {
			m_log.warn("getLogfileListForFacilityId() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}

		return ret;
	}
}
