/*

Copyright (C) 2011 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.notify.util;

import java.text.SimpleDateFormat;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.commons.util.HinemosProperties;
import com.clustercontrol.fault.FacilityNotFound;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.MonitorNotFound;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.session.MonitorSettingControllerBean;
import com.clustercontrol.notify.bean.OutputBasicInfo;
import com.clustercontrol.notify.model.NotifyInfoEntity;
import com.clustercontrol.repository.model.FacilityEntity;
import com.clustercontrol.repository.model.NodeEntity;
import com.clustercontrol.repository.session.RepositoryControllerBean;
import com.clustercontrol.repository.util.FacilityUtil;

/**
 * 通知に関するUtilityクラス<br/>
 * 
 * @author takahatats
 */
public class NotifyUtil {

	private static Log log = LogFactory.getLog(NotifyUtil.class);

	/** 監視キー */
	private static final String _KEY_PRIORITY = "PRIORITY";
	private static final String _KEY_PRIORITY_NUM = "PRIORITY_NUM";
	private static final String _KEY_PRIORITY_JP = "PRIORITY_JP";
	private static final String _KEY_PRIORITY_EN = "PRIORITY_EN";

	private static final String _KEY_PLUGIN_ID = "PLUGIN_ID";
	private static final String _KEY_MONITOR_ID = "MONITOR_ID";
	private static final String _KEY_MONITOR_DESCRIPTION = "MONITOR_DESCRIPTION";
	private static final String _KEY_FACILITY_ID = "FACILITY_ID";
	private static final String _KEY_SCOPE = "SCOPE";

	private static final String _KEY_FACILITY_NAME = "FACILITY_NAME";
	private static final String _KEY_IP_ADDRESS = "IP_ADDRESS";
	private static final String _KEY_IP_ADDRESS_V4 = "IP_ADDRESS_V4";
	private static final String _KEY_IP_ADDRESS_V6 = "IP_ADDRESS_V6";
	private static final String _KEY_NODE_NAME = "NODE_NAME";

	private static final String _KEY_GENERATION_DATE = "GENERATION_DATE";
	private static final String _KEY_APPLICATION = "APPLICATION";
	private static final String _KEY_MESSAGE_ID = "MESSAGE_ID";
	private static final String _KEY_MESSAGE = "MESSAGE";
	private static final String _KEY_ORG_MESSAGE = "ORG_MESSAGE";

	private static final String _KEY_MONITOR_OWNER_ROLE_ID = "MONITOR_OWNER_ROLE_ID";
	private static final String _KEY_CALENDAR_ID = "CALENDAR_ID";

	/** 通知キー */
	private static final String _KEY_NOTIFY_ID = "NOTIFY_ID";
	private static final String _KEY_NOTIFY_DESCRIPTION = "NOTIFY_DESCRIPTION";

	/** 日時フォーマット。 */
	private static String SUBJECT_DATE_FORMAT ;

	/** 日時フォーマットデフォルト。 */
	private static String SUBJECT_DATE_FORMAT_DEFAULT = "yyyy-MM-dd HH:mm:ss" ;

	static {
		SUBJECT_DATE_FORMAT = HinemosProperties.getProperty("common.notify.date.format", SUBJECT_DATE_FORMAT_DEFAULT);
		if(log.isDebugEnabled()){
			log.debug("TextReplacer.static SUBJECT_DATE_FORMAT = " + SUBJECT_DATE_FORMAT);
		}
	}

	/**
	 * 通知情報をハッシュとして返す。
	 * @param outputInfo 通知情報
	 * @return 通知情報のハッシュ
	 */
	public static Map<String, String> createParameter(OutputBasicInfo outputInfo) {
		return createParameter(outputInfo, null);
	}
	
	/**
	 * 通知情報をハッシュとして返す。
	 * @param outputInfo 通知情報
	 * @param notifyInfo 通知設定情報
	 * @return 通知情報のハッシュ
	 */
	public static Map<String, String> createParameter(OutputBasicInfo outputInfo, NotifyInfoEntity notifyInfo) {
		Map<String, String> param = null;
		SimpleDateFormat sdf = null;
		param = new HashMap<String, String>();

		if (outputInfo != null) {
			sdf = new SimpleDateFormat(SUBJECT_DATE_FORMAT);

			param.put(_KEY_PRIORITY_NUM, String.valueOf(outputInfo.getPriority()));
			if (PriorityConstant.typeToString(outputInfo.getPriority()) != null) {
				param.put(_KEY_PRIORITY, PriorityConstant.typeToString(outputInfo.getPriority()));
			} else {
				param.put(_KEY_PRIORITY, null);
			}
			if (PriorityConstant.typeToStringJP(outputInfo.getPriority()) != null) {
				param.put(_KEY_PRIORITY_JP, PriorityConstant.typeToStringJP(outputInfo.getPriority()));
			} else {
				param.put(_KEY_PRIORITY_JP, null);
			}
			if (PriorityConstant.typeToStringEN(outputInfo.getPriority()) != null) {
				param.put(_KEY_PRIORITY_EN, PriorityConstant.typeToStringEN(outputInfo.getPriority()));
			} else {
				param.put(_KEY_PRIORITY_EN, null);
			}

			String pluginId = outputInfo.getPluginId();
			param.put(_KEY_PLUGIN_ID, pluginId);
			String monitorId = outputInfo.getMonitorId();
			param.put(_KEY_MONITOR_ID, outputInfo.getMonitorId());
			if (monitorId != null && pluginId != null && pluginId.startsWith("MON_")) {
				MonitorSettingControllerBean controller = new MonitorSettingControllerBean();
				try {
					MonitorInfo monitorInfo = controller.getMonitor(monitorId, pluginId);
					param.put(_KEY_MONITOR_DESCRIPTION, monitorInfo.getDescription());
					param.put(_KEY_CALENDAR_ID, monitorInfo.getCalendarId());
					param.put(_KEY_MONITOR_OWNER_ROLE_ID, monitorInfo.getOwnerRoleId());
				} catch (MonitorNotFound e) {
					log.debug("createParameter() : monitor not found. " + e.getMessage());
				} catch (HinemosUnknown e) {
					log.debug("createParameter() : HinemosUnknown. " + e.getMessage());
				} catch (InvalidRole e) {
					log.debug("createParameter() : InvalidRole. " + e.getMessage());
				}
			} else {
				param.put(_KEY_MONITOR_DESCRIPTION, "");
				param.put(_KEY_CALENDAR_ID, "");
				param.put(_KEY_MONITOR_OWNER_ROLE_ID, "");
			}


			param.put(_KEY_FACILITY_ID, outputInfo.getFacilityId());
			param.put(_KEY_SCOPE, outputInfo.getScopeText());

			if (outputInfo.getGenerationDate() != null) {
				param.put(_KEY_GENERATION_DATE, sdf.format(outputInfo.getGenerationDate()));
			} else {
				param.put(_KEY_GENERATION_DATE, null);
			}
			param.put(_KEY_APPLICATION, outputInfo.getApplication());
			param.put(_KEY_MESSAGE_ID, outputInfo.getMessageId());
			param.put(_KEY_MESSAGE, outputInfo.getMessage());
			param.put(_KEY_ORG_MESSAGE, outputInfo.getMessageOrg());

			if (outputInfo.getFacilityId() != null) {
				try {
					FacilityEntity facility
					= new RepositoryControllerBean().getFacilityEntityByPK(outputInfo.getFacilityId());
					param.put(_KEY_FACILITY_NAME, facility.getFacilityName());
					if (FacilityUtil.isNode(facility)) {
						NodeEntity node = facility.getNodeEntity();
						param.put(_KEY_NODE_NAME, node.getNodeName());
						param.put(_KEY_IP_ADDRESS_V4, node.getIpAddressV4());
						param.put(_KEY_IP_ADDRESS_V6, node.getIpAddressV6());
						param.put(_KEY_IP_ADDRESS, node.getIpAddressVersion() == 4 ? node.getIpAddressV4() : node.getIpAddressV6());
					}
				} catch (FacilityNotFound e) {
					log.debug("createParameter() : facility not found. " + e.getMessage());
				} catch (InvalidRole e) {
					log.debug("createParameter() : InvalidRole. " + e.getMessage());
				} catch (HinemosUnknown e) {
					log.debug("createParameter() : HinemosUnknown. " + e.getMessage());
				} catch (Exception e) {
					log.warn("facility not found. (" + outputInfo.getFacilityId() + ") : "
							+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
				}
			}
		}

		if (notifyInfo != null) {
			param.put(_KEY_NOTIFY_ID, String.valueOf(notifyInfo.getNotifyId()));
			param.put(_KEY_NOTIFY_DESCRIPTION, notifyInfo.getDescription());
		}
		
		if(log.isTraceEnabled()){
			for(String key : param.keySet()){
				log.trace("createParameter() : param[" + key + "]=" + param.get(key));
			}
		}

		return param;
	}
}
