/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.performance.util;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Date;
import java.util.concurrent.ConcurrentHashMap;

import javax.persistence.EntityExistsException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.commons.util.HinemosProperties;
import com.clustercontrol.commons.util.JpaTransactionManager;
import com.clustercontrol.performance.bean.CollectedDataErrorTypeConstant;
import com.clustercontrol.performance.bean.PerfData;
import com.clustercontrol.performance.bean.Sample;
import com.clustercontrol.performance.model.CalculatedDataEntity;

/**
 * 性能情報を登録するユーティティクラス<BR>
 *
 * @version 4.0.0
 * @since 4.0.0
 */
public class PerformanceDataUtil {

	/** ログ出力のインスタンス。 */
	private static Log m_log = LogFactory.getLog( PerformanceDataUtil.class );

	// NaNのときに昔のデータを利用するためのキャッシュ。
	private static ConcurrentHashMap<PerfKey, PerfValue> perfCache =
			new ConcurrentHashMap<PerfKey, PerfValue>();
	private static int useOld = -1;

	/**
	 * 性能情報を登録するために Queue に put する
	 * 
	 * @param sample 性能情報
	 * @throws EntityExistsException
	 */
	public static void put(Sample sample) throws EntityExistsException {
		m_log.debug("put() start");

		JpaTransactionManager jtm = new JpaTransactionManager();

		//  for debug
		if(m_log.isDebugEnabled()){
			m_log.debug("put() collectorId = " + sample.getCollectorId() + ", dateTime = " + sample.getDateTime());
			ArrayList<PerfData> list = sample.getPerfDataList();
			for (PerfData data : list){
				m_log.info("put() list facilityId = " + data.getFacilityId() + ", value = " + data.getValue());
			}
		}

		try{

			ArrayList<PerfData> list = sample.getPerfDataList();
			for (PerfData data : list){
				m_log.debug("persist itemCode = " + data.getItemCode());
				// インスタンス生成
				CalculatedDataEntity entity = new CalculatedDataEntity(
						sample.getCollectorId(),
						data.getItemCode(),
						data.getDisplayName(),
						sample.getDateTime() == null ? null : new Timestamp(sample.getDateTime().getTime()),
								data.getFacilityId());
				// 重複チェック
				jtm.checkEntityExists(CalculatedDataEntity.class, entity.getId());

				PerfKey perfKey = new PerfKey(
						sample.getCollectorId(),
						data.getItemCode(),
						data.getDisplayName(),
						data.getFacilityId());

				// 性能値
				if(data.getErrorType() == CollectedDataErrorTypeConstant.NOT_ENOUGH_COLLECT_COUNT
						|| data.getErrorType() == CollectedDataErrorTypeConstant.POLLING_TIMEOUT
						|| data.getErrorType() == CollectedDataErrorTypeConstant.FACILITY_NOT_FOUND
						|| data.getErrorType() == CollectedDataErrorTypeConstant.UNKNOWN
						|| Double.isNaN(data.getValue())){
					/*
					 * 性能値が算出不能だった場合は前の値を利用する。
					 * 660秒以上前のデータは利用しない。
					 */
					try {
						if (useOld < 0) {
							useOld = Integer.parseInt(
									HinemosProperties.getProperty("performance.use.old", "660"));
						}
					} catch (Exception e) {
						m_log.warn("put() : "
								+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
					}
					PerfValue perfValue = perfCache.get(perfKey);
					if (sample.getDateTime() != null &&
							perfValue != null &&
							perfValue.getTime() != null &&
							sample.getDateTime().getTime() - perfValue.getTime().getTime()
							< useOld * 1000) {
						m_log.info("insert() : use old perf-data " + perfKey);
						entity.setValue(perfValue.getValue());
					}
				} else {
					// 性能値が算出可能だった場合はそのまま設定する
					entity.setValue(data.getValue());
					perfCache.put(
							perfKey,
							new PerfValue(sample.getDateTime(), data.getValue())
							);
				}
			}
		} catch (EntityExistsException e) {
			m_log.info("put() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage());
			throw e;
		}
		if(m_log.isDebugEnabled()){
			m_log.debug("insert() end : collectorId = " + sample.getCollectorId() + ", dateTime = " + sample.getDateTime());
		}
		m_log.debug("put() end");
	}
}

class PerfKey {
	private String collectorId = null;
	private String itemCode = null;
	private String displayName = null;
	private String facilityId = null;
	public PerfKey (String collectorId, String itemCode, String displayName, String facilityId) {
		this.collectorId = collectorId;
		this.itemCode = itemCode;
		this.displayName = displayName;
		this.facilityId = facilityId;
	}

	@Override
	public boolean equals(Object other) {
		PerfKey otherKey = (PerfKey) other;
		if (!collectorId.equals(otherKey.collectorId)){
			return false;
		}
		if (!itemCode.equals(otherKey.itemCode)){
			return false;
		}
		if (!displayName.equals(otherKey.displayName)){
			return false;
		}
		if (!facilityId.equals(otherKey.facilityId)){
			return false;
		}
		return true;
	}

	@Override
	public String toString() {
		return collectorId + "," + itemCode + "," + displayName + "," + facilityId;
	}

	@Override
	public int hashCode() {
		int result = collectorId.hashCode();
		result = 37 * result + itemCode.hashCode();
		result = 37 * result + displayName.hashCode();
		result = 37 * result + facilityId.hashCode();
		return result;
	}
}
class PerfValue {
	private Date time = null;
	private double value = 0;
	public PerfValue (Date time, double value) {
		this.time = time;
		this.value = value;
	}
	public Date getTime () {
		return time;
	}
	public double getValue() {
		return value;
	}
}
