/*

Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.winevent.session;

import java.util.ArrayList;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.calendar.bean.CalendarInfo;
import com.clustercontrol.calendar.session.CalendarControllerBean;
import com.clustercontrol.commons.util.JpaTransactionManager;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.MonitorNotFound;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.repository.session.RepositoryControllerBean;
import com.clustercontrol.winevent.factory.SelectMonitorWinEvent;

/**
 * 
 * @since 4.1
 */
public class MonitorWinEventControllerBean {

	private static Log m_log = LogFactory.getLog( MonitorWinEventControllerBean.class );

	/**
	 * Windowsイベント監視一覧リストを返します。
	 * 
	 * 
	 * @return Objectの2次元配列
	 * @throws MonitorNotFound
	 * @throws InvalidRole
	 * @throws HinemosUnknown
	 */
	public ArrayList<MonitorInfo> getWinEventList() throws MonitorNotFound, InvalidRole, HinemosUnknown{

		JpaTransactionManager jtm = null;
		ArrayList<MonitorInfo> list = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			list = new SelectMonitorWinEvent().getMonitorList(HinemosModuleConstant.MONITOR_WINEVENT);
			jtm.commit();
		} catch (MonitorNotFound e) {
			jtm.rollback();
			throw e;
		} catch (InvalidRole e) {
			jtm.rollback();
			throw e;
		} catch (Exception e) {
			m_log.warn("getWinEventList() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return list;
	}
	
	private static Object winEventLock = new Object();
	private static ArrayList<MonitorInfo> winEventCache = null;
	
	public static void clearWinEventCache () {
		m_log.info("clearWinEventCache()");
		synchronized (winEventLock) {
			winEventCache = null;
		}
	}
	
	/**
	 * 
	 * <注意！> このメソッドはAgentユーザ以外で呼び出さないこと！
	 * <注意！> キャッシュの都合上、Agentユーザ以外から呼び出すと、正常に動作しません。
	 * 
	 * 要求してきたエージェントのノードに設定されたWindowsイベント監視設定を返す
	 * @param requestedFacilityId エージェントが対応するノードのfacilityId
	 * @return Windowsイベント監視設定の一覧
	 * @throws InvalidRole
	 * @throws HinemosUnknown 予期せぬエラーが発生した場合
	 * 
	 */
	public ArrayList<MonitorInfo> getWinEventList(String requestedFacilityId) throws InvalidRole, HinemosUnknown {
		JpaTransactionManager jtm = null;
		ArrayList<MonitorInfo> list = new ArrayList<MonitorInfo>();
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			
			ArrayList<MonitorInfo> monitorList = null;
			synchronized(winEventLock) {
				if (winEventCache == null) {
					m_log.info("getWinEventList : create winEventCache");
					winEventCache = getWinEventList();
				}
				// Cacheがrefreshされるタイミングによっては、
				// 下のfor文でNullPointerExceptionが発生するので、
				// monitorListに値を移しておく。
				monitorList = winEventCache;
			}
			
			for (MonitorInfo monitorInfo : monitorList) {
				String scope = monitorInfo.getFacilityId();
				ArrayList<String> facilityIdList
				= new RepositoryControllerBean().getExecTargetFacilityIdList(scope, monitorInfo.getOwnerRoleId());
				if (facilityIdList != null && facilityIdList.contains(requestedFacilityId)) {
					String calendarId = monitorInfo.getCalendarId();
					if(calendarId != null){
						CalendarInfo calendar = new CalendarControllerBean().getCalendarFull(calendarId);
						monitorInfo.setCalendar(calendar);
					}
					list.add(monitorInfo);
				}
			}

			jtm.commit();
		} catch (InvalidRole e) {
			jtm.rollback();
			throw e;
		} catch (HinemosUnknown e) {
			jtm.rollback();
			throw e;
		} catch (Exception e) {
			m_log.warn("getWinEventList() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return list;
	}

}
