/*
Copyright (C) 2010 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.ws.monitor;

import java.util.ArrayList;

import javax.annotation.Resource;
import javax.xml.ws.WebServiceContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.InvalidUserPass;
import com.clustercontrol.fault.MonitorNotFound;
import com.clustercontrol.accesscontrol.bean.SystemPrivilegeInfo;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.snmptrap.bean.SnmpTrapMasterInfo;
import com.clustercontrol.snmptrap.entity.SnmpTrapMibMasterData;
import com.clustercontrol.snmptrap.session.MonitorSnmpTrapControllerBean;
import com.clustercontrol.ws.util.HttpAuthenticator;

/**
 * SNMPTRAP監視用のWebAPIエンドポイント
 */
@javax.jws.WebService(targetNamespace = "http://monitor.ws.clustercontrol.com")
public class MonitorSnmpTrapEndpoint {
	@Resource
	WebServiceContext wsctx;

	private static Log m_log = LogFactory.getLog( MonitorSnmpTrapEndpoint.class );
	private static Log m_opelog = LogFactory.getLog("HinemosOperation");

	/**
	 * echo(WebサービスAPI疎通用)
	 * 
	 * 権限必要なし（ユーザ名チェックのみ実施）
	 * 
	 * @param str
	 * @return
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws HinemosUnknown
	 */
	public String echo(String str) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		ArrayList<SystemPrivilegeInfo> systemPrivilegeList = new ArrayList<SystemPrivilegeInfo>();
		HttpAuthenticator.authCheck(wsctx, systemPrivilegeList);

		return str + ", " + str;
	}

	/**
	 * MIB一覧を取得します。<BR>
	 * 戻り値はMIBの名前(String)のArrayListです。
	 * 
	 * MonitorSettingRead権限が必要
	 * 
	 * @return MIB一覧
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public ArrayList<String> getMibList() throws HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("getMibList : ");
		ArrayList<SystemPrivilegeInfo> systemPrivilegeList = new ArrayList<SystemPrivilegeInfo>();
		HttpAuthenticator.authCheck(wsctx, systemPrivilegeList, false);

		// 認証済み操作ログ
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_MONITOR_SNMPTRAP + " Get MIB, Method=getMibList, User="
				+ HttpAuthenticator.getUserAccountString(wsctx));

		return new MonitorSnmpTrapControllerBean().getMibList();
	}

	/**
	 * 監視マスタ情報一覧を取得します。<BR>
	 * 戻り値はSnmpTrapMasterInfoのArrayListです。
	 * 
	 * MonitorSettingRead権限が必要
	 * 
	 * @return MIB一覧
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @see com.clustercontrol.snmptrap.bean.SnmpTrapMasterInfo
	 */
	public ArrayList<SnmpTrapMasterInfo> getMasterList(String mib) throws HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("getMasterList : mib=" + mib);
		ArrayList<SystemPrivilegeInfo> systemPrivilegeList = new ArrayList<SystemPrivilegeInfo>();
		HttpAuthenticator.authCheck(wsctx, systemPrivilegeList, false);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", MIB=");
		msg.append(mib);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_MONITOR_SNMPTRAP + " Get OID, Method=getMasterList, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return new MonitorSnmpTrapControllerBean().getMasterList(mib);
	}

	/**
	 * MIBの内容を取得します。<BR>
	 * 戻り値はMIBのDTO(SnmpTrapMibMasterData)です。
	 * 
	 * MonitorSettingRead権限が必要
	 * 
	 * @return MIB一覧
	 * @throws HinemosUnknown
	 * @throws MonitorNotFound
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public SnmpTrapMibMasterData getMib(String Mib) throws MonitorNotFound, HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("getMib : mib=" + Mib);
		ArrayList<SystemPrivilegeInfo> systemPrivilegeList = new ArrayList<SystemPrivilegeInfo>();
		HttpAuthenticator.authCheck(wsctx, systemPrivilegeList, false);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", MIB=");
		msg.append(Mib);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_MONITOR_SNMPTRAP + " Get MIB, Method=getMib, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return new MonitorSnmpTrapControllerBean().getMib(Mib);
	}

	/**
	 * MIB情報の登録
	 *
	 * MonitorSettingWrite権限が必要
	 *
	 * @param MibData
	 * @return
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public boolean addMibMaster(SnmpTrapMibMasterData MibData) throws HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("addMibMaster : snmpTrapMibMasterData=" + MibData);
		ArrayList<SystemPrivilegeInfo> systemPrivilegeList = new ArrayList<SystemPrivilegeInfo>();
		HttpAuthenticator.authCheck(wsctx, systemPrivilegeList, true);

		boolean ret = false;

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		if(MibData != null){
			msg.append(", MIB=");
			msg.append(MibData.getMib());
		}

		try {
			ret = new MonitorSnmpTrapControllerBean().addMibMaster(MibData);
		} catch (Exception e) {
			m_opelog.warn(HinemosModuleConstant.LOG_PREFIX_MONITOR_SNMPTRAP + " Add MIB Failed, Method=addMibMaster, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
			throw e;
		}
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_MONITOR_SNMPTRAP + " Add MIB, Method=addMibMaster, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return ret;
	}

	/**
	 * MIB情報の変更
	 * 
	 * MonitorSettingWrite権限が必要
	 * 
	 * @param MibData
	 * @return
	 * @throws HinemosUnknown
	 * @throws MonitorNotFound
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public boolean modifyMibMaster(SnmpTrapMibMasterData MibData) throws MonitorNotFound, HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("modifyMibMaster : snmpTrapMibMasterData=" + MibData);
		ArrayList<SystemPrivilegeInfo> systemPrivilegeList = new ArrayList<SystemPrivilegeInfo>();
		HttpAuthenticator.authCheck(wsctx, systemPrivilegeList, true);

		boolean ret = false;

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		if(MibData != null){
			msg.append(", MIB=");
			msg.append(MibData.getMib());
		}

		try {
			ret = new MonitorSnmpTrapControllerBean().modifyMibMaster(MibData);
		} catch (Exception e) {
			m_opelog.warn(HinemosModuleConstant.LOG_PREFIX_MONITOR_SNMPTRAP + " Change MIB Failed, Method=modifyMibMaster, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
			throw e;
		}
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_MONITOR_SNMPTRAP + " Change MIB, Method=modifyMibMaster, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return ret;
	}

	/**
	 * MIB情報の削除
	 * 
	 * MonitorSettingWrite権限が必要
	 * 
	 * @param mib
	 * @return
	 * @throws HinemosUnknown
	 * @throws MonitorNotFound
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public boolean deleteMibMaster(String Mib) throws MonitorNotFound, HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("deleteMibMaster : mib=" + Mib);
		ArrayList<SystemPrivilegeInfo> systemPrivilegeList = new ArrayList<SystemPrivilegeInfo>();
		HttpAuthenticator.authCheck(wsctx, systemPrivilegeList, true);

		boolean ret = false;

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", MIB=");
		msg.append(Mib);

		try {
			ret = new MonitorSnmpTrapControllerBean().deleteMibMaster(Mib);
		} catch (Exception e) {
			m_opelog.warn(HinemosModuleConstant.LOG_PREFIX_MONITOR_SNMPTRAP + " Delete MIB Failed, Method=deleteMibMaster, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
			throw e;
		}
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_MONITOR_SNMPTRAP + " Delete MIB, Method=deleteMibMaster, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return ret;
	}


	/**
	 * OID情報の登録
	 *
	 * MonitorSettingWrite権限が必要
	 *
	 * @param MibData
	 * @return
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public boolean addOidMaster(SnmpTrapMasterInfo Oid) throws HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("addOidMaster : snmpTrapMasterInfo="+ Oid);
		ArrayList<SystemPrivilegeInfo> systemPrivilegeList = new ArrayList<SystemPrivilegeInfo>();
		HttpAuthenticator.authCheck(wsctx, systemPrivilegeList, true);

		boolean ret = false;

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		if(Oid != null){
			msg.append(", MIB=");
			msg.append(Oid.getMib());
			msg.append(", Trap=");
			msg.append(Oid.getUei());
			msg.append(", OID=");
			msg.append(Oid.getTrapOid());
			msg.append(", GenericID=");
			msg.append(Oid.getGenericId());
			msg.append(", SpecificID=");
			msg.append(Oid.getSpecificId());
		}

		try {
			ret = new MonitorSnmpTrapControllerBean().addOidMaster(Oid);
		} catch (Exception e) {
			m_opelog.warn(HinemosModuleConstant.LOG_PREFIX_MONITOR_SNMPTRAP + " Add OID Failed, Method=addOidMaster, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
			throw e;
		}
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_MONITOR_SNMPTRAP + " Add OID, Method=addOidMaster, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return ret;
	}

	/**
	 * OID情報の変更
	 * 
	 * MonitorSettingWrite権限が必要
	 * 
	 * @param MibData
	 * @return
	 * @throws HinemosUnknown
	 * @throws MonitorNotFound
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public boolean modifyOidMaster(SnmpTrapMasterInfo Oid) throws MonitorNotFound, HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("modifyOidMaster : snmpTrapMasterInfo="+ Oid);
		ArrayList<SystemPrivilegeInfo> systemPrivilegeList = new ArrayList<SystemPrivilegeInfo>();
		HttpAuthenticator.authCheck(wsctx, systemPrivilegeList, true);

		boolean ret = false;

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		if(Oid != null){
			msg.append(", MIB=");
			msg.append(Oid.getMib());
			msg.append(", Trap=");
			msg.append(Oid.getUei());
			msg.append(", OID=");
			msg.append(Oid.getTrapOid());
			msg.append(", GenericID=");
			msg.append(Oid.getGenericId());
			msg.append(", SpecificID=");
			msg.append(Oid.getSpecificId());
		}
		try {
			ret = new MonitorSnmpTrapControllerBean().modifyOidMaster(Oid);
		} catch (Exception e) {
			m_opelog.warn(HinemosModuleConstant.LOG_PREFIX_MONITOR_SNMPTRAP + " Change OID Failed, Method=modifyOidMaster, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
			throw e;
		}
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_MONITOR_SNMPTRAP + " Change OID, Method=modifyOidMaster, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return ret;
	}

	/**
	 * OID情報の削除
	 * 
	 * MonitorSettingWrite権限が必要
	 * 
	 * @param mib
	 * @return
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws MonitorNotFound
	 */
	public boolean deleteOidMaster(SnmpTrapMasterInfo oid) throws InvalidUserPass, InvalidRole, HinemosUnknown, MonitorNotFound {
		m_log.debug("deleteOidMaster : snmpTrapMasterInfo="+ oid);
		ArrayList<SystemPrivilegeInfo> systemPrivilegeList = new ArrayList<SystemPrivilegeInfo>();
		HttpAuthenticator.authCheck(wsctx, systemPrivilegeList, true);

		boolean ret = false;

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		if(oid != null){
			msg.append(", MIB=");
			msg.append(oid.getMib());
			msg.append(", Trap=");
			msg.append(oid.getUei());
			msg.append(", OID=");
			msg.append(oid.getTrapOid());
			msg.append(", GenericID=");
			msg.append(oid.getGenericId());
			msg.append(", SpecificID=");
			msg.append(oid.getSpecificId());
		}

		try {
			ret = new MonitorSnmpTrapControllerBean().deleteOidMaster(oid);
		} catch (Exception e) {
			m_opelog.warn(HinemosModuleConstant.LOG_PREFIX_MONITOR_SNMPTRAP + " Delete OID Failed, Method=deleteOidMaster, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
			throw e;
		}
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_MONITOR_SNMPTRAP + " Delete OID, Method=deleteOidMaster, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return ret;
	}
}
