#!/bin/bash

# Copyright (C) 2012 NTT DATA Corporation
# 
# This program is free software; you can redistribute it and/or
# Modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation, version 2.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details

# bash configuration
SCRIPT_DIR=$(cd $(dirname $0);pwd)
. ${SCRIPT_DIR}/../hinemos.cfg
. ${SCRIPT_DIR}/../sbin/hinemos_utility.sh

export PROG=`basename $0`
DIR=`dirname $0`
USER=`/usr/bin/whoami`
HOST=`hostname`
PROP_FILE=${SCRIPT_DIR}/../etc/scheduler-dbms.properties

cd ${HINEMOS_HOME}/var/log

########################################
#  Local Variable
########################################

# startup check interval [sec]
#   this variable is valid with -W option
STARTUP_CHECK_INTERVAL=1

# startup check timeout [sec]
#   this variable is valid with -W option
STARTUP_CHECK_TIMEOUT=300

# log file for monitor startup
STARTUP_MONITOR_FILE=${HINEMOS_LOG_DIR}/boot.log

# java version check
JAVA_CHECK_REGEX='java version "1.7.0.*"'
OPENJDK_CHECK_REGEX=OpenJDK

########################################
#  Local Message
########################################

# INFO
MSG_I001="removing queue data (asynchronous queue)..."
MSG_I002="waiting for Java Virtual Machine startup..."
MSG_I003="Java Virtual Machine started"

# WARN


# ERRORs
MSG_E001="PID file(/var/run/snmptrapd.pid) is found. stop snmptrapd service."
MSG_E002="startup timeout. please check status of Java Virtual Machine"

########################################
# Function
########################################

function usage {
	echo "usage : ${PROG} [-c|-F|-M|-W]"
	echo "options:"
	echo "  -c   clean temporally queue data before startup"
	echo "  -F   force startup without process check (dependency, current status)"
	echo "  -M   startup maintenance mode"
	echo "  -W   do not wait until operation completes"
}

function initResourceParameter()
{
	if [ "x${JAVA_FD_MAXNUM}" != "x" ]
	then
		ulimit -n ${JAVA_FD_MAXNUM}
	fi
	
	if [ "x${NET_CORE_RMEM_MAX}" != "x" ]
	then
		sysctl -q -w net.core.rmem_max=${NET_CORE_RMEM_MAX}
	fi
	
	if [ "x${NET_CORE_RMEM_DEF}" != "x" ]
	then
		sysctl -q -w net.core.rmem_default=${NET_CORE_RMEM_DEF}
	fi
}

function initClassPath()
{
	CLASSPATH=${HINEMOS_ETC_DIR}
	
	for CFG in `ls ${HINEMOS_ETC_DIR}/classpath.d | grep ".cfg$"`
	do
		for CP in `cat ${HINEMOS_ETC_DIR}/classpath.d/${CFG} | grep -v "^#" | grep -v "^$"`
		do
			CLASSPATH=${CLASSPATH}:${CP}
		done
	done
}

function initJavaOpt()
{
	JAVA_OPTS="${JAVA_OPTS} -server"
	JAVA_OPTS="${JAVA_OPTS} -Dprogram.name=hinemos_manager"
	JAVA_OPTS="${JAVA_OPTS} -Dhinemos.manager.hostname=${HOST}"
	JAVA_OPTS="${JAVA_OPTS} -Dhinemos.manager.home.dir=${HINEMOS_HOME}"
	JAVA_OPTS="${JAVA_OPTS} -Dhinemos.manager.etc.dir=${HINEMOS_ETC_DIR}"
	JAVA_OPTS="${JAVA_OPTS} -Dhinemos.manager.log.dir=${HINEMOS_LOG_DIR}"
	JAVA_OPTS="${JAVA_OPTS} ${JVM_HEAP_OPTS}"
	JAVA_OPTS="${JAVA_OPTS} -XX:CMSInitiatingOccupancyFraction=75 -XX:+UseCMSInitiatingOccupancyOnly"
	JAVA_OPTS="${JAVA_OPTS} -XX:+UseConcMarkSweepGC -XX:+CMSParallelRemarkEnabled -XX:+UseParNewGC -XX:+DisableExplicitGC"
	JAVA_OPTS="${JAVA_OPTS} -XX:SurvivorRatio=3 -XX:MaxTenuringThreshold=32 -XX:TargetSurvivorRatio=90"
	JAVA_OPTS="${JAVA_OPTS} ${JVM_GCLOG_OPTS}"
	JAVA_OPTS="${JAVA_OPTS} ${JVM_JCONSOLE_OPTS}"
	JAVA_OPTS="${JAVA_OPTS} ${JVM_KEYSTORE_OPTS}"
	JAVA_OPTS="${JAVA_OPTS} ${JVM_OOM_OPTS}"
	
	if [ "x${MAINTENANCE}" = "xtrue" ]
	then
		export STARTUP_MODE="MAINTENANCE"
	else
		export STARTUP_MODE="NORMAL"
	fi
}

function getProperty()
{
	RET=`sed '/^\#/d' ${PROP_FILE} | grep "^${1}"  | tail -n 1 | sed -e 's/^.*=//'`
	echo ${RET}
}


function initScheduler()
{
	THRESHOLD=`getProperty org.quartz.jobStore.misfireThreshold`
	SQL="UPDATE qrtz_triggers SET next_fire_time = (SELECT TRUNC(EXTRACT(EPOCH FROM CURRENT_TIMESTAMP)*1000)) WHERE (SELECT TRUNC(EXTRACT(EPOCH FROM CURRENT_TIMESTAMP)*1000)) - next_fire_time < ${THRESHOLD} AND (SELECT TRUNC(EXTRACT(EPOCH FROM CURRENT_TIMESTAMP)*1000)) - next_fire_time > 0;"
	export PGPASSWORD=`getProperty org.quartz.dataSource.SchedulerDS.password`
	PGUSER=`getProperty org.quartz.dataSource.SchedulerDS.user`
	MSG=`${PG_HOME}/bin/psql -p 24001 -U ${PGUSER} -d hinemos -c "${SQL}" 2>&1`
	RET=$?
	if [ ${RET} -ne 0 ]
	then
		Logging "${MSG}"
		Logging "${MSG_E001}"
		exit ${RET}
	fi
}

function CheckSnmpTrapdStopped() {
	if [ -f /var/run/snmptrapd.pid ]
	then
		Logging "${MSG_E001}"
		exit 1
	fi
}

function startup {
	Logging "${MSG_I002}"
	
	rm -rf ${STARTUP_MONITOR_FILE}
	nohup ${JAVA_HOME}/bin/java ${JAVA_OPTS} -cp ${CLASSPATH} com.clustercontrol.HinemosManagerMain >> ${HINEMOS_LOG_DIR}/jvm_stdout.log 2>&1 &
	echo $! > ${JVM_PID_FILE}
	
	TIME_COUNT=0
	while [ 1 ]
	do
		if [ "x${STARTUP_WAIT}" = "xfalse" ]
		then
				Logging "${MSG_I003} (with -W option)"
				break
		fi
		
		if [ -f ${STARTUP_MONITOR_FILE} ]
		then
			if [ `grep -a 'Hinemos Manager Started' ${STARTUP_MONITOR_FILE} | wc -l` -gt 0 ]
			then
				echo "done"
				Logging "${MSG_I003}"
				break
			fi
		fi
		
		if [ "${TIME_COUNT}" -ge ${STARTUP_CHECK_TIMEOUT} ]
		then
			Logging "${MSG_E002}"
			exit 1
		fi
	
		sleep ${STARTUP_CHECK_INTERVAL}
		TIME_COUNT=$((${TIME_COUNT} + ${STARTUP_CHECK_INTERVAL}))
		echo -n "."
	done
}

function filecheck {
	echo "##############################" > ${HINEMOS_LOG_DIR}/jvm_stdout.log
	echo "HinemosManager Info" >> ${HINEMOS_LOG_DIR}/jvm_stdout.log
	echo "##############################" >> ${HINEMOS_LOG_DIR}/jvm_stdout.log
	echo OutputDate:`date +%Y%m%d%H%M%S` >> ${HINEMOS_LOG_DIR}/jvm_stdout.log
	echo VersionFile:`cat ${HINEMOS_HOME}/_version` >> /${HINEMOS_LOG_DIR}/jvm_stdout.log
	echo SerialFile:`cat ${HINEMOS_HOME}/var/_serial` >> /${HINEMOS_LOG_DIR}/jvm_stdout.log
	echo PackageFile\(MD5, CRC\): >> /${HINEMOS_LOG_DIR}/jvm_stdout.log
	md5sum ${HINEMOS_HOME}/lib/Hinemos*.jar >> ${HINEMOS_LOG_DIR}/jvm_stdout.log
	cksum ${HINEMOS_HOME}/lib/Hinemos*.jar >> ${HINEMOS_LOG_DIR}/jvm_stdout.log
	echo "##############################" >> ${HINEMOS_LOG_DIR}/jvm_stdout.log
	echo "JAVA OPTIONS" >> ${HINEMOS_LOG_DIR}/jvm_stdout.log
	echo "##############################" >> ${HINEMOS_LOG_DIR}/jvm_stdout.log

#	JAR_LIST=`find /opt/hinemos/lib | grep \.jar$`
#	for jar in ${JAR_LIST}
#	do
#		md5sum ${jar} >> ${HINEMOS_LOG_DIR}/jvm_stdout.log
#	done
}

########################################
# SHELL
########################################

# check argument
for OPT in $@
do
	case ${OPT} in
		--help)
			usage
			exit 0
			;;
	esac
done

# option check
FORCE="false"
QUEUE_CLEANUP="false"
STARTUP_WAIT="true"
MAINTENANCE="false"
while getopts cFMW OPT
do
	case ${OPT} in
		c)
			QUEUE_CLEANUP="true"
			;;
		F)
			FORCE="true"
			;;
		M)
			MAINTENANCE="true"
			;;
		W)
			STARTUP_WAIT="false"
			;;
		*)
			ExitIllegalOptionErrorWithoutLogger
			;;
	esac
done

shift $(( $OPTIND - 1 ))

if [ ! $# = 0 ]
then
	ExitIllegalArgumentError
fi

# check user executed
CheckUser ${JVM_USER}

# check process
if [ "x${FORCE}" != "xtrue" ]
then
	CheckPostgreSQLProcessRunning
	CheckSnmpTrapdStopped
	CheckJVMProcessStopped
fi

# check java version
if [ `${JAVA_HOME}/bin/java -version 2>&1 | grep "${JAVA_CHECK_REGEX}" | wc -l` -eq 0 ]
then
	Logging "${MSG_E003}"
	${JAVA_HOME}/bin/java -version 2>&1
	exit -9
fi
if [ `${JAVA_HOME}/bin/java -version 2>&1 | grep "${OPENJDK_CHECK_REGEX}" | wc -l` -eq 0 ]
then
	Logging "${MSG_E003}"
	${JAVA_HOME}/bin/java -version 2>&1
	exit -9
fi

# cleanup pid file
rm -f ${JAVA_PID_FILE}

# cleanup Queue
if [ "x${QUEUE_CLEANUP}" = "xtrue" ]
then
	Logging "${MSG_I001}"
	${HINEMOS_HOME}/sbin/mng/hinemos_clear_queue.sh
fi

# cleanup temporary data
rm -rf ${EXPORT_TMP_DIR}/*

# set parameter
initResourceParameter
initClassPath
initJavaOpt
initScheduler

#startup version and filecheck
filecheck

# startup JVM
startup

exit 0
