/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.accesscontrol.dialog;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Locale;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.List;
import org.eclipse.swt.widgets.Shell;

import com.clustercontrol.accesscontrol.action.CheckPermission;
import com.clustercontrol.accesscontrol.bean.FunctionConstant;
import com.clustercontrol.accesscontrol.bean.PrivilegeConstant.SystemPrivilegeMode;
import com.clustercontrol.accesscontrol.util.AccessEndpointWrapper;
import com.clustercontrol.accesscontrol.util.SystemPrivilegePropertyUtil;
import com.clustercontrol.bean.PropertyDefineConstant;
import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.access.InvalidRole_Exception;
import com.clustercontrol.ws.access.InvalidSetting_Exception;
import com.clustercontrol.ws.access.RoleInfo;
import com.clustercontrol.ws.access.SystemPrivilegeInfo;
import com.clustercontrol.ws.access.UnEditableRole_Exception;
import com.clustercontrol.ws.access.UserDuplicate_Exception;

/**
 * アカウント[システム権限設定]ダイアログクラスです。
 * 
 * @version 4.0.0
 * @since 2.0.0
 */
public class SystemPrivilegeDialog extends CommonDialog {

	// ログ
	private static Log m_log = LogFactory.getLog( SystemPrivilegeDialog.class );

	/** ロールID */
	private String roleId = "";

	/** モード */
	private int mode = 0;

	/** 全システム権限一覧　リストボックス */
	private List listNotRoleSystemPrivilege = null;

	/** 付与システム権限一覧　リストボックス */
	private List listRoleSystemPrivilege = null;

	/** 権限付与　ボタン */
	private Button buttonRoleSystemPrivilege = null;

	/** 権限解除　ボタン */
	private Button buttonNotRoleSystemPrivilege = null;

	/** カラム数 */
	public static final int WIDTH	 = 15;

	/** カラム数（ラベル）。 */
	public static final int WIDTH_LABEL = 4;

	/** カラム数（テキスト）。 */
	public static final int WIDTH_TEXT = 10;

	/** 入力値を保持するオブジェクト。 */
	private RoleInfo inputData = null;

	private boolean permission = false;		// 現在のユーザが変更権限をもつか否か

	/**
	 * コンストラクタ
	 * 
	 * @param parent 親シェル
	 * @param uid ユーザID
	 */
	public SystemPrivilegeDialog(Shell parent, String roleId) {
		super(parent);

		this.roleId = roleId;

		// 現在のユーザがアカウント機能の設定権限をもつか否かを取得する
		permission = new CheckPermission().check(FunctionConstant.ACCESSCONTROL, SystemPrivilegeMode.WRITE);

	}

	/**
	 * ダイアログの初期サイズを返します。
	 * 
	 * @return 初期サイズ
	 * 
	 * @see org.eclipse.jface.window.Window#getInitialSize()
	 */
	@Override
	protected Point getInitialSize() {
		return new Point(600, 600);
	}

	/**
	 * ダイアログエリアを生成します。
	 * 
	 * @param parent 親コンポジット
	 * 
	 * @see com.clustercontrol.dialog.CommonDialog#customizeDialog(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	protected void customizeDialog(Composite parent) {
		Shell shell = this.getShell();

		// タイトル
		shell.setText(Messages
				.getString("dialog.accesscontrol.system.privilege.setting"));

		// プロパティの取得及び設定
		if (permission) {
			this.mode = PropertyDefineConstant.MODE_MODIFY;
		} else {
			this.mode = PropertyDefineConstant.MODE_SHOW;
		}

		// ロール情報の取得
		RoleInfo info = null;
		try {
			info = AccessEndpointWrapper.getRoleInfo(this.roleId);
		} catch (InvalidRole_Exception e) {
			MessageDialog.openInformation(null, Messages.getString("message"),
					Messages.getString("message.accesscontrol.16"));
		} catch (Exception e) {
			m_log.warn("customizeDialog(), " + e.getMessage(), e);
			MessageDialog.openError(
					null,
					Messages.getString("failed"),
					Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());
		}

		// レイアウト
		GridLayout layout = new GridLayout(1, true);
		layout.marginWidth = 10;
		layout.marginHeight = 10;
		layout.numColumns = WIDTH;
		parent.setLayout(layout);

		/*
		 * ロール名
		 */
		// ラベル
		Label label = new Label(parent, SWT.NONE);
		GridData gridData = new GridData();
		gridData.horizontalSpan = WIDTH;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("role.name") + " : " + info.getName());

		/*
		 * 全システム権限一覧用コンポジット
		 */
		Composite compositeNotRole = new Composite(parent, SWT.NONE);
		layout = new GridLayout(1, true);
		layout.numColumns = 1;
		compositeNotRole.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		gridData.horizontalSpan = 6;
		gridData.verticalSpan = 2;
		compositeNotRole.setLayoutData(gridData);

		// 全システム権限一覧 ラベル
		label = new Label(compositeNotRole, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("SystemPrivilegeDialog.not_role_system_privilege"));

		// 全システム権限一覧 リスト
		this.listNotRoleSystemPrivilege = new List(compositeNotRole, SWT.MULTI | SWT.BORDER | SWT.V_SCROLL);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.heightHint = this.listNotRoleSystemPrivilege.getItemHeight() * 12;
		this.listNotRoleSystemPrivilege.setLayoutData(gridData);

		/*
		 * 操作ボタン用コンポジット
		 */
		Composite compositeButton = new Composite(parent, SWT.NONE);
		layout = new GridLayout(1, true);
		layout.numColumns = 1;
		compositeButton.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		gridData.horizontalSpan = 3;
		compositeButton.setLayoutData(gridData);

		// 空
		label = new Label(compositeButton, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

		// 権限付与ボタン
		@SuppressWarnings("unused")
		Label dummy = new Label(compositeButton, SWT.NONE);
		this.buttonRoleSystemPrivilege = this.createButton(compositeButton, Messages.getString("SystemPrivilegeDialog.role_system_privilege_button"));
		this.buttonRoleSystemPrivilege.addSelectionListener(new SelectionAdapter()
		{
			@Override
			public void widgetSelected(SelectionEvent e) {
				String[] items = listNotRoleSystemPrivilege.getSelection();
				for (String item : items) {
					listNotRoleSystemPrivilege.remove(item);
					listRoleSystemPrivilege.add(item);
				}
			}
		});

		// 権限解除ボタン
		dummy = new Label(compositeButton, SWT.NONE);
		this.buttonNotRoleSystemPrivilege = this.createButton(compositeButton, Messages.getString("SystemPrivilegeDialog.not_role_system_privilege_button"));
		this.buttonNotRoleSystemPrivilege.addSelectionListener(new SelectionAdapter()
		{
			@Override
			public void widgetSelected(SelectionEvent e) {
				String[] items = listRoleSystemPrivilege.getSelection();
				for (String item : items) {
					listRoleSystemPrivilege.remove(item);
					listNotRoleSystemPrivilege.add(item);
				}
			}
		});

		// 空
		label = new Label(compositeButton, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

		/*
		 * 付与システム権限一覧用コンポジット
		 */
		Composite compositeRole = new Composite(parent, SWT.NONE);
		layout = new GridLayout(1, true);
		layout.numColumns = 1;
		compositeRole.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		gridData.horizontalSpan = 6;
		gridData.verticalSpan = 2;
		compositeRole.setLayoutData(gridData);

		// 付与システム権限一覧 ラベル
		label = new Label(compositeRole, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("SystemPrivilegeDialog.role_system_privilege"));

		// 付与システム権限一覧 リスト
		this.listRoleSystemPrivilege = new List(compositeRole, SWT.MULTI | SWT.BORDER | SWT.V_SCROLL);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.heightHint = this.listRoleSystemPrivilege.getItemHeight() * 12;
		this.listRoleSystemPrivilege.setLayoutData(gridData);

		// サイズを最適化
		// グリッドレイアウトを用いた場合、こうしないと横幅が画面いっぱいになります。
		shell.pack();
		shell.setSize(new Point(550, shell.getSize().y));

		// 画面中央に
		Display display = shell.getDisplay();
		shell.setLocation((display.getBounds().width - shell.getSize().x) / 2,
				(display.getBounds().height - shell.getSize().y) / 2);

		this.setInputData(info);
	}

	/**
	 * 入力値チェックをします。
	 * 
	 * @return 検証結果
	 * 
	 * @see com.clustercontrol.dialog.CommonDialog#validate()
	 */
	@Override
	protected ValidateResult validate() {
		// 入力値生成
		this.inputData = this.createInputData();

		if (this.inputData != null) {
			return super.validate();
		} else {
			return null;
		}
	}

	/**
	 * 入力値をマネージャに登録します。
	 * 
	 * @return true：正常、false：異常
	 * 
	 * @see com.clustercontrol.dialog.CommonDialog#action()
	 */
	@Override
	protected boolean action() {
		boolean result = false;

		RoleInfo roleInfo = this.inputData;
		if(roleInfo == null){
			return result;
		}

		try {
			AccessEndpointWrapper.replaceSystemPrivilegeRole(this.roleId, roleInfo.getSystemPrivilegeList());
			result = true;

			// 完了メッセージ
			MessageDialog.openInformation(
					null,
					Messages.getString("successful"),
					Messages.getString("message.accesscontrol.47"));

		} catch (UserDuplicate_Exception e) {
			//ロールID取得
			String args[] = { roleInfo.getId() };

			// ロールIDが重複している場合、エラーダイアログを表示する
			MessageDialog.openInformation(
					null,
					Messages.getString("message"),
					Messages.getString("message.accesscontrol.48", args));

		} catch (InvalidSetting_Exception e) {
			// 「リポジトリ - 参照」が登録するシステム権限に含まれていない
			String args[] = { 
					Messages.getString("repository.read", Locale.getDefault())
			};
			
			MessageDialog.openInformation(
					null, 
					Messages.getString("message"),
					Messages.getString("message.accesscontrol.51", args));
			
			
		} catch (Exception e) {
			String errMessage = "";
			if (e instanceof InvalidRole_Exception) {
				// 権限なし
				MessageDialog.openInformation(null, Messages.getString("message"),
						Messages.getString("message.accesscontrol.16"));
			} else if (e instanceof UnEditableRole_Exception) {
				// システム権限の変更なロールの場合はエラー（ADMINISTRATORS）
				MessageDialog.openInformation(null, Messages.getString("message"),
						Messages.getString("message.accesscontrol.44"));
			} else {
				errMessage = ", " + e.getMessage();
			}
			MessageDialog.openError(
					null,
					Messages.getString("failed"),
					Messages.getString("message.accesscontrol.48") + errMessage);

		}
	

		return result;
	}

	/**
	 * ロール用プロパティを設定します。
	 * 
	 * @param roleInfo 設定値として用いるロール情報
	 */
	protected void setInputData(RoleInfo roleInfo) {

		this.inputData = roleInfo;

		// 各項目に反映

		java.util.List<String> allSystemPrivilegeList = null;
		java.util.List<SystemPrivilegeInfo> roleSystemPrivilegeKeyList = null;
		// 全システム権限を取得
		allSystemPrivilegeList = SystemPrivilegePropertyUtil.getValueList();
		//TODO 保留！！！この修正では最初の時点でのソートしかできない。以降はSWTのオブジェクトを直接いじっているように見える
		java.util.Collections.sort(allSystemPrivilegeList);
		// 付与システム権限を取得
		try {
			roleSystemPrivilegeKeyList = AccessEndpointWrapper.getSystemPrivilegeInfoListByRoleId(roleId);
		} catch (InvalidRole_Exception e) {
			// 権限なし
			MessageDialog.openInformation(null, Messages.getString("message"),
					Messages.getString("message.accesscontrol.16"));

		} catch (Exception e) {
			// 上記以外の例外
			m_log.warn("getOwnUserList(), " + e.getMessage(), e);
			MessageDialog.openError(
					null,
					Messages.getString("failed"),
					Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());
		}
		java.util.List<String> roleSystemPrivilegeValueList
		= getSystemPrivilegeValueList(roleSystemPrivilegeKeyList);
		// リストを振り分ける
		for (String SystemPrivilege : allSystemPrivilegeList) {
			if (roleSystemPrivilegeValueList.contains(SystemPrivilege)) {
				this.listRoleSystemPrivilege.add(SystemPrivilege);
			} else {
				this.listNotRoleSystemPrivilege.add(SystemPrivilege);
			}
		}
	}

	/**
	 * 入力値を設定したロール情報を返します。<BR>
	 * 入力値チェックを行い、不正な場合は<code>null</code>を返します。
	 * 
	 * @return ロール情報
	 */
	private RoleInfo createInputData() {
		RoleInfo info = this.inputData;

		// 付与システム権限一覧からシステム権限情報を取得する
		java.util.List<SystemPrivilegeInfo> systemPrivilegeList = this.inputData.getSystemPrivilegeList();
		if (systemPrivilegeList == null) {
			systemPrivilegeList = new ArrayList<SystemPrivilegeInfo>();
		}

		java.util.List<SystemPrivilegeInfo> roleSystemPrivilegeList = info.getSystemPrivilegeList();
		roleSystemPrivilegeList.clear();
		if (this.listRoleSystemPrivilege.getItemCount() > 0) {
			roleSystemPrivilegeList.addAll(getSystemPrivilegeKeyList(Arrays.asList(this.listRoleSystemPrivilege.getItems())));
		}

		return info;
	}

	/**
	 * ＯＫボタンのテキストを返します。
	 * 
	 * @return ＯＫボタンのテキスト
	 */
	@Override
	protected String getOkButtonText() {
		return Messages.getString("setup");
	}

	/**
	 * キャンセルボタンのテキストを返します。
	 * 
	 * @return キャンセルボタンのテキスト
	 */
	@Override
	protected String getCancelButtonText() {
		return Messages.getString("cancel"); //$NON-NLS-1$
	}

	/**
	 * 共通のボタンを生成します。
	 * 
	 * @param parent
	 *            親のコンポジット
	 * @param label
	 *            ボタンのラベル
	 * @return 生成されたボタン
	 */
	private Button createButton(Composite parent, String label) {
		Button button = new Button(parent, SWT.NONE);

		GridData gridData = new GridData(GridData.FILL_HORIZONTAL);
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		button.setLayoutData(gridData);

		button.setText(label);

		return button;
	}

	/**
	 * 表示名称リストからシステム権限リストを返す
	 * 
	 * @param beforeList 表示名称リスト
	 * @return システム権限リスト
	 */
	private java.util.List<SystemPrivilegeInfo> getSystemPrivilegeKeyList(java.util.List<String> beforeList) {
		java.util.List<SystemPrivilegeInfo> afterList = new ArrayList<SystemPrivilegeInfo>();
		for (String beforeStr : beforeList) {
			String key = SystemPrivilegePropertyUtil.getKey(beforeStr);
			// 画面変更前の仮対応
			SystemPrivilegeInfo info = new SystemPrivilegeInfo();
			String regexp = "(.+?)(" + SystemPrivilegeMode.READ.name() + ")";
			Pattern pattern = Pattern.compile(regexp);
			Matcher matcher = pattern.matcher(key);
			if (matcher.find()) {
				info.setSystemFunction(matcher.group(1));
				info.setSystemPrivilege(SystemPrivilegeMode.READ.name());
				afterList.add(info);
				continue;
			}
			regexp = "(.+?)(" + SystemPrivilegeMode.WRITE.name() + ")";
			pattern = Pattern.compile(regexp);
			matcher = pattern.matcher(key);
			if (matcher.find()) {
				info.setSystemFunction(matcher.group(1));
				info.setSystemPrivilege(SystemPrivilegeMode.WRITE.name());
				afterList.add(info);
				continue;
			}
			regexp = "(.+?)(" + SystemPrivilegeMode.ADD.name() + ")";
			pattern = Pattern.compile(regexp);
			matcher = pattern.matcher(key);
			if (matcher.find()) {
				info.setSystemFunction(matcher.group(1));
				info.setSystemPrivilege(SystemPrivilegeMode.ADD.name());
				afterList.add(info);
				continue;
			}
			regexp = "(.+?)(" + SystemPrivilegeMode.EXEC.name() + ")";
			pattern = Pattern.compile(regexp);
			matcher = pattern.matcher(key);
			if (matcher.find()) {
				info.setSystemFunction(matcher.group(1));
				info.setSystemPrivilege(SystemPrivilegeMode.EXEC.name());
				afterList.add(info);
				continue;
			}
		}
		return afterList;
	}

	/**
	 * システム権限リストから表示名称リストを返す
	 * 
	 * @param beforeList システム権限リスト
	 * @return 表示名称リスト
	 */
	private java.util.List<String> getSystemPrivilegeValueList(java.util.List<SystemPrivilegeInfo> beforeList) {
		java.util.List<String> afterList = new ArrayList<String>();
		for (SystemPrivilegeInfo systemPrivilegeInfo : beforeList) {
			afterList.add(SystemPrivilegePropertyUtil.getValue(systemPrivilegeInfo.getSystemFunction(), systemPrivilegeInfo.getSystemPrivilege()));
		}
		return afterList;
	}
}