/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.jobmanagement.view.action;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.ui.IViewActionDelegate;
import org.eclipse.ui.IViewPart;

import com.clustercontrol.jobmanagement.composite.JobListComposite;
import com.clustercontrol.jobmanagement.composite.JobTreeComposite;
import com.clustercontrol.jobmanagement.util.JobEditStateUtil;
import com.clustercontrol.jobmanagement.util.JobEndpointWrapper;
import com.clustercontrol.jobmanagement.util.JobPropertyUtil;
import com.clustercontrol.jobmanagement.util.JobTreeItemUtil;
import com.clustercontrol.jobmanagement.util.JobUtil;
import com.clustercontrol.jobmanagement.view.JobListView;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.jobmanagement.InvalidRole_Exception;
import com.clustercontrol.ws.jobmanagement.JobTreeItem;
import com.clustercontrol.ws.jobmanagement.OtherUserGetLock_Exception;

/**
 * ジョブ[一覧]ビューの「編集モード」のクライアント側アクションクラス<BR>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class EditModeAction implements IViewActionDelegate {
	// ログ
	private static Log m_log = LogFactory.getLog( EditModeAction.class );

	/** アクションID */
	public static final String ID = "com.clustercontrol.jobmanagement.view.action.EditModeAction";
	/** ビュー */
	private IViewPart m_view;

	/**
	 * ビューを保持します。ワークベンチにロードされた際に呼ばれます。
	 * 
	 * @param view ビューのインスタンス
	 * 
	 * @see org.eclipse.ui.IViewActionDelegate#init(org.eclipse.ui.IViewPart)
	 */
	@Override
	public void init(IViewPart view) {
		m_view = view;
	}

	/**
	 * ジョブ[一覧]ビューの「編集モード」が押された場合に、編集ロックを取得します。
	 * 
	 * @see org.eclipse.ui.IActionDelegate#run(org.eclipse.jface.action.IAction)
	 * @see com.clustercontrol.jobmanagement.view.JobPlanListView
	 */
	@Override
	public void run(IAction action) {
		JobTreeItem selectedItem = null; // 選択されたジョブツリーアイテム
		JobTreeItem item = null; //編集モードにするジョブユニット
		JobTreeItem parent = null;
		boolean isSelected = false; // ジョブが正しく選択されていないときに警告を出すためのフラグ

		if (m_view instanceof JobListView) {
			JobListView view = (JobListView) m_view
					.getAdapter(JobListView.class);
			JobTreeComposite tree = view.getJobTreeComposite();
			JobListComposite list = view.getJobListComposite();
			if (tree.getTree().isFocusControl()) {
				selectedItem = tree.getSelectItem();
				item = JobUtil.getTopJobUnitTreeItem(selectedItem);
				parent = item.getParent();
			} else if (list.getTable().isFocusControl()) {
				// selectedItemはツリーで選択されているアイテム(テーブルで選択しているジョブの親)にする
				selectedItem = list.getSelectItem().getParent();
				item = JobUtil.getTopJobUnitTreeItem(list.getSelectItem());
				if (item instanceof JobTreeItem) {
					parent = item.getParent();
				}
			} else {
				// 編集モードボタンをOFFにする
				action.setChecked(false);
			}

			if (parent instanceof JobTreeItem && item instanceof JobTreeItem) {
				isSelected = true;
				String jobunitId = item.getData().getJobunitId();

				if (action.isChecked()) {
					if (JobEditStateUtil.getLockedJobunitList().contains(item.getData())) {
						// ここにくることはないはず(編集モードにいるのに編集モードに入るアクション)
						return;
					}
					// 編集モードに入る
					Long updateTime = JobEditStateUtil.getJobunitUpdateTime(jobunitId);
					Integer result = null;
					try {
						result =JobUtil.getEditLock(jobunitId, updateTime, false);
					} catch (OtherUserGetLock_Exception e) {
						// 他のユーザがロックを取得している
						String message = e.getMessage();
						if (MessageDialog.openQuestion(
								null,
								Messages.getString("confirmed"),
								message)) {
							try {
								result = JobUtil.getEditLock(jobunitId, updateTime, true);
							} catch (Exception e1) {
								// ここには絶対にこないはず
								m_log.error("run() : logical error");
							}
						}
					}

					if (result != null) {
						// ロックを取得した
						m_log.debug("run() : get editLock(jobunitId="+jobunitId+")");
						JobEditStateUtil.addLockedJobunit(item.getData(), JobTreeItemUtil.clone(item, null), result);
						view.getJobTreeComposite().refresh(parent);
						tree.getTreeViewer().setSelection(
								new StructuredSelection(selectedItem), true);
					} else {
						// ロックの取得に失敗した
						m_log.debug("run() : cannot get editLock(jobunitId="+jobunitId+")");
						action.setChecked(false);
					}
				} else {
					// 編集モードから抜ける
					if (!JobEditStateUtil.getLockedJobunitList().contains(item.getData())) {
						// ここにくることはないはず(編集モードにいないのに編集モードから抜けるアクション)
						return;
					}
					try {
						if (MessageDialog.openQuestion(
								null,
								Messages.getString("confirmed"),
								Messages.getString("message.job.103"))) {
							// 編集ロックの開放
							JobEndpointWrapper.releaseEditLock(JobEditStateUtil.getEditSession(item.getData()));

							//バックアップに切り戻す
							JobTreeItem backup = JobEditStateUtil.getLockedJobunitBackup(item.getData());
							JobTreeItemUtil.removeChildren(parent, item);
							if (backup != null) {
								JobPropertyUtil.setJobFullTree(backup);
								JobTreeItemUtil.addChildren(parent, backup);
							}

							JobEditStateUtil.exitEditMode(item);
							view.getJobTreeComposite().getTreeViewer().sort(parent);
							view.getJobTreeComposite().refresh();
						} else {
							//編集ロックを開放しない
							action.setChecked(true);
						}
					} catch (InvalidRole_Exception e) {
						// アクセス権なしの場合、エラーダイアログを表示する
						MessageDialog.openInformation(
								null,
								Messages.getString("message"),
								Messages.getString("message.accesscontrol.16"));
					} catch (Exception e) {
						m_log.warn("updateJobunitUpdateTime() : " + e.getMessage(), e);
						MessageDialog.openError(
								null,
								Messages.getString("failed"),
								Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());
					}
				}
			}
		}
		if (!isSelected) {
			// ジョブツリーアイテムが選択されていないときには、メッセージを表示する
			MessageDialog.openInformation(null, Messages.getString("message"),
					Messages.getString("message.job.110"));
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.IActionDelegate#selectionChanged(org.eclipse.jface.action.IAction, org.eclipse.jface.viewers.ISelection)
	 */
	@Override
	public void selectionChanged(IAction action, ISelection selection) {

	}
}