/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.monitor.run.dialog;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;

import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.RunIntervalConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.composite.RoleIdListComposite;
import com.clustercontrol.composite.RoleIdListComposite.Mode;
import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.monitor.run.bean.MonitorTypeConstant;
import com.clustercontrol.monitor.run.composite.MonitorBasicScopeComposite;
import com.clustercontrol.monitor.run.composite.MonitorRuleComposite;
import com.clustercontrol.notify.composite.NotifyInfoComposite;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.monitor.MonitorInfo;

/**
 * 監視設定共通ダイアログクラス
 * 
 */
public class CommonMonitorDialog extends CommonDialog {

	public static final int WIDTH_TITLE = 4;
	public static final int WIDTH_VALUE = 2;
	public static final int WIDTH_TEXT = 8;

	public static final int TIMEOUT_SEC = 5000;

	// ----- instance フィールド ----- //

	/** 入力値を保持するオブジェクト */
	protected MonitorInfo inputData = null;

	/** 入力値の正当性を保持するオブジェクト */
	protected ValidateResult validateResult = null;

	/** 変更対象の監視項目ID */
	protected String monitorId = null;

	/** 変更するかどうかのフラグ（true：変更する） modify、copy時に使用　 */
	protected boolean updateFlg = false;

	/** 監視基本情報 */
	protected MonitorBasicScopeComposite m_monitorBasic = null;

	/** 監視条件 共通部分 */
	protected MonitorRuleComposite m_monitorRule = null;

	/** 通知情報 */
	protected NotifyInfoComposite m_notifyInfo = null;

	/** 監視を有効にする */
	protected Button confirmMonitorValid = null;

	/** 監視判定タイプ  */
	protected int m_monitorType;

	/** 入力値から生成する監視情報 **/
	protected MonitorInfo monitorInfo = null;

	/** 未登録ノード スコープを表示するかフラグ*/
	protected boolean m_unregistered = false;

	// ----- 共通メンバ変数 ----- //
	protected Shell shell = null;
	protected Group groupRule = null;				// 条件グループ
	protected Group groupMonitor = null;			// 監視グループ
	protected Group groupDetermine = null;			// 判定グループ
	protected Group groupNotifyAttribute = null;	// 通知グループ

	// ----- コンストラクタ ----- //

	/**
	 * 作成用ダイアログのインスタンスを返します。
	 * 
	 * @param parent
	 *            親のシェルオブジェクト
	 */
	public CommonMonitorDialog(Shell parent) {
		super(parent);
	}

	// ----- instance メソッド ----- //

	/**
	 * ダイアログエリアを生成します。
	 * 
	 * @param parent 親のインスタンス
	 */
	@Override
	protected void customizeDialog(Composite parent) {
		shell = this.getShell();

		// 変数として利用されるラベル
		Label label = null;
		// 変数として利用されるグリッドデータ
		GridData gridData = null;

		// レイアウト
		GridLayout layout = new GridLayout(1, true);
		layout.marginWidth = 10;
		layout.marginHeight = 10;
		layout.numColumns = 15;
		parent.setLayout(layout);

		// 監視基本情報
		m_monitorBasic = new MonitorBasicScopeComposite(parent, SWT.NONE, m_unregistered, this);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		m_monitorBasic.setLayoutData(gridData);

		/*
		 * 条件グループ
		 */
		groupRule = new Group(parent, SWT.NONE);
		layout = new GridLayout(1, true);
		layout.marginWidth = 5;
		layout.marginHeight = 5;
		layout.numColumns = 15;
		groupRule.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		groupRule.setLayoutData(gridData);
		groupRule.setText(Messages.getString("monitor.rule"));

		m_monitorRule = new MonitorRuleComposite(groupRule, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		m_monitorRule.setLayoutData(gridData);

		/*
		 * 監視グループ
		 */
		groupMonitor = new Group(parent, SWT.NONE);
		layout = new GridLayout(1, true);
		layout.marginWidth = 5;
		layout.marginHeight = 5;
		layout.numColumns = 15;
		groupMonitor.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		groupMonitor.setLayoutData(gridData);
		groupMonitor.setText(Messages.getString("monitor.run"));

		// 監視（有効／無効）
		this.confirmMonitorValid = new Button(groupMonitor, SWT.CHECK);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.confirmMonitorValid.setLayoutData(gridData);
		this.confirmMonitorValid.setText(Messages.getString("monitor.run"));
		this.confirmMonitorValid.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				// 判定、通知部分を有効/無効化
				if(confirmMonitorValid.getSelection()){
					setMonitorEnabled(true);
				}else{
					setMonitorEnabled(false);
				}
			}
		});

		/*
		 * 判定グループ（監視グループの子グループ）
		 * なお、判定内容は継承先のクラスにて実装する。
		 */
		groupDetermine = new Group(groupMonitor, SWT.NONE);
		layout = new GridLayout(1, true);
		layout.marginWidth = 5;
		layout.marginHeight = 5;
		layout.numColumns = 1;
		groupDetermine.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		groupDetermine.setLayoutData(gridData);
		groupDetermine.setText(Messages.getString("determine"));

		/*
		 * 通知グループ（監視グループの子グループ）
		 */
		groupNotifyAttribute = new Group(groupMonitor, SWT.NONE);
		layout = new GridLayout(1, true);
		layout.marginWidth = 5;
		layout.marginHeight = 5;
		layout.numColumns = 1;
		groupNotifyAttribute.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		groupNotifyAttribute.setLayoutData(gridData);
		groupNotifyAttribute.setText(Messages.getString("notify.attribute"));
		this.m_notifyInfo = new NotifyInfoComposite(groupNotifyAttribute, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 1;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_notifyInfo.setLayoutData(gridData);

		// ラインを引く
		Label line = new Label(parent, SWT.SEPARATOR | SWT.HORIZONTAL);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 15;
		line.setLayoutData(gridData);

	}

	/**
	 * 監視エリアを有効/無効化します。
	 * 
	 */
	protected void setMonitorEnabled(boolean enabled){
		m_notifyInfo.setEnabled(enabled);
	}


	/**
	 * ダイアログエリアを調整します。
	 * 
	 */
	protected void adjustDialog(){
		// サイズを最適化
		// グリッドレイアウトを用いた場合、こうしないと横幅が画面いっぱいになります。
		shell.pack();
		shell.setSize(new Point(600, shell.getSize().y));

		// 画面中央に配置
		Display display = shell.getDisplay();
		shell.setLocation((display.getBounds().width - shell.getSize().x) / 2,
				(display.getBounds().height - shell.getSize().y) / 2);
	}


	/**
	 * 入力内容を返します。
	 * 
	 * @return 入力内容を保持した通知情報
	 */
	public MonitorInfo getInputData() {
		return this.inputData;
	}
	
	public boolean getUpdateFlg(){
		return this.updateFlg;
	}

	public MonitorRuleComposite getMonitorRule(){
		return this.m_monitorRule;
	}
	
	public MonitorBasicScopeComposite getMonitorBasicScope(){
		return this.m_monitorBasic;
	}
	
	public NotifyInfoComposite getNotifyInfo(){
		return this.m_notifyInfo;
	}
	
	/**
	 * 各項目に入力値を設定します。
	 * 
	 * @param monitor 設定値として用いる監視情報
	 */
	protected void setInputData(MonitorInfo monitor) {

		// 監視基本情報
		m_monitorBasic.setInputData(monitor, this.updateFlg);

		// 監視条件
		m_monitorRule.setInputData(monitor);

		//通知情報の設定
		if(monitor.getNotifyId() != null){
			this.m_notifyInfo.setNotify(monitor.getNotifyId());
		}

		if (monitor.getApplication() != null) {
			this.m_notifyInfo.setApplication(monitor.getApplication());
			this.m_notifyInfo.update();
		}

		// 監視
		if (monitor.getMonitorFlg() == ValidConstant.TYPE_VALID) {
			this.confirmMonitorValid.setSelection(true);
		}else{
			this.setMonitorEnabled(false);
		}


	}

	/**
	 * 入力値を用いて通知情報を生成します。
	 * 
	 * @return 入力値を保持した通知情報
	 */
	protected MonitorInfo createInputData() {
		monitorInfo = new MonitorInfo();
		setInfoInitialValue(monitorInfo);

		// 監視基本情報
		validateResult = m_monitorBasic.createInputData(monitorInfo);
		if(validateResult != null){
			return null;
		}

		// 監視条件 共通部分
		validateResult = m_monitorRule.createInputData(monitorInfo);
		if(validateResult != null){
			return null;
		}

		// 監視
		if (this.confirmMonitorValid.getSelection()) {
			monitorInfo.setMonitorFlg(ValidConstant.TYPE_VALID);
		} else {
			monitorInfo.setMonitorFlg(ValidConstant.TYPE_INVALID);
		}

		return monitorInfo;
	}

	/**
	 * 入力値チェックをします。
	 * 
	 * @return 検証結果
	 * 
	 * @see com.clustercontrol.dialog.CommonDialog#validate()
	 */
	@Override
	protected ValidateResult validate() {
		// 入力値生成
		this.inputData = this.createInputData();

		if (this.inputData != null) {
			return super.validate();
		} else {
			return validateResult;
		}
	}


	/**
	 * ＯＫボタンのテキストを返します。
	 * 
	 * @return ＯＫボタンのテキスト
	 */
	@Override
	protected String getOkButtonText() {
		return Messages.getString("ok");
	}

	/**
	 * キャンセルボタンのテキストを返します。
	 * 
	 * @return キャンセルボタンのテキスト
	 */
	@Override
	protected String getCancelButtonText() {
		return Messages.getString("cancel");
	}

	/**
	 * 無効な入力値の情報を設定します
	 * 
	 */
	protected void setValidateResult(String id, String message) {

		this.validateResult = new ValidateResult();
		this.validateResult.setValid(false);
		this.validateResult.setID(id);
		this.validateResult.setMessage(message);
	}

	/**
	 * MonitorInfoに初期値を設定します
	 * 
	 */
	protected void setInfoInitialValue(MonitorInfo monitor) {

		// 監視判定タイプ(真偽値/数値/文字列)
		monitor.setMonitorType(MonitorTypeConstant.TYPE_TRUTH);
		// 実行間隔（秒）
		monitor.setRunInterval(RunIntervalConstant.TYPE_MIN_05);
		// 値失敗時の重要度
		monitor.setFailurePriority(PriorityConstant.TYPE_UNKNOWN);
		// 監視有効フラグ
		monitor.setMonitorFlg(ValidConstant.TYPE_VALID);
		// 収集有効フラグ
		monitor.setCollectorFlg(ValidConstant.TYPE_INVALID);
	}

	/**
	 * 更新処理
	 */
	protected void update(){
	}
}