/*

 Copyright (C) 2006 NTT DATA Corporation

 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.

 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.http.util;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.Header;
import org.apache.http.HttpHost;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.conn.params.ConnRoutePNames;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.params.CoreConnectionPNames;

import com.clustercontrol.util.Messages;

/**
 * HTTPアクセスを実施するクライアントクラス<BR>
 *
 * @version 2.1.0
 * @since 2.1.0
 */
public class GetHttpResponse {

	private static Log m_log = LogFactory.getLog( GetHttpResponse.class );

	/** ボディ取得対象のContent-Type */
	private static final String TARGET_CONTENT_TYPE = "text";

	/** プロキシサーバ　ホスト名 */
	private String m_proxyHost;

	/** プロキシサーバ ポート */
	private int m_proxyPort;

	/** ステータスコード */
	private int m_statusCode;

	/** ヘッダ */
	private StringBuffer m_header;

	/** ボディ */
	private String m_responseBody;

	/** 応答時間（ミリ秒） */
	private long m_responseTime;

	/** エラーメッセージ */
	private String m_errorMessage;


	/**
	 * コンストラクタ
	 */
	public GetHttpResponse() {
		super();
	}

	/**
	 * リクエストURLから情報取得
	 * 
	 * @param url URL
	 * @param timeout タイムアウト（ミリ秒）
	 * @return
	 */
	public boolean execute(String url, int timeout) {

		boolean result = false;

		// Create an instance of HttpClient.
		HttpClient client = new DefaultHttpClient();
		try {

			// proxyの設定
			if(m_proxyHost != null && !"".equals(m_proxyHost.trim())){
				client.getParams().setParameter(ConnRoutePNames.DEFAULT_PROXY, new HttpHost(m_proxyHost, m_proxyPort));
			}

			// Create a method instance.
			HttpGet method = new HttpGet(url);

			// タイムアウト設定
			if(timeout > 0){
				// establish a connection within n milliseconds
				method.getParams().setIntParameter(CoreConnectionPNames.CONNECTION_TIMEOUT, timeout);

				// socket timeout in milliseconds
				method.getParams().setIntParameter(CoreConnectionPNames.SO_TIMEOUT, timeout);
			}

			// Execute the method.
			long start = System.currentTimeMillis();
			HttpResponse response = client.execute(method);
			m_statusCode = response.getStatusLine().getStatusCode();
			long end = System.currentTimeMillis();
			m_responseTime = end - start;

			// HttpResponseでタイムアウトしていないため、見た目上のタイムアウトを修正
			if(m_responseTime > timeout){
				m_log.debug("execute() url = " + url + ". m_responseTime is replace . from " + m_responseTime + " to " + timeout);
				m_responseTime = timeout;
			}

			// Header
			Header[] headers = response.getAllHeaders();
			if (headers != null && headers.length>0) {
				m_header = new StringBuffer();
				for (int i = 0; i < headers.length; i++) {
					m_header.append((i != 0 ? "\n" : "") + headers[i]);
				}
			}

			if (m_statusCode == HttpStatus.SC_OK) {
				// Content-Typeがtext文書の場合のみ、Bodyを取得
				Header header = response.getFirstHeader("Content-Type");
				if(header != null && header.getValue() != null && header.getValue().indexOf(TARGET_CONTENT_TYPE) != -1){

					InputStreamReader in = new InputStreamReader(response.getEntity().getContent(), "JISAutoDetect");
					StringBuffer stb = new StringBuffer();

					int c = 0;
					while((c = in.read()) != -1){
						stb.append((char)c);
					}

					m_responseBody = stb.toString();
					result = true;

				}
				else{
					m_errorMessage = Messages.getString("message.http.2");
				}
			}
			else{
				m_errorMessage = response.getStatusLine().toString();
			}

		} catch (UnsupportedEncodingException e) {
			m_log.info("execute(): " + e.getMessage() + " class=" + e.getClass().getName());
			m_errorMessage = "http receiving failure. (unsupported encoding)";
		} catch (IOException e) {
			m_log.info("execute(): Fatal transport error. " + e.getMessage() + " class=" + e.getClass().getName());
			m_errorMessage = "http requesting failure. (I/O error : unreachable or timeout)";
		} catch (Exception e) {
			m_log.info("execute(): " + e.getMessage() + " class=" + e.getClass().getName());
			m_errorMessage = "http requesting failure. " + e.getMessage() + "(" + e.getClass().getSimpleName() + ")";
		} finally {
			// Release the connection.
			client.getConnectionManager().closeExpiredConnections();
		}
		return result;
	}

	/**
	 * @return m_header を戻します。
	 */
	public String getHeader() {

		if(m_header != null){
			return m_header.toString();
		}
		return null;
	}

	/**
	 * @return m_responseBody を戻します。
	 */
	public String getResponseBody() {
		return m_responseBody;
	}

	/**
	 * @return m_responseTime を戻します。
	 */
	public long getResponseTime() {
		return m_responseTime;
	}

	/**
	 * @return m_statusCode を戻します。
	 */
	public int getStatusCode() {
		return m_statusCode;
	}

	/**
	 * @return m_errorMessage を戻します。
	 */
	public String getErrorMessage() {
		return m_errorMessage;
	}
}
