/*

Copyright (C) since 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.repository.factory;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import javax.persistence.EntityExistsException;
import javax.persistence.EntityManager;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.accesscontrol.bean.PrivilegeConstant.ObjectPrivilegeMode;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.commons.util.Ipv6Util;
import com.clustercontrol.commons.util.JpaTransactionManager;
import com.clustercontrol.commons.util.ObjectValidator;
import com.clustercontrol.fault.FacilityNotFound;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.UsedFacility;
import com.clustercontrol.repository.bean.FacilityConstant;
import com.clustercontrol.repository.bean.FacilityInfo;
import com.clustercontrol.repository.bean.FacilityTreeAttributeConstant;
import com.clustercontrol.repository.bean.NodeCpuInfo;
import com.clustercontrol.repository.bean.NodeDeviceInfo;
import com.clustercontrol.repository.bean.NodeDiskInfo;
import com.clustercontrol.repository.bean.NodeFilesystemInfo;
import com.clustercontrol.repository.bean.NodeHostnameInfo;
import com.clustercontrol.repository.bean.NodeInfo;
import com.clustercontrol.repository.bean.NodeMemoryInfo;
import com.clustercontrol.repository.bean.NodeNetworkInterfaceInfo;
import com.clustercontrol.repository.bean.NodeNoteInfo;
import com.clustercontrol.repository.bean.NodeVariableInfo;
import com.clustercontrol.repository.bean.ScopeInfo;
import com.clustercontrol.repository.entity.CollectorPlatformMstData;
import com.clustercontrol.repository.entity.CollectorSubPlatformMstData;
import com.clustercontrol.repository.model.CollectorPlatformMstEntity;
import com.clustercontrol.repository.model.CollectorSubPlatformMstEntity;
import com.clustercontrol.repository.model.FacilityEntity;
import com.clustercontrol.repository.model.FacilityRelationEntity;
import com.clustercontrol.repository.model.NodeCpuEntity;
import com.clustercontrol.repository.model.NodeCpuEntityPK;
import com.clustercontrol.repository.model.NodeDeviceEntity;
import com.clustercontrol.repository.model.NodeDeviceEntityPK;
import com.clustercontrol.repository.model.NodeDiskEntity;
import com.clustercontrol.repository.model.NodeDiskEntityPK;
import com.clustercontrol.repository.model.NodeEntity;
import com.clustercontrol.repository.model.NodeFilesystemEntity;
import com.clustercontrol.repository.model.NodeFilesystemEntityPK;
import com.clustercontrol.repository.model.NodeHostnameEntity;
import com.clustercontrol.repository.model.NodeHostnameEntityPK;
import com.clustercontrol.repository.model.NodeMemoryEntity;
import com.clustercontrol.repository.model.NodeMemoryEntityPK;
import com.clustercontrol.repository.model.NodeNetworkInterfaceEntity;
import com.clustercontrol.repository.model.NodeNetworkInterfaceEntityPK;
import com.clustercontrol.repository.model.NodeNoteEntity;
import com.clustercontrol.repository.model.NodeNoteEntityPK;
import com.clustercontrol.repository.model.NodeVariableEntity;
import com.clustercontrol.repository.model.NodeVariableEntityPK;
import com.clustercontrol.repository.util.FacilityUtil;
import com.clustercontrol.repository.util.QueryUtil;

/**
 * ファシリティの更新処理を実装したクラス<BR>
 */
public class FacilityModifier {

	private static Log m_log = LogFactory.getLog(FacilityModifier.class);

	/** プラットフォーム定義情報を登録する。<BR>
	 * 
	 * @param CollectorPlatformMstData プラットフォーム定義情報
	 * @throws CollectorPlatformMstData
	 * @throws EntityExistsException
	 */
	public static void addCollectorPratformMst(CollectorPlatformMstData data) throws EntityExistsException {
		JpaTransactionManager jtm = new JpaTransactionManager();

		try {
			// インスタンス生成
			CollectorPlatformMstEntity entity = new CollectorPlatformMstEntity(data.getPlatformId());
			// 重複チェック
			jtm.checkEntityExists(CollectorPlatformMstEntity.class, entity.getPlatformId());
			entity.setOrderNo(data.getOrderNo().intValue());
			entity.setPlatformName(data.getPlatformName());
		} catch (EntityExistsException e) {
			m_log.info("addCollectorPratformMst() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage());
			throw e;
		}
	}

	/**
	 * 指定されたプラットフォームIDのプラットフォーム定義情報を削除する。<BR>
	 * 
	 * @param platformId プラットフォームID
	 * @throws FacilityNotFound
	 */
	public static void deleteCollectorPratformMst(String platformId) throws FacilityNotFound {
		EntityManager em = new JpaTransactionManager().getEntityManager();

		// 該当するプラットフォーム定義情報を取得
		CollectorPlatformMstEntity entity = QueryUtil.getCollectorPlatformMstPK(platformId);
		// 削除
		em.remove(entity);
	}

	/** サブプラットフォーム定義情報を登録する。<BR>
	 * 
	 * @param addCollectorSubPratformMst サブプラットフォーム定義情報
	 * @throws addCollectorSubPratformMst
	 * @throws EntityExistsException
	 */
	public static void addCollectorSubPratformMst(CollectorSubPlatformMstData data) throws EntityExistsException {
		JpaTransactionManager jtm = new JpaTransactionManager();

		try {
			// インスタンス生成
			CollectorSubPlatformMstEntity entity = new CollectorSubPlatformMstEntity(data.getSubPlatformId());
			// 重複チェック
			jtm.checkEntityExists(CollectorSubPlatformMstEntity.class, entity.getSubPlatformId());
			entity.setSubPlatformName(data.getSubPlatformName());
			entity.setType(data.getType());
			entity.setOrderNo(data.getOrderNo().intValue());
		} catch (EntityExistsException e) {
			m_log.info("addCollectorSubPratformMst() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage());
			throw e;
		}
	}

	/**
	 * 指定されたサブプラットフォームIDのサブプラットフォーム定義情報を削除する。<BR>
	 * 
	 * @param subPlatformId サブプラットフォームID
	 * @throws FacilityNotFound
	 */
	public static void deleteCollectorSubPratformMst(String subPlatformId) throws FacilityNotFound {
		EntityManager em = new JpaTransactionManager().getEntityManager();

		// 該当するプラットフォーム定義情報を取得
		CollectorSubPlatformMstEntity entity = QueryUtil.getCollectorSubPlatformMstPK(subPlatformId);
		// 削除
		em.remove(entity);
	}
	
	/** スコープを追加する。<BR>
	 * 
	 * @param parentFacilityId スコープを配置する親スコープのファシリティID（空文字の場合はルートスコープとなる）
	 * @param property 追加するスコープ情報
	 * @param modifyUserId 作業ユーザID
	 * @param displaySortOrder 表示ソート順位
	 * @param topicSendFlg 更新を周知する場合はtrue, 周知しない場合はfalse
	 * @throws FacilityNotFound
	 * @throws EntityExistsException
	 * @throws HinemosUnknown
	 */
	public static void addScope(String parentFacilityId, ScopeInfo property, String modifyUserId, int displaySortOrder, boolean topicSendFlg)
			throws FacilityNotFound, EntityExistsException, HinemosUnknown {

		JpaTransactionManager jtm = new JpaTransactionManager();

		/** ローカル変数 */
		FacilityEntity parentFacility = null;
		FacilityEntity facility = null;
		String facilityId = null;

		/** メイン処理 */
		m_log.debug("adding a scope...");

		try {
			// 入力値（ファシリティID）の格納
			facilityId = property.getFacilityId();

			// 親ファシリティがスコープかどうかを確認する
			if (! ObjectValidator.isEmptyString(parentFacilityId)) {
				parentFacility = QueryUtil.getFacilityPK_NONE(parentFacilityId);
				if (!FacilityUtil.isScope(parentFacility)) {
					HinemosUnknown e = new HinemosUnknown("a parent's facility is not a scope. (parentFacilityId = "
							+ parentFacilityId + ", facilityId = " + facilityId + ")");
					m_log.info("addScope() : "
							+ e.getClass().getSimpleName() + ", " + e.getMessage());
					throw e;
				}
			}

			// ファシリティインスタンスの生成
			facility = new FacilityEntity(facilityId);
			// 重複チェック
			jtm.checkEntityExists(FacilityEntity.class, facility.getFacilityId());
			facility.setDisplaySortOrder(displaySortOrder);
			facility.setFacilityType(FacilityConstant.TYPE_SCOPE);
			facility.setValid(ValidConstant.TYPE_VALID);
			setFacility(facility, property, modifyUserId, false);

			// ファシリティ関連インスタンスの生成
			if (! ObjectValidator.isEmptyString(parentFacilityId)) {
				assignFacilityToScope(parentFacilityId, facilityId, false);
			}
		} catch (FacilityNotFound e) {
			throw e;
		} catch (EntityExistsException e) {
			m_log.info("addScope() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage());
			throw e;
		} catch (HinemosUnknown e) {
			throw e;
		}

		m_log.info("addScope() successful in adding a scope . (parentFacilityId = " + parentFacilityId + ", facilityId = " + facilityId + ")");
	}

	/**
	 * スコープを変更する。<BR>
	 * 
	 * @param property 変更後のスコープ情報
	 * @param modifyUserId 作業ユーザID
	 * @param topicSendFlg 更新を周知する場合はtrue, 周知しない場合はfalse
	 * @throws FacilityNotFound
	 * @throws InvalidRole
	 */
	public static void modifyScope(ScopeInfo property, String modifyUserId, boolean topicSendFlg) throws FacilityNotFound, InvalidRole {

		/** ローカル変数 */
		FacilityEntity facility = null;
		String facilityId = null;

		/** メイン処理 */
		m_log.debug("modifing a scope...");

		// 入力値（ファシリティID）の格納
		facilityId = property.getFacilityId();

		// ファシリティインスタンスを取得
		facility = QueryUtil.getFacilityPK(facilityId, ObjectPrivilegeMode.WRITE);

		// ファシリティがスコープかどうかを確認する
		if (!FacilityUtil.isScope(facility)) {
			FacilityNotFound e = new FacilityNotFound("this facility is not a scope. (facilityId = " + facilityId + ")");
			m_log.info("modifyScope() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage());
			throw e;
		}

		// 変更後の値を格納する
		setFacility(facility, property, modifyUserId, false);

		m_log.info("modifyScope() successful in modifing a scope. (facilityId = " + facilityId + ")");
	}

	/**
	 * オーナーロールスコープを削除する。<BR>
	 * 
	 * @param facilityId 削除するスコープのファシリティID
	 * @param modifyUserId 作業ユーザID
	 * @param topicSendFlg 更新を周知する場合はtrue, 周知しない場合はfalse
	 * @throws UsedFacility
	 * @throws FacilityNotFound
	 * @throws InvalidRole
	 * @throws HinemosUnknown
	 */
	public static void deleteOwnerRoleScope(String facilityId, String modifyUserId, boolean topicSendFlg) throws UsedFacility, FacilityNotFound, InvalidRole, HinemosUnknown {

		/** メイン処理 */
		m_log.debug("deleting a owner role scope with sub scopes...");


		// 該当するファシリティインスタンスを取得（オブジェクト権限チェックなし）
		FacilityEntity facility = QueryUtil.getFacilityPK_NONE(facilityId);

		// 関連インスタンス、ファシリティインスタンスを削除する
		deleteScopeRecursive(facility);

		m_log.info("deleteScope() successful in deleting a owner role scope with sub scopes. (facilityId = " + facilityId + ")");
	}
	
	/**
	 * スコープを削除する。<BR>
	 * 
	 * @param facilityId 削除するスコープのファシリティID
	 * @param modifyUserId 作業ユーザID
	 * @param topicSendFlg 更新を周知する場合はtrue, 周知しない場合はfalse
	 * @throws UsedFacility
	 * @throws FacilityNotFound
	 * @throws InvalidRole
	 * @throws HinemosUnknown
	 */
	public static void deleteScope(String facilityId, String modifyUserId, boolean topicSendFlg) throws UsedFacility, FacilityNotFound, InvalidRole, HinemosUnknown {

		/** メイン処理 */
		m_log.debug("deleting a scope with sub scopes...");


		// 該当するファシリティインスタンスを取得
		FacilityEntity facility = QueryUtil.getFacilityPK(facilityId, ObjectPrivilegeMode.WRITE);

		// 関連インスタンス、ファシリティインスタンスを削除する
		deleteScopeRecursive(facility);

		m_log.info("deleteScope() successful in deleting a scope with sub scopes. (facilityId = " + facilityId + ")");
	}

	/**
	 * サブスコープを含めて、スコープを削除する。<BR>
	 * 
	 * @param scope 削除するスコープインスタンス
	 * @throws UsedFacility
	 * @throws HinemosUnknown
	 */
	private static void deleteScopeRecursive(FacilityEntity scope) throws UsedFacility, HinemosUnknown, FacilityNotFound {
		EntityManager em = new JpaTransactionManager().getEntityManager();

		/** ローカル変数 */
		String facilityId = null;

		/** メイン処理 */
		facilityId = scope.getFacilityId();

		// スコープでない場合はエラーとする
		if (!FacilityUtil.isScope(scope)) {
			HinemosUnknown e = new HinemosUnknown("this facility is not a scope. (facilityId = " + facilityId + ")");
			m_log.info("deleteScopeRecursive() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage());
			throw e;
		}

		// 直下にスコープを存在する場合、そのスコープおよびサブスコープを削除する
		List<FacilityEntity> childEntities = QueryUtil.getChildFacilityEntity(scope.getFacilityId());
		if (childEntities != null && childEntities.size() > 0) {
			Iterator<FacilityEntity> iter = childEntities.iterator();
			while(iter.hasNext()) {
				FacilityEntity childEntity = iter.next();
				if (FacilityUtil.isScope(childEntity)) {
					childEntity.tranSetUncheckFlg(true);
					// リレーションを削除する
					FacilityRelationEntity facilityRelationEntity
					= QueryUtil.getFacilityRelationPk(scope.getFacilityId(), childEntity.getFacilityId());
					em.remove(facilityRelationEntity);
					deleteScopeRecursive(childEntity);
				}
			}
		}
		em.remove(scope);

		m_log.info("deleteScopeRecursive() successful in deleting a scope. (facilityId = " + facilityId + ")");
	}

	/**
	 * ノードを追加する。<BR>
	 * 
	 * @param property 追加するノード情報
	 * @param modifyUserId 作業ユーザID
	 * @param displaySortOrder 表示ソート順位
	 * @param topicSendFlg 更新を周知する場合はtrue, 周知しない場合はfalse
	 * @throws EntityExistsException
	 * @throws FacilityNotFound
	 * @throws HinemosUnknown
	 */
	public static void addNode(NodeInfo property, String modifyUserId, int displaySortOrder, boolean topicSendFlg)
			throws EntityExistsException, FacilityNotFound, HinemosUnknown {

		JpaTransactionManager jtm = new JpaTransactionManager();

		/** ローカル変数 */
		FacilityEntity facility = null;
		NodeEntity node = null;
		String facilityId = null;
		Boolean valid = true;

		/** メイン処理 */
		m_log.debug("adding a node...");

		try {
			facilityId = property.getFacilityId();

			valid = property.isValid();

			// インスタンス生成
			facility = new FacilityEntity(facilityId);
			// 重複チェック
			jtm.checkEntityExists(FacilityEntity.class, facility.getFacilityId());
			facility.setFacilityType(FacilityConstant.TYPE_NODE);
			facility.setDisplaySortOrder(displaySortOrder);
			facility.setValid(FacilityUtil.getValid(valid));
			setFacility(facility, property, modifyUserId, false);

			// ノードインスタンスの生成
			node = new NodeEntity(facility);
			// 重複チェック
			jtm.checkEntityExists(NodeEntity.class, node.getFacilityId());
			setNode(node, property, false);

			// ファシリティ関連インスタンスの生成
			assignFacilityToScope(FacilityTreeAttributeConstant.REGISTEREFD_SCOPE, facilityId, false);

			// オーナー別スコープへの登録
			assignFacilityToScope(facility.getOwnerRoleId(), facilityId, false);
		} catch (FacilityNotFound e) {
			throw e;
		} catch (EntityExistsException e) {
			m_log.info("addNode() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage());
			throw e;
		} catch (HinemosUnknown e) {
			throw e;
		}

		m_log.info("addNode() successful in adding a node. (facilityId = " + facilityId + ")");
	}

	/**
	 * ノードを変更する。<BR>
	 * 
	 * @param property 変更後のノード情報
	 * @param modifyUserId 作業ユーザID
	 * @param topicSendFlg 更新を周知する場合はtrue, 周知しない場合はfalse
	 * @throws FacilityNotFound
	 * @throws InvalidRole
	 * @throws HinemosUnknown
	 */
	public static void modifyNode(NodeInfo property, String modifyUserId, boolean topicSendFlg)
			throws FacilityNotFound, InvalidRole, HinemosUnknown {

		/** ローカル変数 */
		FacilityEntity facility = null;
		NodeEntity node = null;
		String facilityId = null;
		Boolean valid = true;

		/** メイン処理 */
		m_log.debug("modifing a node...");

		facilityId = property.getFacilityId();

		valid = property.isValid();

		// ファシリティインスタンスの生成
		facility = QueryUtil.getFacilityPK(facilityId, ObjectPrivilegeMode.WRITE);
		if (valid == null) {
			HinemosUnknown e = new HinemosUnknown("node's valid is invalid . (valid = " + valid + ")");
			m_log.info("modifyNode() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage());
			throw e;
		} else {
			facility.setValid(FacilityUtil.getValid(valid));
		}

		// ファシリティがノードかどうかを確認する
		if (!FacilityUtil.isNode(facility)) {
			FacilityNotFound e = new FacilityNotFound("this facility is not a node. (facilityId = " + facilityId + ")");
			m_log.info("modifyNode() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage());
			throw e;
		}

		setFacility(facility, property, modifyUserId, false);

		// ノードインスタンスの取得
		node = facility.getNodeEntity();

		// 変更情報の反映
		setNode(node, property, false);

		m_log.info("modifyNode() successful in modifing a node. (facilityId = " + facilityId + ")");
	}

	/**
	 * ノードを削除する。<BR>
	 * 
	 * @param facilityId 削除するノードのファシリティID
	 * @param modifyUserId 作業ユーザID
	 * @param topicSendFlg 更新を周知する場合はtrue, 周知しない場合はfalse
	 * @throws InvalidRole
	 * @throws FacilityNotFound
	 */
	public static void deleteNode(String facilityId, String modifyUserId, boolean topicSendFlg) throws FacilityNotFound, InvalidRole {

		EntityManager em = new JpaTransactionManager().getEntityManager();

		/** ローカル変数 */
		FacilityEntity facility = null;

		/** メイン処理 */
		m_log.debug("deleting a node...");

		// 該当するファシリティインスタンスを取得
		facility = QueryUtil.getFacilityPK(facilityId, ObjectPrivilegeMode.WRITE);

		// ノードでない場合はエラーとする
		if (!FacilityUtil.isNode(facility)) {
			FacilityNotFound e = new FacilityNotFound("this facility is not a node. (facilityId = " + facilityId + ")");
			m_log.info("deleteNode() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage());
			throw e;
		}

		// ファシリティインスタンスを削除する（DELETE CASCADEで関連するインスタンスやも削除される）
		List<FacilityEntity> parentList = QueryUtil.getParentFacilityEntity(facility.getFacilityId());
		if (parentList != null) {
			for (FacilityEntity parent : parentList) {
				List<FacilityEntity> childList = QueryUtil.getChildFacilityEntity(parent.getFacilityId());
				Iterator<FacilityEntity> iter = childList.iterator();
				while(iter.hasNext()) {
					FacilityEntity childEntity = iter.next();
					if (childEntity.getFacilityId().equals(facilityId)) {
						iter.remove();
					}
				}
			}
		}
		em.remove(facility);

		m_log.info("deleteNode() successful in deleting a node. (facilityId = " + facilityId + ")");
	}


	/**
	 * スコープにノードを割り当てる。<BR>
	 * 
	 * @param parentFacilityId スコープのファシリティID
	 * @param facilityId ノードのファシリティID
	 * @param topicSendFlg 更新を周知する場合はtrue, 周知しない場合はfalse
	 * @throws FacilityNotFound
	 */
	private static void assignFacilityToScope(String parentFacilityId, String facilityId, boolean topicSendFlg) throws FacilityNotFound {
		/** ローカル変数 */
		String[] facilityIds = { facilityId };

		/** メイン処理 */
		assignFacilityToScope(parentFacilityId, facilityIds, topicSendFlg);
	}

	/**
	 * スコープにノードを割り当てる。<BR>
	 * 
	 * @param parentFacilityId スコープのファシリティID
	 * @param facilityIds ノードのファシリティID配列
	 * @param topicSendFlg 更新を周知する場合はtrue, 周知しない場合はfalse
	 * @throws FacilityNotFound
	 */
	public static void assignFacilityToScope(String parentFacilityId, String[] facilityIds, boolean topicSendFlg) throws FacilityNotFound {

		/** ローカル変数 */
		FacilityEntity scope = null;
		boolean relationExist = false;
		String facilityId = null;

		/** メイン処理 */
		m_log.debug("assigning facilities to a scope...");

		try {
			// スコープのファシリティインスタンスの取得
			scope = QueryUtil.getFacilityPK_NONE(parentFacilityId);
			scope.tranSetUncheckFlg(true);

			if (!FacilityUtil.isScope(scope)) {
				FacilityNotFound e = new FacilityNotFound("parent's facility is not a scope. (facilityId = " + parentFacilityId + ")");
				m_log.info("assignFacilityToScope() : "
						+ e.getClass().getSimpleName() + ", " + e.getMessage());
				throw e;
			}

			for (int i = 0; i < facilityIds.length; i++) {
				facilityId = facilityIds[i];

				// 割り当て済みのファシリティかどうかを確認する
				relationExist = false;
				List<FacilityEntity> childFacilityEntities
				= QueryUtil.getChildFacilityEntity(scope.getFacilityId());
				if (childFacilityEntities != null) {
					for (FacilityEntity childEntity : childFacilityEntities) {
						if (childEntity.getFacilityId().compareTo(facilityId) == 0) {
							relationExist = true;
							break;
						}
					}
				}
				if (relationExist) {
					m_log.info("assignFacilityToScope() skipped assinging a facility to a scope. (parentFacilityId = " + parentFacilityId + ", facilityId = " + facilityId + ")");
				} else {
					new FacilityRelationEntity(scope.getFacilityId(), facilityId);
					m_log.info("assignFacilityToScope() successful in assinging a facility to a scope. (parentFacilityId = " + parentFacilityId + ", facilityId = " + facilityId + ")");
				}
			}
		} catch (FacilityNotFound e) {
			throw e;
		}

		m_log.info("assignFacilityToScope() successful in assigning facilities to a scope.");
	}

	/**
	 * スコープからノードの割り当てを解除する。<BR>
	 * 
	 * @param parentFacilityId スコープのファシリティID
	 * @param facilityIds ノードのファシリティID
	 * @param modifyUserId 作業ユーザID
	 * @param topicSendFlg 更新を周知する場合はtrue, 周知しない場合はfalse
	 * 
	 * @throws FacilityNotFound
	 * @throws InvalidRole
	 */
	public static void releaseNodeFromScope(String parentFacilityId, String[] facilityIds, String modifyUserId, boolean topicSendFlg)
			throws FacilityNotFound, InvalidRole {
		EntityManager em = new JpaTransactionManager().getEntityManager();
		
		/** ローカル変数 */
		FacilityEntity facility = null;
		String facilityId = null;

		/** メイン処理 */
		m_log.debug("releasing nodes from a scope...");

		// 該当するファシリティインスタンスを取得
		facility = QueryUtil.getFacilityPK(parentFacilityId);

		if (!FacilityUtil.isScope(facility)) {
			FacilityNotFound e = new FacilityNotFound("parent's facility is not a scope. (parentFacilityId = " + parentFacilityId + ")");
			m_log.info("releaseNodeFromScope() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage());
			throw e;
		}
		
		for (int i = 0; i < facilityIds.length; i++) {
			facilityId = facilityIds[i];
			FacilityRelationEntity relation 
			= QueryUtil.getFacilityRelationPk(parentFacilityId, facilityId);
			em.remove(relation);
			m_log.info("releaseNodeFromScope() successful in releaseing a node. (parentFacilityId = " + parentFacilityId + ", facilityId = " + facilityId + ")");
		}
		
		m_log.info("releaseNodeFromScope() successful in releasing nodes from a scope.");
	}

	/**
	 * Propertyインスタンスに格納された値をファシリティインスタンスに格納する。<BR>
	 * 
	 * @param facility 格納先となるファシリティインスタンス
	 * @param property 格納する情報
	 * @param modifyUserId 作業ユーザID
	 * @param skipIfEmptyFlg trueにすると、Propertyインスタンスの各格納値がnullあるいは空文字の場合に格納しない
	 * @throws FacilityNotFound
	 */
	private static void setFacility(FacilityEntity facility, FacilityInfo property, String modifyUserId, boolean skipIfEmptyFlg) throws FacilityNotFound {
		/** ローカル変数 */
		String facilityName = null;
		String description = null;
		Timestamp now = null;

		/** メイン処理 */
		// 現在日時を取得
		now = new Timestamp(new Date().getTime());

		if (FacilityUtil.isScope(facility)) {
			// 入力値（ファシリティ名）の格納
			facilityName = property.getFacilityName();

			// 入力値（説明）の格納
			description = property.getDescription();

		} else if (FacilityUtil.isNode(facility)) {
			// 入力値（ファシリティ名）の格納
			facilityName = property.getFacilityName();

			// 入力値（説明）の格納
			description = property.getDescription();
		} else {
			FacilityNotFound e = new FacilityNotFound("this facility's type is invalid. (facilityType = " + facility.getFacilityType() + ")");
			m_log.info("setFacility() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage());
			throw e;
		}

		// ファシリティインスタンスへの入力値の格納
		if ( ! (skipIfEmptyFlg && ObjectValidator.isEmptyString(facilityName)) ) {
			facility.setFacilityName(facilityName);
		}
		if ( ! (skipIfEmptyFlg && ObjectValidator.isEmptyString(description)) ) {
			facility.setDescription(description);
		}
		// アイコン名の格納
		facility.setIconImage(property.getIconImage());
		// 入力値（オーナーロールID）の格納
		facility.setOwnerRoleId(property.getOwnerRoleId());
		if (ObjectValidator.isEmptyString(facility.getCreateUserId())) {
			facility.setCreateUserId(modifyUserId);
			facility.setCreateDatetime(now);
		}
		facility.setModifyUserId(modifyUserId);
		facility.setModifyDatetime(now);
	}

	/**
	 * Propertyインスタンス上のノード情報をノードインスタンスに格納する。<BR>
	 * 
	 * @param node 格納先となるノードインスタンス
	 * @param property ノード情報
	 * @param skipIfEmptyFlg trueにすると、Propertyインスタンスの各格納値がnullあるいは空文字の場合に格納しない
	 * @throws EntityExistsException
	 * @throws HinemosUnknown
	 */
	private static void setNode(NodeEntity node, NodeInfo property, boolean skipIfEmptyFlg) throws EntityExistsException, HinemosUnknown {
		m_log.debug("setNode() : facilityId = " + property.getFacilityId());

		/** ローカル変数 */

		// HW
		String platformFamily = null;
		String subPlatformFamily = null;
		String hardwareType = null;

		// IPアドレス
		Integer ipAddressVersion = null;
		String ipAddressV4 = null;
		String ipAddressV6 = null;

		// OS
		String nodeName = null;
		String osName = null;
		String osRelease = null;
		String osVersion = null;
		String characterSet = null;

		// Hinemosエージェント
		Integer agentAwakePort = null;

		// JOB
		Integer jobPriority = null;
		Integer jobMultiplicity = null;

		// SNMP
		Integer snmpPort = null;
		String snmpCommunity = null;
		String snmpVersion = null;
		Integer snmpTimeout = null;
		Integer snmpRetryCount = null;

		// WBEM
		String wbemUser = null;
		String wbemUserPassword = null;
		Integer wbemPort = null;
		String wbemProtocol = null;
		Integer wbemTimeout = null;
		Integer wbemRetryCount = null;

		// IPMI
		String ipmiIpAddress = null;
		Integer ipmiPort = null;
		String ipmiUser = null;
		String ipmiUserPassword = null;
		Integer ipmiTimeout = null;
		Integer ipmiRetries = null;
		String ipmiProtocol = null;
		String ipmiLevel = null;

		// WinRM
		String winrmUser = null;
		String winrmUserPassword = null;
		String winrmVersion = null;
		Integer winrmPort = null;
		String winrmProtocol = null;
		Integer winrmTimeout = null;
		Integer winrmRetries = null;

		// デバイス(主キー項目)
		Integer deviceIndex = null;
		String deviceType = null;
		String deviceName = null;

		// サーバ仮想化
		String vmNodeType = null;
		String vmManagementNode = null;
		String vmName = null;
		String vmId = null;
		String vmUser = null;
		String vmUserPassword = null;
		String vmProtocol = null;

		// ネットワーク仮想化
		String vNetSwitchType = null;
		String vNetHostNode = null;
		String openFlowDataPathId = null;
		String openFlowCtrlIpAddress = null;

		// クラウド管理
		String cloudNodeType = null;
		String cloudService = null;
		String cloudAccountResouce = null;
		String cloudResourceType = null;
		String cloudResourceId = null;
		String cloudRegion = null;
		String cloudZone = null;

		// 保守
		String administrator = null;
		String contact = null;

		/** メイン処理 */

		// HW
		platformFamily = property.getPlatformFamily();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(platformFamily))) {
			node.setPlatformFamily(platformFamily);
		}
		subPlatformFamily = property.getSubPlatformFamily();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(subPlatformFamily))) {
			node.setSubPlatformFamily(subPlatformFamily);
		}
		hardwareType = property.getHardwareType();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(hardwareType))) {
			node.setHardwareType(hardwareType);
		}


		// IPアドレス関連
		ipAddressVersion = property.getIpAddressVersion();
		if (! (skipIfEmptyFlg && ipAddressVersion == -1)) {
			node.setIpAddressVersion(ipAddressVersion);
		}
		ipAddressV4 = property.getIpAddressV4();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(ipAddressV4))) {
			node.setIpAddressV4(ipAddressV4);
		}
		ipAddressV6 = property.getIpAddressV6();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(ipAddressV6))) {
			node.setIpAddressV6(Ipv6Util.expand(ipAddressV6));
		}


		// ホスト名(複数項目)
		if (! skipIfEmptyFlg) {
			List<NodeHostnameEntityPK> nodeHostnameEntityPkList = new ArrayList<NodeHostnameEntityPK>();

			if (property.getNodeHostnameInfo() != null) {
				for (NodeHostnameInfo hostname : property.getNodeHostnameInfo()) {
					NodeHostnameEntityPK entityPk = new NodeHostnameEntityPK(node.getFacilityId(), hostname.getHostname());
					try {
						QueryUtil.getNodeHostnamePK(entityPk);
					} catch (FacilityNotFound e) {
						// 新規登録
						new NodeHostnameEntity(node, hostname.getHostname());
					}
					nodeHostnameEntityPkList.add(entityPk);
				}
			}
			// 不要なNodeHostnameEntityを削除
			node.deleteNodeHostnameEntities(nodeHostnameEntityPkList);
		}

		// OS関連
		nodeName = property.getNodeName();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(nodeName))) {
			node.setNodeName(nodeName);
		}
		osName = property.getOsName();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(osName))) {
			node.setOsName(osName);
		}
		osRelease = property.getOsRelease();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(osRelease))) {
			node.setOsRelease(osRelease);
		}
		osVersion = property.getOsVersion();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(osVersion))) {
			node.setOsVersion(osVersion);
		}
		characterSet = property.getCharacterSet();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(characterSet))) {
			node.setCharacterSet(characterSet);
		}

		// Hinemosエージェント
		agentAwakePort = property.getAgentAwakePort();
		if (! (skipIfEmptyFlg && agentAwakePort == -1)) {
			node.setAgentAwakePort(agentAwakePort);
		}

		// JOB
		jobPriority = property.getJobPriority();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(jobPriority))) {
			node.setJobPriority(jobPriority);
		}
		jobMultiplicity = property.getJobMultiplicity();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(jobMultiplicity))) {
			node.setJobMultiplicity(jobMultiplicity);
		}

		// SNMP関連
		snmpPort = property.getSnmpPort();
		if (! (skipIfEmptyFlg && snmpPort == -1)) {
			node.setSnmpPort(snmpPort);
		}
		snmpCommunity = property.getSnmpCommunity();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(snmpCommunity))) {
			node.setSnmpCommunity(snmpCommunity);
		}
		snmpVersion = property.getSnmpVersion();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(snmpVersion))) {
			node.setSnmpVersion(snmpVersion);
		}
		snmpTimeout = property.getSnmpTimeout();
		if (! (skipIfEmptyFlg && snmpTimeout == -1)) {
			node.setSnmpTimeout(snmpTimeout);
		}
		snmpRetryCount = property.getSnmpRetryCount();
		if (! (skipIfEmptyFlg && snmpRetryCount == -1)) {
			node.setSnmpRetryCount(snmpRetryCount);
		}

		// WBEM関連
		wbemUser = property.getWbemUser();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(wbemUser))) {
			node.setWbemUser(wbemUser);
		}
		wbemUserPassword = property.getWbemUserPassword();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(wbemUserPassword))) {
			node.setWbemUserPassword(wbemUserPassword);
		}
		wbemPort = property.getWbemPort();
		if (! (skipIfEmptyFlg && wbemPort == -1)) {
			node.setWbemPort(wbemPort);
		}
		wbemProtocol = property.getWbemProtocol();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(wbemProtocol))) {
			node.setWbemProtocol(wbemProtocol);
		}
		wbemTimeout = property.getWbemTimeout();
		if (! (skipIfEmptyFlg && wbemTimeout == -1)) {
			node.setWbemTimeout(wbemTimeout);
		}
		wbemRetryCount = property.getWbemRetryCount();
		if (! (skipIfEmptyFlg && wbemRetryCount == -1)) {
			node.setWbemRetryCount(wbemRetryCount);
		}

		// IPMI関連
		ipmiIpAddress = property.getIpmiIpAddress();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(ipmiIpAddress))) {
			node.setIpmiIpAddress(ipmiIpAddress);
		}
		ipmiPort = property.getIpmiPort();
		if (! (skipIfEmptyFlg && ipmiPort == -1)) {
			node.setIpmiPort(ipmiPort);
		}
		ipmiUser = property.getIpmiUser();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(ipmiUser))) {
			node.setIpmiUser(ipmiUser);
		}
		ipmiUserPassword = property.getIpmiUserPassword();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(ipmiUserPassword))) {
			node.setIpmiUserPassword(ipmiUserPassword);
		}
		ipmiTimeout = property.getIpmiTimeout();
		if (! (skipIfEmptyFlg && ipmiTimeout == -1)) {
			node.setIpmiTimeout(ipmiTimeout);
		}
		ipmiRetries = property.getIpmiRetries();
		if (! (skipIfEmptyFlg && ipmiRetries == -1)) {
			node.setIpmiRetryCount(ipmiRetries);
		}
		ipmiProtocol = property.getIpmiProtocol();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(ipmiProtocol))) {
			node.setIpmiProtocol(ipmiProtocol);
		}
		ipmiLevel = property.getIpmiLevel();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(ipmiLevel))) {
			node.setIpmiLevel(ipmiLevel);
		}

		// WinRM関連
		winrmUser = property.getWinrmUser();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(winrmUser))) {
			node.setWinrmUser(winrmUser);
		}
		winrmUserPassword = property.getWinrmUserPassword();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(winrmUserPassword))) {
			node.setWinrmUserPassword(winrmUserPassword);
		}
		winrmVersion = property.getWinrmVersion();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(winrmVersion))) {
			node.setWinrmVersion(winrmVersion);
		}
		winrmPort = property.getWinrmPort();
		if (! (skipIfEmptyFlg && winrmPort == -1)) {
			node.setWinrmPort(winrmPort);
		}
		winrmProtocol = property.getWinrmProtocol();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(winrmProtocol))) {
			node.setWinrmProtocol(winrmProtocol);
		}
		winrmTimeout = property.getWinrmTimeout();
		if (! (skipIfEmptyFlg && winrmTimeout == -1)) {
			node.setWinrmTimeout(winrmTimeout);
		}
		winrmRetries = property.getWinrmRetries();
		if (! (skipIfEmptyFlg && winrmRetries == -1)) {
			node.setWinrmRetryCount(winrmRetries);
		}
		// デバイス関連
		// 汎用デバイス関連
		if (! skipIfEmptyFlg) {
			List<NodeDeviceEntityPK> nodeDeviceEntityPkList = new ArrayList<NodeDeviceEntityPK>();
			if (property.getNodeDeviceInfo() != null) {
				for (NodeDeviceInfo deviceProperty : property.getNodeDeviceInfo()) {
					if(deviceProperty != null){
						// 入力チェック
						if (deviceProperty.getDeviceIndex() != -1
								&& ! ObjectValidator.isEmptyString(deviceProperty.getDeviceType())
								&& ! ObjectValidator.isEmptyString(deviceProperty.getDeviceName())) {
							NodeDeviceEntityPK entityPk
							= new NodeDeviceEntityPK(node.getFacilityId(),
									deviceProperty.getDeviceIndex(),
									deviceProperty.getDeviceType(),
									deviceProperty.getDeviceName());
							NodeDeviceEntity entity = null;
							try {
								entity = QueryUtil.getNodeDeviceEntityPK(entityPk);
							} catch (FacilityNotFound e) {
								// 新規登録
								entity = new NodeDeviceEntity(entityPk, node);
							}
							entity.setDeviceDisplayName(deviceProperty.getDeviceDisplayName());
							entity.setDeviceSize(deviceProperty.getDeviceSize());
							entity.setDeviceSizeUnit(deviceProperty.getDeviceSizeUnit());
							entity.setDeviceDescription(deviceProperty.getDeviceDescription());
							nodeDeviceEntityPkList.add(entityPk);
						} else {
							HinemosUnknown e = new HinemosUnknown("both type and index of device are required. " +
									"(facilityId = " + node.getFacilityId() + ", deviceType = " + deviceType + ", deviceIndex = " + deviceIndex + ", deviceName = " + deviceName + ")");
							m_log.info("setNode() : "
									+ e.getClass().getSimpleName() + ", " + e.getMessage());
							throw e;
						}
					}
				}
			}
			// 不要なNodeDeviceEntityを削除
			node.deleteNodeDeviceEntities(nodeDeviceEntityPkList);
		}

		// CPUデバイス関連
		if (! skipIfEmptyFlg) {
			List<NodeCpuEntityPK> nodeCpuEntityPkList = new ArrayList<NodeCpuEntityPK>();

			if (property.getNodeCpuInfo() != null) {
				for (NodeCpuInfo cpuProperty : property.getNodeCpuInfo()) {
					if(cpuProperty != null){
						// 入力チェック
						if (cpuProperty.getDeviceIndex() != -1
								&& ! ObjectValidator.isEmptyString(cpuProperty.getDeviceType())
								&& ! ObjectValidator.isEmptyString(cpuProperty.getDeviceName())) {
							NodeCpuEntityPK entityPk
							= new NodeCpuEntityPK(node.getFacilityId(),
									cpuProperty.getDeviceIndex(),
									cpuProperty.getDeviceType(),
									cpuProperty.getDeviceName());
							NodeCpuEntity entity = null;
							try {
								entity = QueryUtil.getNodeCpuEntityPK(entityPk);
							} catch (FacilityNotFound e) {
								// 新規登録
								entity = new NodeCpuEntity(entityPk, node);
							}
							entity.setDeviceDisplayName(cpuProperty.getDeviceDisplayName());
							entity.setDeviceSize(cpuProperty.getDeviceSize());
							entity.setDeviceSizeUnit(cpuProperty.getDeviceSizeUnit());
							entity.setDeviceDescription(cpuProperty.getDeviceDescription());
							nodeCpuEntityPkList.add(entityPk);
						} else {
							HinemosUnknown e = new HinemosUnknown("both type and index of cpu are required. " +
									"(facilityId = " + node.getFacilityId() + ", deviceType = " + deviceType + ", deviceIndex = " + deviceIndex + ", deviceName = " + deviceName + ")");
							m_log.info("setNode() : "
									+ e.getClass().getSimpleName() + ", " + e.getMessage());
							throw e;
						}
					}
				}
			}
			// 不要なNodeCpuEntityを削除
			node.deleteNodeCpuEntities(nodeCpuEntityPkList);
		}

		// MEMデバイス関連
		if (!skipIfEmptyFlg) {
			List<NodeMemoryEntityPK> nodeMemoryEntityPkList = new ArrayList<NodeMemoryEntityPK>();

			if (property.getNodeMemoryInfo() != null) {
				for (NodeMemoryInfo memoryProperty : property.getNodeMemoryInfo()) {
					if(memoryProperty != null){
						// 入力チェック
						if (memoryProperty.getDeviceIndex() != -1
								&& ! ObjectValidator.isEmptyString(memoryProperty.getDeviceType())
								&& ! ObjectValidator.isEmptyString(memoryProperty.getDeviceName())) {
							NodeMemoryEntityPK entityPk
							= new NodeMemoryEntityPK(node.getFacilityId(),
									memoryProperty.getDeviceIndex(),
									memoryProperty.getDeviceType(),
									memoryProperty.getDeviceName());
							NodeMemoryEntity entity = null;
							try {
								entity = QueryUtil.getNodeMemoryEntityPK(entityPk);
							} catch (FacilityNotFound e) {
								// 新規登録
								entity = new NodeMemoryEntity(entityPk, node);
							}
							entity.setDeviceDisplayName(memoryProperty.getDeviceDisplayName());
							entity.setDeviceSize(memoryProperty.getDeviceSize());
							entity.setDeviceSizeUnit(memoryProperty.getDeviceSizeUnit());
							entity.setDeviceDescription(memoryProperty.getDeviceDescription());
							nodeMemoryEntityPkList.add(entityPk);
						} else {
							HinemosUnknown e = new HinemosUnknown("both type and index of memory are required. " +
									"(facilityId = " + node.getFacilityId() + ", deviceType = " + deviceType + ", deviceIndex = " + deviceIndex + ", deviceName = " + deviceName + ")");
							m_log.info("setNode() : "
									+ e.getClass().getSimpleName() + ", " + e.getMessage());
							throw e;
						}
					}
				}
			}
			// 不要なNodeMemoryEntityを削除
			node.deleteNodeMemoryEntities(nodeMemoryEntityPkList);
		}

		// NICデバイス関連
		if (! skipIfEmptyFlg) {
			List<NodeNetworkInterfaceEntityPK> nodeNetworkInterfaceEntityPkList = new ArrayList<NodeNetworkInterfaceEntityPK>();

			if (property.getNodeNetworkInterfaceInfo() != null) {
				for (NodeNetworkInterfaceInfo nicProperty : property.getNodeNetworkInterfaceInfo()) {
					if(nicProperty != null){
						// 入力チェック
						if (nicProperty.getDeviceIndex() != -1
								&& ! ObjectValidator.isEmptyString(nicProperty.getDeviceType())
								&& ! ObjectValidator.isEmptyString(nicProperty.getDeviceName())) {
							NodeNetworkInterfaceEntityPK entityPk
							= new NodeNetworkInterfaceEntityPK(node.getFacilityId(),
									nicProperty.getDeviceIndex(),
									nicProperty.getDeviceType(),
									nicProperty.getDeviceName());
							NodeNetworkInterfaceEntity entity = null;
							try {
								entity = QueryUtil.getNodeNetworkInterfaceEntityPK(entityPk);
							} catch (FacilityNotFound e) {
								// 新規登録
								entity = new NodeNetworkInterfaceEntity(entityPk, node);
							}
							entity.setDeviceDisplayName(nicProperty.getDeviceDisplayName());
							entity.setDeviceSize(nicProperty.getDeviceSize());
							entity.setDeviceSizeUnit(nicProperty.getDeviceSizeUnit());
							entity.setDeviceDescription(nicProperty.getDeviceDescription());
							entity.setDeviceNicIpAddress(nicProperty.getNicIpAddress());
							entity.setDeviceNicMacAddress(nicProperty.getNicMacAddress());
							nodeNetworkInterfaceEntityPkList.add(entityPk);
						} else {
							HinemosUnknown e = new HinemosUnknown("both type and index of nic are required. " +
									"(facilityId = " + node.getFacilityId() + ", deviceType = " + deviceType + ", deviceIndex = " + deviceIndex + ", deviceName = " + deviceName + ")");
							m_log.info("setNode() : "
									+ e.getClass().getSimpleName() + ", " + e.getMessage());
							throw e;
						}
					}
				}
			}
			// 不要なNodeNetworkInterfaceEntityを削除
			node.deleteNodeNetworkInterfaceEntities(nodeNetworkInterfaceEntityPkList);
		}

		// DISKデバイス関連
		if (! skipIfEmptyFlg) {
			List<NodeDiskEntityPK> nodeDiskEntityPkList = new ArrayList<NodeDiskEntityPK>();

			if (property.getNodeDiskInfo() != null) {
				for (NodeDiskInfo diskProperty : property.getNodeDiskInfo()) {
					if(diskProperty != null){
						// 入力チェック
						if (diskProperty.getDeviceIndex() != -1
								&& ! ObjectValidator.isEmptyString(diskProperty.getDeviceType())
								&& ! ObjectValidator.isEmptyString(diskProperty.getDeviceName())) {
							NodeDiskEntityPK entityPk
							= new NodeDiskEntityPK(node.getFacilityId(),
									diskProperty.getDeviceIndex(),
									diskProperty.getDeviceType(),
									diskProperty.getDeviceName());
							NodeDiskEntity entity = null;
							try {
								entity = QueryUtil.getNodeDiskEntityPK(entityPk);
							} catch (FacilityNotFound e) {
								// 新規登録
								entity = new NodeDiskEntity(entityPk, node);
							}
							entity.setDeviceDisplayName(diskProperty.getDeviceDisplayName());
							entity.setDeviceSize(diskProperty.getDeviceSize());
							entity.setDeviceSizeUnit(diskProperty.getDeviceSizeUnit());
							entity.setDeviceDescription(diskProperty.getDeviceDescription());
							entity.setDeviceDiskRpm(diskProperty.getDiskRpm());
							nodeDiskEntityPkList.add(entityPk);
						} else {
							HinemosUnknown e = new HinemosUnknown("both type and index of disk are required. " +
									"(facilityId = " + node.getFacilityId() + ", deviceType = " + deviceType + ", deviceIndex = " + deviceIndex + ", deviceName = " + deviceName + ")");
							m_log.info("setNode() : "
									+ e.getClass().getSimpleName() + ", " + e.getMessage());
							throw e;
						}
					}
				}
			}
			// 不要なNodeDiskEntityを削除
			node.deleteNodeDiskEntities(nodeDiskEntityPkList);
		}

		// ファイルシステム関連
		if (! skipIfEmptyFlg) {
			List<NodeFilesystemEntityPK> nodeFilesystemEntityPkList = new ArrayList<NodeFilesystemEntityPK>();

			if (property.getNodeFilesystemInfo() != null) {
				for (NodeFilesystemInfo filesystemProperty : property.getNodeFilesystemInfo()) {
					if(filesystemProperty != null){
						// 入力チェック
						if (filesystemProperty.getDeviceIndex() != -1
								&& ! ObjectValidator.isEmptyString(filesystemProperty.getDeviceType())
								&& ! ObjectValidator.isEmptyString(filesystemProperty.getDeviceName())) {
							NodeFilesystemEntityPK entityPk
							= new NodeFilesystemEntityPK(node.getFacilityId(),
									filesystemProperty.getDeviceIndex(),
									filesystemProperty.getDeviceType(),
									filesystemProperty.getDeviceName());
							NodeFilesystemEntity entity = null;
							try {
								entity = QueryUtil.getFilesystemDiskEntityPK(entityPk);
							} catch (FacilityNotFound e) {
								// 新規登録
								entity = new NodeFilesystemEntity(entityPk, node);
							}
							entity.setDeviceDisplayName(filesystemProperty.getDeviceDisplayName());
							entity.setDeviceSize(filesystemProperty.getDeviceSize());
							entity.setDeviceSizeUnit(filesystemProperty.getDeviceSizeUnit());
							entity.setDeviceDescription(filesystemProperty.getDeviceDescription());
							entity.setDeviceFilesystemType(filesystemProperty.getFilesystemType());
							nodeFilesystemEntityPkList.add(entityPk);
						} else {
							HinemosUnknown e = new HinemosUnknown("both type and index of filesystem are required. " +
									"(facilityId = " + node.getFacilityId() + ", deviceType = " + deviceType + ", deviceIndex = " + deviceIndex + ", deviceName = " + deviceName + ")");
							m_log.info("setNode() : "
									+ e.getClass().getSimpleName() + ", " + e.getMessage());
							throw e;
						}
					}
				}
			}
			// 不要なNodeFilesystemEntityを削除
			node.deleteNodeFilesystemEntities(nodeFilesystemEntityPkList);
		}


		// サーバ仮想化関連
		vmNodeType = property.getVmNodeType();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(vmNodeType))) {
			node.setVmNodeType(vmNodeType);
		}
		vmManagementNode = property.getVmManagementNode();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(vmManagementNode))) {
			node.setVmManagementNode(vmManagementNode);
		}
		vmName = property.getVmName();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(vmName))) {
			node.setVmName(vmName);
		}
		vmId = property.getVmId();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(vmId))) {
			node.setVmId(vmId);
		}
		vmUser = property.getVmUser();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(vmUser))) {
			node.setVmUser(vmUser);
		}
		vmUserPassword = property.getVmUserPassword();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(vmUserPassword))) {
			node.setVmUserPassword(vmUserPassword);
		}
		vmProtocol = property.getVmProtocol();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(vmProtocol))) {
			node.setVmProtocol(vmProtocol);
		}

		// ネットワーク仮想化関連
		vNetSwitchType = property.getvNetSwitchType();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(vNetSwitchType))) {
			node.setVnetSwitchType(vNetSwitchType);
		}
		vNetHostNode = property.getvNetHostNode();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(vNetHostNode))) {
			node.setVnetHostNode(vNetHostNode);
		}
		openFlowDataPathId = property.getOpenFlowDataPathId();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(openFlowDataPathId))) {
			node.setOpenflowDataPathId(openFlowDataPathId);
		}
		openFlowCtrlIpAddress = property.getOpenFlowCtrlIpAddress();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(openFlowCtrlIpAddress))) {
			node.setOpenflowCtrlIpAddress(openFlowCtrlIpAddress);
		}

		// クラウド管理関連
		cloudNodeType = property.getCloudNodeType();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(cloudNodeType))) {
			node.setCloudNodeType(cloudNodeType);
		}
		cloudService = property.getCloudService();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(cloudService))) {
			node.setCloudService(cloudService);
		}
		cloudAccountResouce = property.getCloudAccountResouce();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(cloudAccountResouce))) {
			node.setCloudAccountResouce(cloudAccountResouce);
		}
		cloudResourceType = property.getCloudResourceType();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(cloudResourceType))) {
			node.setCloudResourceType(cloudResourceType);
		}
		cloudResourceId = property.getCloudResourceId();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(cloudResourceId))) {
			node.setCloudResourceId(cloudResourceId);
		}
		cloudRegion = property.getCloudRegion();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(cloudRegion))) {
			node.setCloudRegion(cloudRegion);
		}
		cloudZone = property.getCloudZone();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(cloudZone))) {
			node.setCloudZone(cloudZone);
		}

		
		// ノード変数
		if (! skipIfEmptyFlg) {
			List<NodeVariableEntityPK> nodeVariableEntityPkList = new ArrayList<NodeVariableEntityPK>();

			if (property.getNodeVariableInfo() != null) {
				for (NodeVariableInfo variable : property.getNodeVariableInfo()) {
					if (variable != null) {
						NodeVariableEntityPK entityPk = new NodeVariableEntityPK(node.getFacilityId(), variable.getNodeVariableName());
						NodeVariableEntity entity = null;
						try {
							entity = QueryUtil.getNodeVariableEntityPK(entityPk);
						} catch (FacilityNotFound e) {
							// 新規登録
							entity = new NodeVariableEntity(node, variable.getNodeVariableName());
						}
						entity.setNodeVariableValue(variable.getNodeVariableValue());
						nodeVariableEntityPkList.add(entityPk);
					}
				}
			}
			// 不要なNodeVariableEntityを削除
			node.deleteNodeVariableEntities(nodeVariableEntityPkList);
		}

		// 管理情報関連
		administrator = property.getAdministrator();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(administrator))) {
			node.setAdministrator(administrator);
		}
		contact = property.getContact();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(contact))) {
			node.setContact(contact);
		}

		// 備考
		if (! skipIfEmptyFlg) {
			List<NodeNoteEntityPK> nodeNoteEntityPkList = new ArrayList<NodeNoteEntityPK>();

			if (property.getNodeNoteInfo() != null) {
				for (NodeNoteInfo note : property.getNodeNoteInfo()) {
					if(note != null){
						NodeNoteEntity entity = null;
						NodeNoteEntityPK entityPk = new NodeNoteEntityPK(node.getFacilityId(), note.getNoteId());
						try {
							entity = QueryUtil.getNodeNoteEntityPK(entityPk);
						} catch (FacilityNotFound e) {
							// 新規登録
							entity = new NodeNoteEntity(entityPk, node);
						}
						entity.setNote(note.getNote());
						nodeNoteEntityPkList.add(entityPk);
					}
				}
			}
			// 不要なNodeNoteEntityを削除
			node.deleteNodeNoteEntities(nodeNoteEntityPkList);
		}
	}
}
