/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.notify.util;

import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;
import java.nio.charset.CharsetEncoder;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.Locale;
import java.util.Map;
import java.util.Properties;
import java.util.StringTokenizer;
import java.util.Vector;

import javax.mail.AuthenticationFailedException;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeUtility;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.maintenance.HinemosPropertyTypeConstant;
import com.clustercontrol.maintenance.util.HinemosPropertyUtil;
import com.clustercontrol.notify.bean.NotifyRequestMessage;
import com.clustercontrol.notify.bean.OutputBasicInfo;
import com.clustercontrol.notify.mail.bean.MailTemplateInfo;
import com.clustercontrol.notify.mail.session.MailTemplateControllerBean;
import com.clustercontrol.notify.model.NotifyMailInfoEntity;
import com.clustercontrol.util.Messages;
import com.clustercontrol.util.StringBinder;
import com.clustercontrol.util.apllog.AplLogger;
import com.sun.mail.smtp.SMTPAddressFailedException;

/**
 * メールを送信するクラス<BR>
 *
 * @version 3.0.0
 * @since 3.0.0
 */
public class SendMail implements Notifier {

	/** ログ出力のインスタンス。 */
	private static Log m_log = LogFactory.getLog(SendMail.class);

	/** 日時フォーマット。 */
	private static final String SUBJECT_DATE_FORMAT = "yyyy-MM-dd HH:mm:ss";

	/** ロケール情報。 */
	private static Locale m_local = Locale.getDefault();

	/** アドレスロケール */
	public static final String _charsetAddressDefault = "UTF-8";

	/** メール件名ロケール */
	public static final String _charsetSubjectDefault = "UTF-8";

	/**`メール本文ロケール */
	public static final String _charsetContentDefault = "UTF-8";

	/**
	 * メールの送信を行います。
	 *
	 * @param outputInfo 出力・通知情報
	 */
	@Override
	public synchronized void notify(NotifyRequestMessage message) {

		sendMail(message.getOutputInfo(), message.getNotifyId());
	}

	/**
	 * メールの送信を行います。
	 *
	 */
	private void sendMail(OutputBasicInfo outputInfo, String notifyId) {

		if (m_log.isDebugEnabled()) {
			m_log.debug("sendMail() " + outputInfo);
		}

		try {
			NotifyMailInfoEntity mailInfo = QueryUtil.getNotifyMailInfoPK(notifyId);

			// メールの件名を指定
			String subject = getSubject(outputInfo, mailInfo);

			// メールの内容を指定
			String content = getContent(outputInfo, mailInfo);

			/**
			 * メール送信
			 */
			String address = null;
			switch (outputInfo.getPriority()) {
			case PriorityConstant.TYPE_INFO:
				address = mailInfo.getInfoMailAddress();
				break;
			case PriorityConstant.TYPE_WARNING:
				address = mailInfo.getWarnMailAddress();
				break;
			case PriorityConstant.TYPE_CRITICAL:
				address = mailInfo.getCriticalMailAddress();
				break;
			case PriorityConstant.TYPE_UNKNOWN:
				address = mailInfo.getUnknownMailAddress();
				break;
			default:
				break;
			}
			if (address == null) {
				m_log.info("address is null");
				return;
			}
			if (address.length() == 0) {
				m_log.info("address.length()==0");
				return;
			}

			String changeAddress = null;
			try {
				Map<String, String> param = NotifyUtil.createParameter(outputInfo);
				StringBinder binder = new StringBinder(param);
				changeAddress = binder.bindParam(address);
			} catch (Exception e) {
				m_log.warn("sendMail() : "
						+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
				changeAddress = address;
			}
			StringTokenizer t = new StringTokenizer(changeAddress, ";");
			ArrayList<String> toAddressList = new ArrayList<String>();
			ArrayList<String> ccAddressList = new ArrayList<String>();
			ArrayList<String> bccAddressList = new ArrayList<String>();
			String separator = ":";
			String ccPrefix = "CC" + separator;
			String bccPrefix = "BCC" + separator;
			while (t.hasMoreTokens()) {
				String addr = t.nextToken();
				if (addr.startsWith(ccPrefix)) {
					ccAddressList.add(addr.substring(ccPrefix.length()));
				} else if (addr.startsWith(bccPrefix)) {
					bccAddressList.add(addr.substring(bccPrefix.length()));
				} else {
					toAddressList.add(addr);
				}
			}
			String[] toAddress = toAddressList.toArray(new String[0]);

			if (toAddress == null || toAddress.length <= 0) {
				m_log.debug("sendMail() : mail address is empty");
				return;
			}

			try {
				this.sendMail(toAddress, ccAddressList.toArray(new String[0]), bccAddressList.toArray(new String[0]), subject, content);
			} catch (AuthenticationFailedException e) {
				String detailMsg = "cannot connect to the mail server due to an Authentication Failure";
				m_log.warn("sendMail() " + e.getMessage() + " : " + detailMsg + " : "
						+ e.getClass().getSimpleName() + ", " + e.getMessage());
				internalErrorNotify(notifyId, "007", detailMsg);
			} catch (SMTPAddressFailedException e) {
				String detailMsg = e.getMessage() + "(SMTPAddressFailedException)";
				m_log.warn("sendMail() " + e.getMessage() + " : " + detailMsg + " : "
						+ e.getClass().getSimpleName() + ", " + e.getMessage());
				internalErrorNotify(notifyId, "007", detailMsg);
			} catch (MessagingException e) {
				String detailMsg = e.getCause() != null ? e.getMessage() + "\nCause : " + e.getCause().getMessage() : e.getMessage();
				m_log.warn("sendMail() " + e.getMessage() + " : " + detailMsg + " : "
						+ e.getClass().getSimpleName() + ", " + e.getMessage());
				internalErrorNotify(notifyId, "007", detailMsg);
			}
		} catch (Exception e1) {
			String detailMsg = e1.getCause() != null ? e1.getMessage() + "\nCause : " + e1.getCause().getMessage() : e1.getMessage();
			m_log.warn("sendMail() " + e1.getMessage() + " : " + detailMsg + detailMsg + " : "
					+ e1.getClass().getSimpleName() + ", " + e1.getMessage(), e1);
			internalErrorNotify(notifyId, "007", detailMsg);
		}
	}

	/**
	 * メールを送信します。
	 *
	 * <p>
	 * 下記の情報は、ファイルより取得します。
	 * <p>
	 * <ul>
	 * <li>差出人アドレス</li>
	 * <li>差出人個人名</li>
	 * <li>返信の送信先アドレス</li>
	 * <li>返信の送信先個人名</li>
	 * <li>エラー送信先アドレス</li>
	 * </ul>
	 *
	 * @param addressTo
	 *            送信先アドレス
	 * @param source
	 *            出力内容
	 * @return 送信に成功した場合、<code> true </code>
	 * @throws MessagingException
	 * @throws NamingException
	 * @throws UnsupportedEncodingException
	 */
	public void sendMail(String[] toAddressStr, String subject, String content)
			throws MessagingException, UnsupportedEncodingException {
		sendMail(toAddressStr, null, null, subject, content);
	}

	public void sendMail(String[] toAddressStr, String[] ccAddressStr, String[] bccAddressStr, String subject, String content)
			throws MessagingException, UnsupportedEncodingException {

		if (toAddressStr == null || toAddressStr.length <= 0) {
			// 何もせず終了
			return;
		}

		/*
		 *  https://javamail.java.net/nonav/docs/api/com/sun/mail/smtp/package-summary.html
		 */
		Properties _properties = new Properties();
		_properties.setProperty("mail.debug", Boolean.toString(HinemosPropertyUtil.getHinemosPropertyBool("mail.debug", false)));
		_properties.setProperty("mail.store.protocol", HinemosPropertyUtil.getHinemosPropertyStr("mail.store.protocol", "pop3"));
		String protocol = HinemosPropertyUtil.getHinemosPropertyStr("mail.transport.protocol", "smtp");
		_properties.setProperty("mail.transport.protocol", protocol);
		_properties.put("mail.smtp.socketFactory", javax.net.SocketFactory.getDefault());
		_properties.put("mail.smtp.ssl.socketFactory", javax.net.ssl.SSLSocketFactory.getDefault());
		
		setProperties(_properties, "mail." + protocol + ".user", HinemosPropertyTypeConstant.TYPE_STRING);
		setProperties(_properties, "mail." + protocol + ".host", HinemosPropertyTypeConstant.TYPE_STRING);
		setProperties(_properties, "mail." + protocol + ".port", HinemosPropertyTypeConstant.TYPE_NUMERIC);
		setProperties(_properties, "mail." + protocol + ".connectiontimeout", HinemosPropertyTypeConstant.TYPE_NUMERIC);
		setProperties(_properties, "mail." + protocol + ".timeout", HinemosPropertyTypeConstant.TYPE_NUMERIC);
		setProperties(_properties, "mail." + protocol + ".writetimeout", HinemosPropertyTypeConstant.TYPE_NUMERIC);
		setProperties(_properties, "mail." + protocol + ".from", HinemosPropertyTypeConstant.TYPE_STRING);
		setProperties(_properties, "mail." + protocol + ".localhost", HinemosPropertyTypeConstant.TYPE_STRING);
		setProperties(_properties, "mail." + protocol + ".localaddress", HinemosPropertyTypeConstant.TYPE_STRING);
		setProperties(_properties, "mail." + protocol + ".localport", HinemosPropertyTypeConstant.TYPE_NUMERIC);
		setProperties(_properties, "mail." + protocol + ".ehlo", HinemosPropertyTypeConstant.TYPE_TRUTH);
		setProperties(_properties, "mail." + protocol + ".auth", HinemosPropertyTypeConstant.TYPE_TRUTH);
		setProperties(_properties, "mail." + protocol + ".auth.mechanisms", HinemosPropertyTypeConstant.TYPE_STRING);
		setProperties(_properties, "mail." + protocol + ".auth.login.disable", HinemosPropertyTypeConstant.TYPE_TRUTH);
		setProperties(_properties, "mail." + protocol + ".auth.plain.disable", HinemosPropertyTypeConstant.TYPE_TRUTH);
		setProperties(_properties, "mail." + protocol + ".auth.digest-md5.disable", HinemosPropertyTypeConstant.TYPE_TRUTH);
		setProperties(_properties, "mail." + protocol + ".auth.ntlm.disable", HinemosPropertyTypeConstant.TYPE_TRUTH);
		setProperties(_properties, "mail." + protocol + ".auth.ntlm.domain", HinemosPropertyTypeConstant.TYPE_STRING);
		setProperties(_properties, "mail." + protocol + ".auth.ntlm.flags", HinemosPropertyTypeConstant.TYPE_NUMERIC);
		setProperties(_properties, "mail." + protocol + ".submitter", HinemosPropertyTypeConstant.TYPE_STRING);
		setProperties(_properties, "mail." + protocol + ".dsn.notify", HinemosPropertyTypeConstant.TYPE_STRING);
		setProperties(_properties, "mail." + protocol + ".dsn.ret", HinemosPropertyTypeConstant.TYPE_STRING);
		setProperties(_properties, "mail." + protocol + ".allow8bitmime", HinemosPropertyTypeConstant.TYPE_TRUTH);
		setProperties(_properties, "mail." + protocol + ".sendpartial", HinemosPropertyTypeConstant.TYPE_TRUTH);
		setProperties(_properties, "mail." + protocol + ".sasl.enable", HinemosPropertyTypeConstant.TYPE_TRUTH);
		setProperties(_properties, "mail." + protocol + ".sasl.mechanisms", HinemosPropertyTypeConstant.TYPE_STRING);
		setProperties(_properties, "mail." + protocol + ".sasl.authorizationid", HinemosPropertyTypeConstant.TYPE_STRING);
		setProperties(_properties, "mail." + protocol + ".sasl.realm", HinemosPropertyTypeConstant.TYPE_STRING);
		setProperties(_properties, "mail." + protocol + ".sasl.usecanonicalhostname", HinemosPropertyTypeConstant.TYPE_TRUTH);
		setProperties(_properties, "mail." + protocol + ".quitwait", HinemosPropertyTypeConstant.TYPE_TRUTH);
		setProperties(_properties, "mail." + protocol + ".reportsuccess", HinemosPropertyTypeConstant.TYPE_TRUTH);
		setProperties(_properties, "mail." + protocol + ".socketFactory.class", HinemosPropertyTypeConstant.TYPE_STRING);
		setProperties(_properties, "mail." + protocol + ".socketFactory.fallback", HinemosPropertyTypeConstant.TYPE_TRUTH);
		setProperties(_properties, "mail." + protocol + ".socketFactory.port", HinemosPropertyTypeConstant.TYPE_NUMERIC);
		setProperties(_properties, "mail." + protocol + ".starttls.enable", HinemosPropertyTypeConstant.TYPE_TRUTH);
		setProperties(_properties, "mail." + protocol + ".starttls.required", HinemosPropertyTypeConstant.TYPE_TRUTH);
		setProperties(_properties, "mail." + protocol + ".socks.host", HinemosPropertyTypeConstant.TYPE_STRING);
		setProperties(_properties, "mail." + protocol + ".socks.port", HinemosPropertyTypeConstant.TYPE_STRING);
		setProperties(_properties, "mail." + protocol + ".mailextension", HinemosPropertyTypeConstant.TYPE_STRING);
		setProperties(_properties, "mail." + protocol + ".userset", HinemosPropertyTypeConstant.TYPE_TRUTH);
		setProperties(_properties, "mail." + protocol + ".noop.strict", HinemosPropertyTypeConstant.TYPE_TRUTH);
		
		setProperties(_properties, "mail.smtp.ssl.enable", HinemosPropertyTypeConstant.TYPE_TRUTH);
		setProperties(_properties, "mail.smtp.ssl.checkserveridentity", HinemosPropertyTypeConstant.TYPE_TRUTH);
		setProperties(_properties, "mail.smtp.ssl.trust", HinemosPropertyTypeConstant.TYPE_STRING);
		setProperties(_properties, "mail.smtp.ssl.socketFactory.class", HinemosPropertyTypeConstant.TYPE_STRING);
		setProperties(_properties, "mail.smtp.ssl.socketFactory.port", HinemosPropertyTypeConstant.TYPE_NUMERIC);
		setProperties(_properties, "mail.smtp.ssl.protocols", HinemosPropertyTypeConstant.TYPE_STRING);
		setProperties(_properties, "mail.smtp.ssl.ciphersuites", HinemosPropertyTypeConstant.TYPE_STRING);

		/**
		 * メールの設定をDBから取得
		 */
		String _loginUser = HinemosPropertyUtil.getHinemosPropertyStr("mail.transport.user", "nobody");
		String _loginPassword = HinemosPropertyUtil.getHinemosPropertyStr("mail.transport.password", "password");
		String _fromAddress = HinemosPropertyUtil.getHinemosPropertyStr("mail.from.address", "admin@hinemos.com");
		String _fromPersonalName = HinemosPropertyUtil.getHinemosPropertyStr("mail.from.personal.name", "Hinemos Admin");
		_fromPersonalName = convertNativeToAscii(_fromPersonalName);
		
		String _replyToAddress = HinemosPropertyUtil.getHinemosPropertyStr("mail.reply.to.address", "admin@hinemos.com");
		String _replyToPersonalName =HinemosPropertyUtil.getHinemosPropertyStr("mail.reply.personal.name", "Hinemos Admin");
		_replyToPersonalName = convertNativeToAscii(_replyToPersonalName);
		
		String _errorsToAddress = HinemosPropertyUtil.getHinemosPropertyStr("mail.errors.to.address", "admin@hinemos.com");

		int _transportTries = HinemosPropertyUtil.getHinemosPropertyNum("mail.transport.tries", 1);
		int _transportTriesInterval = HinemosPropertyUtil.getHinemosPropertyNum("mail.transport.tries.interval", 10000);

		String _charsetAddress = HinemosPropertyUtil.getHinemosPropertyStr("mail.charset.address", _charsetAddressDefault);
		String _charsetSubject = HinemosPropertyUtil.getHinemosPropertyStr("mail.charset.subject", _charsetSubjectDefault);
		String _charsetContent = HinemosPropertyUtil.getHinemosPropertyStr("mail.charset.content", _charsetContentDefault);

		m_log.debug("initialized mail sender : from_address = " + _fromAddress
				+ ", From = " + _fromPersonalName + " <" + _replyToAddress + ">"
				+ ", Reply-To = " + _replyToPersonalName + " <" + _replyToAddress + ">"
				+ ", Errors-To = " + _errorsToAddress
				+ ", tries = " + _transportTries
				+ ", tries-interval = " + _transportTriesInterval
				+ ", Charset [address:subject:content] = [" + _charsetAddress + ":" + _charsetSubject + ":" + _charsetContent + "]");

		// JavaMail Sessionリソース検索
		Session session = Session.getInstance(_properties);

		Message mineMsg = new MimeMessage(session);

		// 送信元メールアドレスと送信者名を指定
		if (_fromAddress != null && _fromPersonalName != null) {
			mineMsg.setFrom(new InternetAddress(_fromAddress, _fromPersonalName, _charsetAddress));
		} else if (_fromAddress != null && _fromPersonalName == null) {
			mineMsg.setFrom(new InternetAddress(_fromAddress));
		}
		// REPLY-TOを指定
		if (_replyToAddress != null && _replyToPersonalName != null) {
			InternetAddress reply[] = { new InternetAddress(_replyToAddress, _replyToPersonalName, _charsetAddress) };
			mineMsg.setReplyTo(reply);
			mineMsg.reply(true);
		} else if (_replyToAddress != null && _replyToPersonalName == null) {
			InternetAddress reply[] = { new InternetAddress(_replyToAddress) };
			mineMsg.setReplyTo(reply);
			mineMsg.reply(true);
		}

		// ERRORS-TOを指定
		if (_errorsToAddress != null) {
			mineMsg.setHeader("Errors-To", _errorsToAddress);
		}

		// 送信先メールアドレスを指定
		// TO
		InternetAddress[] toAddress = this.getAddress(toAddressStr);
		if (toAddress != null && toAddress.length > 0) {
			mineMsg.setRecipients(javax.mail.Message.RecipientType.TO, toAddress);
		} else {
			return; // TOは必須
		}
		// CC
		if (ccAddressStr != null) {
			InternetAddress[] ccAddress = this.getAddress(ccAddressStr);
			if (ccAddress != null && ccAddress.length > 0) {
				mineMsg.setRecipients(javax.mail.Message.RecipientType.CC, ccAddress);
			}
		}
		// BCC
		if (bccAddressStr != null) {
			InternetAddress[] bccAddress = this.getAddress(bccAddressStr);
			if (bccAddress != null && bccAddress.length > 0) {
				mineMsg.setRecipients(javax.mail.Message.RecipientType.BCC, bccAddress);
			}
		}
		String message = "";
		if (toAddressStr != null) {
			message += "TO=" + Arrays.asList(toAddressStr);
		}
		if (ccAddressStr != null) {
			message += ", CC=" + Arrays.asList(ccAddressStr);
		}
		if (bccAddressStr != null) {
			message += ", BCC=" + Arrays.asList(bccAddressStr);
		}
		m_log.debug(message);

		// メールの件名を指定
		mineMsg.setSubject(MimeUtility.encodeText(subject, _charsetSubject, "B"));

		// メールの内容を指定
		mineMsg.setContent(content, "text/plain; charset=" + _charsetContent);

		// 送信日付を指定
		mineMsg.setSentDate(new Date());

		// 再送信フラグがtrueかつ再送回数以内の場合
		for (int i = 0; i < _transportTries; i++) {
			Transport transport = null;
			try {
				// メール送信
				transport = session.getTransport();
				boolean flag = HinemosPropertyUtil.getHinemosPropertyBool("mail." + protocol + ".auth", false);
				if(flag) {
					transport.connect(_loginUser, _loginPassword);
				} else {
					transport.connect();
				}
				transport.sendMessage(mineMsg, mineMsg.getAllRecipients());
				break;
			} catch (AuthenticationFailedException e) {
				throw e;
			} catch (SMTPAddressFailedException e) {
				throw e;
			} catch (MessagingException me) {
				//_transportTries中はsleep待ちのみ 
				if (i < (_transportTries - 1)) { 
					m_log.info("sendMail() : retry sendmail. " + me.getMessage());
					try {
						Thread.sleep(_transportTriesInterval);
					} catch (InterruptedException e) { }
				//_transportTriesの最後はINTERNALイベントの通知のためExceptionをthrow 
				} else {
					throw me;
				}
			} finally {
				if (transport != null) {
					transport.close();
				}
			}
		}
	}
	
	private String convertNativeToAscii(String nativeStr) {
		if (HinemosPropertyUtil.getHinemosPropertyBool("mail.native.to.ascii", false)){
			final CharsetEncoder asciiEncoder = Charset.forName("US-ASCII").newEncoder();
			final StringBuilder asciiStr = new StringBuilder();
			for (final Character character : nativeStr.toCharArray()) {
				if (asciiEncoder.canEncode(character)) {
					asciiStr.append(character);
				} else {
					asciiStr.append("\\u");
					asciiStr.append(Integer.toHexString(0x10000 | character).substring(1));
				}
			}
			
			return asciiStr.toString();
		} else {
			return nativeStr;
		}
	}

	/**
	 * 引数で指定された送信先アドレスの<code> InternetAddress </code>オブジェクトを返します。
	 *
	 * @param addressList
	 *            送信先アドレスの文字列配列
	 * @return <code> InternetAddress </code>オブジェクトの配列
	 */
	private InternetAddress[] getAddress(String[] addressList) {
		InternetAddress toAddress[] = null;
		Vector<InternetAddress> list = new Vector<InternetAddress>();
		if (addressList != null) {
			for (String address : addressList) {
				try {
					list.add(new InternetAddress(address));
				} catch (AddressException e) {
					m_log.info("getAddress() : "
							+ e.getClass().getSimpleName() + ", "
							+ address + ", "
							+ e.getMessage());
				}
			}
			if (list.size() > 0) {
				toAddress = new InternetAddress[list.size()];
				list.copyInto(toAddress);
			}
		}
		return toAddress;
	}

	/**
	 * メール件名を返します。
	 *
	 * @param source
	 *            出力内容
	 * @param mailInfo
	 *            通知内容
	 * @return メール件名
	 */
	public String getSubject(OutputBasicInfo source,
			NotifyMailInfoEntity mailInfo) {

		String subject = null;
		try {
			if (mailInfo != null
					&& mailInfo.getMailTemplateInfoEntity() != null
					&& mailInfo.getMailTemplateInfoEntity().getMailTemplateId() != null) {
				MailTemplateInfo templateData
				= new MailTemplateControllerBean().getMailTemplateInfo(
						mailInfo.getMailTemplateInfoEntity().getMailTemplateId());
				Map<String, String> param = NotifyUtil.createParameter(source, mailInfo.getNotifyInfoEntity());
				StringBinder binder = new StringBinder(param);
				subject = binder.bindParam(templateData.getSubject());
			} else {
				subject = Messages.getString("mail.subject", m_local) + "("
						+ PriorityConstant.typeToString(source.getPriority())
						+ ")";
			}
		} catch (Exception e) {
			m_log.warn("getSubject() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			// 例外発生時のメールサブジェクト
			return "Hinemos Notification";
		}

		return subject;
	}

	/**
	 * メール本文を返します。
	 *
	 * @param source
	 *            出力内容
	 * @param mailInfo
	 *            通知内容
	 * @return メール本文
	 */
	public String getContent(OutputBasicInfo source, NotifyMailInfoEntity mailInfo) {

		StringBuffer buf = new StringBuffer();
		SimpleDateFormat sdf = new SimpleDateFormat(SUBJECT_DATE_FORMAT);

		try {
			if (mailInfo != null
					&& mailInfo.getMailTemplateInfoEntity() != null
					&& mailInfo.getMailTemplateInfoEntity().getMailTemplateId() != null) {
				MailTemplateInfo mailData
				= new MailTemplateControllerBean().getMailTemplateInfo(
						mailInfo.getMailTemplateInfoEntity().getMailTemplateId());
				Map<String, String> param = NotifyUtil.createParameter(source,
						mailInfo.getNotifyInfoEntity());
				StringBinder binder = new StringBinder(param);
				buf.append(binder.bindParam(mailData.getBody() + "\n"));
			} else {
				buf.append(Messages.getString("generation.time", m_local)
						+ " : "
						+ sdf.format(source.getGenerationDate()) + "\n");
				buf.append(Messages.getString("application", m_local) + " : "
						+ source.getApplication() + "\n");
				buf.append(Messages.getString("priority", m_local) + " : "
						+ PriorityConstant.typeToString(source.getPriority())
						+ "\n");
				buf.append(Messages.getString("message", m_local) + " : "
						+ source.getMessage() + "\n");
				buf.append(Messages.getString("scope", m_local) + " : "
						+ source.getScopeText() + "\n");
			}
		} catch (Exception e) {
			m_log.warn("getContent() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			// 例外発生時のメール本文
			return "An error occurred creating message.";
		}

		// 改行コードをLFからCRLFに変更する。
		// 本来はJavaMailが変換するはずだが、変換されないこともあるので、
		// 予め変更しておく。
		String ret = buf.toString().replaceAll("\r\n", "\n").replaceAll("\n", "\r\n");

		return ret;
	}

	/**
	 * 通知失敗時の内部エラー通知を定義します
	 */
	@Override
	public void internalErrorNotify(String notifyId, String msgID, String detailMsg) {
		AplLogger apllog = new AplLogger("NOTIFY", "notify");
		String[] args = { notifyId };
		// 通知失敗メッセージを出力
		apllog.put("SYS", msgID, args, detailMsg);
	}

	private void setProperties(Properties prop, String key, int type) {
		switch (type) {
		case HinemosPropertyTypeConstant.TYPE_STRING:
			String strVal = HinemosPropertyUtil.getHinemosPropertyStr(key, null);
			if (strVal != null) {
				prop.setProperty(key, strVal);
			}
			break;
		case HinemosPropertyTypeConstant.TYPE_NUMERIC:
			Integer intVal = HinemosPropertyUtil.getHinemosPropertyNum(key, null);
			if (intVal != null) {
				prop.setProperty(key, intVal.toString());
			}
			break;
		case HinemosPropertyTypeConstant.TYPE_TRUTH:
			Boolean boolVal = HinemosPropertyUtil.getHinemosPropertyBool(key, null);
			if (boolVal != null) {
				prop.setProperty(key, boolVal.toString());
			}
			break;
		default:
			//上記以外はなにもしない
			break;
		}
	}
}
