#/*
# *  Copyright 2007 hkrn <hikarin@users.sourceforge.jp>
# *
# *  Licensed under the Apache License, Version 2.0 (the "License");
# *  you may not use this file except in compliance with the License.
# *  You may obtain a copy of the License at
# *
# *      http://www.apache.org/licenses/LICENSE-2.0
# *
# *  Unless required by applicable law or agreed to in writing, software
# *  distributed under the License is distributed on an "AS IS" BASIS,
# *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# *  See the License for the specific language governing permissions and
# *  limitations under the License.
# */
#
# $Id: bbs.pm 626 2007-05-12 13:42:33Z hikarin $
#

package Zeromin2::App::bbs;

use strict;
use Zeromin::BBS qw();

sub load {
    my ($zApp) = @_;
    $zApp->privilege() or return $zApp->return_value(1);

    require Zeromin::Category;
    my $iKernel   = $zApp->kernel();
    my $zBBS      = Zeromin::BBS->new( $iKernel, { bbs => '' } );
    my $zCategory = Zeromin::Category->new($iKernel);
    my $bbs_range = $zApp->user()->get_bbs_range();
    my ( $unijp, $encoding ) = $zApp->unijp();
    my @ret;

    if (@$bbs_range) {
        @ret = $zBBS->gets_with_page( $bbs_range, $unijp, $zCategory,
            $zApp->page_param() );
    }
    else {
        @ret
            = $zBBS->all_with_page( $unijp, $zCategory, $zApp->page_param() );
    }

    return $zApp->return_value( 0, @ret );
}

sub load_index {
    my ($zApp) = @_;
    $zApp->privilege( undef, 1 ) or return $zApp->return_value(1);

    require Img0ch::App::BBS;
    my $iBBS     = $zApp->bbs();
    my $iRequest = $zApp->request();
    my $iApp = Img0ch::App::BBS->new( $zApp->kernel(), $iRequest->request() );
    my ($param) = $iApp->update_index( $iBBS, 1, 1 );

    require Img0ch::Log;
    require Zeromin2::Util;
    my $privilege = $zApp->privilege('can_view_thread_log');
    my $rewriter  = Zeromin2::Util::get_rewriter($zApp);
    for my $thread ( @{ $param->{Threads} } ) {
        my $iLog = Img0ch::Log->new( $iBBS, $thread->{key} );
        for my $res ( @{ $thread->{Thread} } ) {
            if ($privilege) {
                my $info = $iLog->get( $res->{resno} );
                my $ip   = $info->[1];
                $ip
                    = $ip
                    ? join '.', unpack( 'C*', $info->[1] )
                    : '(null)';
                $res->{ip}     = $ip;
                $res->{serial} = $info->[2] || '(null)';
                $res->{agent}  = $info->[3] || '(null)';
            }
            else {
                $res->{ip}     = '(null)';
                $res->{serial} = '(null)';
                $res->{agent}  = '(null)';
            }
            $rewriter->( \$res->{text} );
        }
    }

    $rewriter->( \$param->{HEAD} );
    $rewriter->( \$param->{FOOT} );
    $rewriter->( \$param->{Banner} );
    $rewriter->( \$param->{SubBanner} );

    $zApp->add_template_param(
        {   %{ $iApp->setting()->get_hash() },
            %$param,
            BBSName => $iBBS->get_name()
        }
    );
    $zApp->encoding('Shift_JIS');
    return $zApp->return_value(0);
}

sub load_threads {
    _load_threads( $_[0], 'Zeromin/Subject.pm', 'Zeromin::Subject', undef );
}

sub load_special_threads {
    _load_threads( $_[0], 'Zeromin/Thread/Special.pm',
        'Zeromin::Thread::Special', undef );
}

sub load_pooled_threads {
    _load_threads( $_[0], 'Zeromin/Pool.pm', 'Zeromin::Pool',
        'can_enter_pool_section' );
}

sub load_archived_threads {
    _load_threads( $_[0], 'Zeromin/Archive.pm', 'Zeromin::Archive',
        'can_enter_archive_section' );
}

sub load_plugins {
    require Zeromin2::App::plugin;
    Zeromin2::App::plugin::load( $_[0], 1 );
}

sub save_plugins {
    require Zeromin2::App::plugin;
    Zeromin2::App::plugin::save( $_[0], 1 );
}

sub _load_threads {
    my ( $zApp, $module_file, $class_name, $privilege ) = @_;
    $zApp->privilege( $privilege, 1 ) or return $zApp->return_value( 1, [] );

    require $module_file;
    my $threads  = [];
    my $iBBS     = $zApp->bbs();
    my $zSubject = $class_name->new($iBBS);
    my ( $unijp, $encoding ) = $zApp->unijp();
    $zSubject->load();

    require Img0ch::Log;
    require Zeromin2::Util;
    my $log_privilege = $zApp->privilege('can_view_thread_log');
    my ( $keys, $page )
        = $zSubject->to_array_with_page( $zApp->page_param() );
    for my $key (@$keys) {
        my $agent = '(null)';
        my $ip    = '(null)';
        my $data  = $zSubject->get_subject($key);
        if ($log_privilege) {
            my $iLog = Img0ch::Log->new( $iBBS, $key );
            my $info = $iLog->get(1);
            $agent = $info->[3];
            if ( my $serial = $info->[2] ) {
                $ip = $serial;
            }
            else {
                $ip = $info->[1];
                $ip = $ip ? join '.', unpack( 'C*', $ip ) : '(null)';
            }
        }
        push @$threads,
            {
            agent   => $agent,
            count   => $data->[1],
            date    => Zeromin2::Util::format_date($key),
            key     => $key,
            ip      => $ip,
            subject => $unijp->set( $data->[0], $encoding )->get(),
            };
    }

    $zApp->add_template_param({ BBSName => $iBBS->get_name() });
    return $zApp->return_value( 0, $threads, $page );
}

sub create {
    my ($zApp) = @_;
    $zApp->check_csrf()                or return $zApp->return_value(1);
    $zApp->privilege('can_create_bbs') or return $zApp->return_value(1);

    my ( $unijp, $encoding ) = $zApp->unijp();
    my $iRequest = $zApp->request();
    my $error    = {};
    my $param    = {};
    _validate_bbs_create_param( $zApp, $param, $error )
        or return $zApp->return_value( $error->{code} );

    my $bbs     = $param->{bbs_dir};
    my $name    = $param->{bbs_name};
    my $subname = $param->{bbs_sub_name};

    require Zeromin::BBS;
    my $zBBS = Zeromin::BBS->new( $zApp->kernel(), { bbs => $bbs } );
    $zBBS->create(
        {   category   => $param->{bbs_category},
            request    => $iRequest,
            sub_title  => $unijp->set( $subname, 'utf8' )->$encoding,
            title      => $unijp->set( $name, 'utf8' )->$encoding,
            uploadable => $param->{bbs_uploadable},
        }
    ) or return $zApp->return_value(7);
    $zApp->logger( 1, 'CREATED BBS: %s', [$bbs] );

    $zApp->add_template_param( { Done_bbs_create => 1 } );
    return $zApp->return_value( 0, $zBBS->get_id() );
}

sub create_form {
    my ($zApp) = @_;
    $zApp->privilege('can_create_bbs') or return $zApp->return_value(1);

    require Zeromin::Category;
    my $iKernel   = $zApp->kernel();
    my $zCategory = Zeromin::Category->new($iKernel);

    my ( $unijp, $encoding ) = $zApp->unijp();
    my $zBBS = Zeromin::BBS->new( $iKernel, { bbs => '' } );
    my $param = {
        BBSList      => $zBBS->all( $unijp, $zCategory ),
        CategoryList => $zCategory->all()
    };

    $zApp->add_template_param($param);
    return $zApp->return_value( 0, $param );
}

sub change_category {
    my ($zApp) = @_;
    $zApp->check_csrf()                or return $zApp->return_value(1);
    $zApp->privilege('can_create_bbs') or return $zApp->return_value(1);

    my $iRequest = $zApp->request();
    my @param    = $iRequest->param('board');
    my $category = $iRequest->param('category');
    my $status   = {};

    for my $id (@param) {
        $id = Img0ch::Kernel::intval($id) || next;
        my $zBBS = Zeromin::BBS->new( $zApp->kernel(), { id => $id } );
        if ( $zBBS->change_category($category) ) {
            $zApp->logger(
                1,
                'CHANGED CATEGORY TO %s: %s',
                [ $category, $id ]
            );
            $status->{$id} = $category;
        }
        else {
            $zApp->logger(
                1,
                'TRIED CHANGING INEXIST CATEGORY (%s): %s',
                [ $category, $id ],
            );
            $status->{$id} = 0;
        }
    }
    $zApp->return_value( 0, $status );
}

sub update_index {
    _task(
        shift, 'can_update_bbs', 'update',
        'UPDATED INDEX ON BBS',
        'TRIED UPDATING INDEX ON BBS',
    );
}

sub update_subject {
    _task(
        shift, 'can_update_subject', 'update_subject',
        'UPDATED SUBJECT ON BBS',
        'TRIED UPDATING SUBJECT ON BBS',
    );
}

sub recreate_subject {
    _task(
        shift, 'can_recreate_subject', 'recreate_subject',
        'RECREATED SUBJECT ON BBS',
        'TRIED RECREATING SUBJECT ON BBS',
    );
}

sub repair_upload_file_table {
    _task(
        shift,
        'can_create_bbs',
        'repair_upload_file_table',
        'REPAIRED UPLOADED FILE TABLE ON BBS',
        'TRIED REPAIRING UPLOADED FILE TABLE ON BBS',
    );
}

sub repair_tag_table {
    _task(
        shift,
        'can_recreate_subject',
        'repair_tag_table',
        'REPAIRED UPLOADED FILE TAG TABLE ON BBS',
        'TRIED REPAIRING UPLOADED FILE TAG TABLE ON BBS',
    );
}

sub stop {
    _task( shift, 'can_remove_bbs', 'stop', 'STOPPED BBS',
        'TRIED STOPPING BBS',
    );
}

sub remove {
    _task( shift, 'can_remove_bbs', 'remove', 'REMOVED BBS',
        'TRIED REMOVING BBS',
    );
}

sub update_bbsmenu {
    _task2( shift, 'can_update_bbslist', 'update_bbsmenu', 'UPDATED BBSMENU',
    );
}

sub repair_bbs_table {
    _task2( shift, 'can_create_bbs', 'repair_bbs_table', 'REPAIRED BBS TABLE',
    );
}

sub pool_thread {
    require Zeromin2::App::pool;
    Zeromin2::App::pool::create(@_);
}

sub remove_thread {
    require Zeromin2::App::thread;
    Zeromin2::App::thread::remove(@_);
}

sub stop_thread {
    require Zeromin2::App::thread;
    Zeromin2::App::thread::stop(@_);
}

sub restart_thread {
    require Zeromin2::App::thread;
    Zeromin2::App::thread::restart(@_);
}

sub repair_thread_upload_file_table {
    require Zeromin2::App::thread;
    Zeromin2::App::thread::repair_upload_file_table(@_);
}

sub restore_pool {
    require Zeromin2::App::pool;
    Zeromin2::App::pool::restore(@_);
}

sub remove_pool {
    require Zeromin2::App::pool;
    Zeromin2::App::pool::remove(@_);
}

sub create_archive {
    require Zeromin2::App::archive;
    Zeromin2::App::archive::create(@_);
}

sub update_archive {
    require Zeromin2::App::archive;
    Zeromin2::App::archive::update(@_);
}

sub remove_archive {
    require Zeromin2::App::archive;
    Zeromin2::App::archive::remove(@_);
}

sub remove_res {
    require Zeromin2::App::thread;
    Zeromin2::App::thread::remove_res(@_);
}

sub erase_res {
    require Zeromin2::App::thread;
    Zeromin2::App::thread::erase_res(@_);
}

sub remove_file {
    require Zeromin2::App::thread;
    Zeromin2::App::thread::remove_file(@_);
}

sub edit_res_form {
    require Zeromin2::App::thread;
    Zeromin2::App::thread::edit_res_form(@_);
}

sub import_thread {
    require Zeromin2::App::thread;
    Zeromin2::App::thread::do_import(@_);
}

sub deny_ip_local {
    require Zeromin2::App::filter;
    Zeromin2::App::filter::deny_ip_local(@_);
}

sub deny_ip_global {
    require Zeromin2::App::filter;
    Zeromin2::App::filter::deny_ip_global(@_);
}

sub deny_serial_local {
    require Zeromin2::App::filter;
    Zeromin2::App::filter::deny_serial_local(@_);
}

sub deny_serial_global {
    require Zeromin2::App::filter;
    Zeromin2::App::filter::deny_serial_global(@_);
}

sub create_special_thread {
    require Zeromin2::App::thread;
    Zeromin2::App::thread::create_special(@_);
}

sub _task {
    my ( $zApp, $priv_meth, $task_meth, $success, $fail ) = @_;
    $zApp->check_csrf()          or return $zApp->return_value(1);
    $zApp->privilege($priv_meth) or return $zApp->return_value(1);

    my $zUser    = $zApp->user();
    my $iRequest = $zApp->request();
    my @param    = $iRequest->param('board');
    scalar @param or return $zApp->return_value(2);

    my $status = {};
    for my $id (@param) {
        $id = Img0ch::Kernel::intval($id) || next;
        $zUser->have_privilege($id) or next;
        my $zBBS = Zeromin::BBS->new( $zApp->kernel(), { id => $id } );
        my $name = $zBBS->get_name();
        if ( $zBBS->$task_meth( $id, $iRequest ) ) {
            $zApp->logger( 1, ( $success . ': %s' ), [$name] );
            $status->{$name} = 1;
        }
        else {
            $zApp->logger( 0, ( $fail . ': %s' ), [$name] );
            $status->{$name} = 0;
        }
    }

    return $zApp->return_value( 0, $status );
}

sub _task2 {
    my ( $zApp, $priv_meth, $task_meth, $log ) = @_;
    $zApp->check_csrf()          or return $zApp->return_value(1);
    $zApp->privilege($priv_meth) or return $zApp->return_value(1);

    my $zBBS = Zeromin::BBS->new( $zApp->kernel(), { id => 0 } );
    my $done = $zBBS->$task_meth();
    $zApp->logger( 1, $log, [] );

    return $zApp->return_value( 0, $done );
}

sub _validate_bbs_create_param {
    my ( $zApp, $param, $error ) = @_;
    my $iRequest = $zApp->request();
    my $category = $iRequest->param_int('category') || 1;
    my $dir      = $iRequest->param('dir');
    my $name     = $iRequest->param('BBS_TITLE');
    my $subname  = $iRequest->param('BBS_SUBTITLE');

    if ( $dir =~ /\A\s*\z/xms ) {
        $zApp->logger( 0, 'NO BBS DIRECTORY WAS SET' );
        %{$error} = ( code => 2 );
        return 0;
    }
    elsif ( $name =~ /\A\s*\z/xms ) {
        $zApp->logger( 0, 'NO BBS NAME WAS SET' );
        %{$error} = ( code => 3 );
        return 0;
    }
    elsif ( $subname =~ /\A\s*\z/xms ) {
        $zApp->logger( 0, 'NO BBS SUB NAME WAS SET' );
        %{$error} = ( code => 4 );
        return 0;
    }
    elsif ( $dir !~ /\A[\w\-\.]+\z/xms ) {
        $zApp->logger( 0, 'INVALID BBS DIRECTORY WAS SET' );
        %{$error} = ( code => 5 );
        return 0;
    }
    elsif ($dir eq 'test'
        or $dir eq '_system'
        or $dir eq '_template'
        or $dir eq 'static' )
    {
        $zApp->logger( 0, 'RESERVED WORD (BBS DIRECTORY) WAS SET' );
        %{$error} = ( code => 6 );
        return 0;
    }

    $param->{bbs_category}   = $category;
    $param->{bbs_dir}        = $dir;
    $param->{bbs_name}       = $name;
    $param->{bbs_sub_name}   = $subname;
    $param->{bbs_uploadable} = $iRequest->param('BBS_MODE') eq 'picture';
    1;
}

1;
__END__
