#/*
# *  Copyright 2007 hkrn <hikarin@users.sourceforge.jp>
# *
# *  Licensed under the Apache License, Version 2.0 (the "License");
# *  you may not use this file except in compliance with the License.
# *  You may obtain a copy of the License at
# *
# *      http://www.apache.org/licenses/LICENSE-2.0
# *
# *  Unless required by applicable law or agreed to in writing, software
# *  distributed under the License is distributed on an "AS IS" BASIS,
# *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# *  See the License for the specific language governing permissions and
# *  limitations under the License.
# */
#
# $Id: user.pm 390 2007-03-20 14:06:06Z hikarin $
#

package Zeromin2::App::user;

use strict;

sub load {
    my ($zApp) = @_;
    $zApp->privilege('can_enter_user_section')
        or return $zApp->return_value( 1, [] );
    return $zApp->return_value( 0,
        $zApp->user()->get_users_with_page( $zApp->page_param() ) );
}

sub save {
    my ($zApp) = @_;
    $zApp->check_csrf() or return $zApp->return_value(1);
    $zApp->privilege('can_edit_user')
        or $zApp->privilege('can_create_user')
        or return $zApp->return_value(1);

    my $zUser    = $zApp->user();
    my $iRequest = $zApp->request();
    my $error    = {};
    my $param    = {};

    _validate_user_param( $zApp, $zUser, $iRequest, $param, $error )
        or return $zApp->return_value( $error->{code} );
    my $id   = $iRequest->param('id');
    my $user = $zUser->get_user($id);
    my $name = $param->{name};
    my $mail = $param->{mail};
    my $pass = $param->{pass};
    my $gid  = $param->{gid};
    $zApp->add_template_param(
        {   Name => $name,
            Mail => $mail
        }
    );

    if ( defined $user->{id} ) {
        $zApp->privilege('can_edit_user') or return $zApp->return_value(1);
        $zUser->edit(
            $id,
            {   mail => $mail,
                pass => $pass,
                gid  => $gid,
            }
        );
        $zUser->save();
        $zApp->logger(
            1,
            'EDIT USER: %s(%s) with %s belongs %d',
            [ $name, $mail, $pass, $gid ],
        );
        $zApp->add_template_param( { Done_user_edit => 1 } );
        $zApp->return_value(0);
    }
    else {
        $zApp->privilege('can_create_user') or return $zApp->return_value(1);
        $zUser->add(
            {   name => $name,
                mail => $mail,
                pass => $pass,
                gid  => $gid,
            }
        ) or return $zApp->return_value(10);
        $zUser->save();
        $zApp->logger(
            1,
            'CREATE USER: %s(%s) with %s belongs %d',
            [ $name, $mail, $pass, $gid ],
        );
        $zApp->add_template_param( { Done_user_create => 1 } );
        $zApp->return_value( 0, $zUser->get_user_id($name) );
    }
}

sub remove {
    my ($zApp) = @_;
    $zApp->check_csrf() or return $zApp->return_value(1);
    $zApp->privilege('can_remove_user')
        or return $zApp->return_value(1);

    my $zUser  = $zApp->user();
    my $count  = $zUser->get_user_count();
    my $status = {};
    my @ids    = $zApp->request()->param('id');
    my $remove = scalar @ids;
    $remove or return $zApp->return_value(2);
    $remove == $count and return $zApp->return_value(3);

    for my $id (@ids) {
        if ( $zUser->remove($id) ) {
            $zApp->logger( 1, 'REMOVE USER: %s', [$id] );
            $status->{$id} = 1;
        }
        else {
            $zApp->logger( 0, 'TRIED REMOVING INEXIST USER: %s', [$id] );
            $status->{$id} = 0;
        }
    }

    $zUser->save();
    return $zApp->return_value( 0, $status );
}

sub create {
    my ($zApp) = @_;
    $zApp->privilege('can_enter_user_section')
        or return $zApp->return_value(1);

    $zApp->add_template_param( { Groups => $zApp->user()->get_groups() } );
    return $zApp->return_value(0);
}

sub edit {
    my ($zApp) = @_;
    $zApp->privilege('can_enter_user_section')
        or return $zApp->return_value(1);

    my $zUser = $zApp->user();
    my $user  = $zUser->get_user( $zApp->request()->param('id') );
    my $gid   = $user->{gid};
    my $group = $zUser->get_groups();
    for my $grp (@$group) {
        $grp->{id} == $gid and $grp->{selected} = 1;
    }

    $zApp->add_template_param( { Groups => $group, %$user } );
    return $zApp->return_value(0);
}

sub _validate_user_param {
    my ( $zApp, $zUser, $iRequest, $param, $error ) = @_;

    my $name = $iRequest->param('name');
    if ( $name eq '' ) {
        $zApp->logger( 0, 'NO USER NAME WAS SET' );
        %{$error} = ( code => 2 );
        return 0;
    }
    my $mail = $iRequest->param('mail');
    if ( $mail eq '' ) {
        $zApp->logger( 0, 'NO USER MAIL WAS SET' );
        %{$error} = ( code => 3 );
        return 0;
    }
    my $group = Img0ch::Kernel::intval( $iRequest->param('group') );
    if ( !$group ) {
        $zApp->logger( 0, 'NO USER GROUP WAS SET' );
        %{$error} = ( code => 4 );
        return 0;
    }
    my $pass = $iRequest->param('pass');
    if ( $pass eq '' ) {
        $zApp->logger( 0, 'NO USER PASS WAS SET' );
        %{$error} = ( code => 5 );
        return 0;
    }
    my $conf = $iRequest->param('confirm');
    if ( $conf eq '' ) {
        $zApp->logger( 0, 'NO USER CONFIRM PASS WAS SET' );
        %{$error} = ( code => 6 );
        return 0;
    }
    if ( $mail !~ /\A[\w\-\.]+\@[\w\-\.]+\z/xms ) {
        $zApp->logger( 0, 'INVALID USER MAIL WAS SET' );
        %{$error} = ( code => 7 );
        return 0;
    }
    if ( $pass ne $conf ) {
        $zApp->logger( 0, 'NOT EQUAL USER PASS AND USER CONFIRM PASS' );
        %{$error} = ( code => 8 );
        return 0;
    }
    if ( $zUser->get_group_name($group) eq '' ) {
        $zApp->logger( 0, 'INVALID USER GROUP WAS SET' );
        %{$error} = ( code => 9 );
        return 0;
    }

    $param->{name} = $name;
    $param->{mail} = $mail;
    $param->{pass} = $pass;
    $param->{gid}  = $group;
    1;
}

1;
__END__
