#/*
# *  Copyright 2007-2009 hkrn <hikarin@users.sourceforge.jp>
# *
# *  Licensed under the Apache License, Version 2.0 (the "License");
# *  you may not use this file except in compliance with the License.
# *  You may obtain a copy of the License at
# *
# *      http://www.apache.org/licenses/LICENSE-2.0
# *
# *  Unless required by applicable law or agreed to in writing, software
# *  distributed under the License is distributed on an "AS IS" BASIS,
# *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# *  See the License for the specific language governing permissions and
# *  limitations under the License.
# */
#
# $Id: Revision.pm 1827 2009-04-08 15:24:43Z hikarin $
#

package Img0ch::Thread::Revision;

use strict;

sub new {
    my ( $iClass, $iBBS, $key ) = @_;
    my $iKernel = $iBBS->get_kernel();
    my $repos_path
        = Img0ch::Maple::create_repository_subdirs( $key, 'revision', $iBBS );
    my $iRepos = $iKernel->get_repos_nocache($repos_path);

    bless {
        _bbs     => $iBBS,
        _count   => $iRepos->get_int('I:T:R.count'),
        _kernel  => $iKernel,
        _key     => $key,
        __now    => time(),
        __prefix => join( '.', 'I:T:R', $iBBS->get_name(), $key ),
        _rs      => $iRepos,
        __revs   => [],
    }, $iClass;
}

sub load {
    my ($iTRevision) = @_;
    $iTRevision->{_rs}->load();
    return 1;
}

sub save {
    my ($iTRevision) = @_;
    my $iRepos       = $iTRevision->{_rs};
    my $removed      = $iTRevision->{__revs};
    if ( !@$removed ) {
        $iRepos->save();
        return 1;
    }

    my $iKernel    = $iTRevision->{_kernel};
    my $count      = $iTRevision->{_count};
    my $prefix_top = $iTRevision->{__prefix};
    for my $one (@$removed) {
        my ( $resno, $now, $name, $mail, $date, $id, $comment ) = @$one;
        my $rev_prefix = join '.', $prefix_top, $resno;
        my $revision   = $iRepos->get_int("${rev_prefix}.revision") + 1;
        my $prefix     = join '.', $prefix_top, $resno, $revision;
        $iRepos->set( "${prefix}.name",         $name );
        $iRepos->set( "${prefix}.mail",         $mail );
        $iRepos->set( "${prefix}.date",         $date );
        $iRepos->set( "${prefix}.id",           $id );
        $iRepos->set( "${prefix}.comment",      $comment );
        $iRepos->set( "${prefix}.modified",     $now );
        $iRepos->set( "${rev_prefix}.revision", $revision );
        $count++;
    }

    @{ $iTRevision->{__revs} } = ();
    $iTRevision->{_count} = $count;
    $iRepos->set( 'I:T:R.count', $count );
    $iRepos->save();

    return 1;
}

sub add {
    my ( $iTRevision, $resno, $content, $time ) = @_;
    $time ||= $iTRevision->{__now};
    push @{ $iTRevision->{__revs} }, [ $resno, $time, @{$content} ];
    return;
}

sub set_key {
    my ( $iTRevision, $key ) = @_;
    my $iBBS = $iTRevision->{_bbs};
    my $repos_path
        = Img0ch::Maple::create_repository_subdirs( $key, 'revision', $iBBS );
    $iTRevision->{_key} = $key;
    $iTRevision->{__prefix}
        = join( '.', 'I:T:R', $iTRevision->{_bbs}->get_name(), $key );
    $iTRevision->{_rs}
        = $iTRevision->{_kernel}->get_repos_nocache($repos_path);
    return;
}

sub get {
    my ( $iTRevision, $resno, $revision ) = @_;
    my $prefix_top = $iTRevision->{__prefix};
    my $iRepos     = $iTRevision->{_rs};

    $revision ||= $iRepos->get_int("${prefix_top}.${resno}.revision");
    my $prefix = join '.', $prefix_top, $resno, $revision;

    if ( my $name = $iRepos->get("${prefix}.name") ) {
        my $mail    = $iRepos->get("${prefix}.mail");
        my $date    = $iRepos->get("${prefix}.date");
        my $id      = $iRepos->get("${prefix}.id");
        my $comment = $iRepos->get("${prefix}.comment");
        return [ $name, $mail, $date, $id, $comment ];
    }
    else {
        return [ '', '', '', '', '' ];
    }
}

sub get_count {
    return $_[0]->{_rs}->get_int('I:T:R.count');
}

sub get_modified {
    my ( $iTRevision, $resno, $revision ) = @_;
    my $prefix_top = $iTRevision->{__prefix};
    my $iRepos     = $iTRevision->{_rs};

    $revision ||= $iRepos->get_int("${prefix_top}.${resno}.revision");
    my $prefix = join '.', $prefix_top, $resno, $revision;

    return $iRepos->get_int("${prefix}.modified");
}

sub get_revisions {
    my ( $iTRevision, $resno ) = @_;
    my $latest_rev = $iTRevision->get_latest_id($resno);
    my $ret        = [];

    for ( my $i = 1; $i <= $latest_rev; $i++ ) {
        my $res = $iTRevision->get( $resno, $i );
        $res->[4] and push @$ret, [ $i, $res ];
    }

    return $ret;
}

sub get_latest_id {
    my ( $iTRevision, $resno ) = @_;
    my $prefix_top = $iTRevision->{__prefix};
    return $iTRevision->{_rs}->get_int("${prefix_top}.${resno}.revision");
}

sub remove {
    my ( $iTRevision, $resno, $revision ) = @_;
    my $prefix_top = $iTRevision->{__prefix};
    my $iRepos     = $iTRevision->{_rs};

    $revision ||= $iRepos->get_int("${prefix_top}.${resno}.revision");
    my $prefix = join '.', $prefix_top, $resno, $revision;

    if ( $iRepos->remove("${prefix}.modified") ) {
        $iRepos->remove("${prefix}.name");
        $iRepos->remove("${prefix}.mail");
        $iRepos->remove("${prefix}.date");
        $iRepos->remove("${prefix}.id");
        $iRepos->remove("${prefix}.comment");
        $iRepos->set( 'I:T:R.count', --$iTRevision->{_count} );
        return 1;
    }
    else {
        return 0;
    }
}

1;
__END__
