#/*
# *  Copyright 2007-2009 hkrn <hikarin@users.sourceforge.jp>
# *
# *  Licensed under the Apache License, Version 2.0 (the "License");
# *  you may not use this file except in compliance with the License.
# *  You may obtain a copy of the License at
# *
# *      http://www.apache.org/licenses/LICENSE-2.0
# *
# *  Unless required by applicable law or agreed to in writing, software
# *  distributed under the License is distributed on an "AS IS" BASIS,
# *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# *  See the License for the specific language governing permissions and
# *  limitations under the License.
# */
#
# $Id: Config.pm 1826 2009-04-08 15:23:25Z hikarin $
#

package Img0ch::Config;

use strict;
use base qw(Config::Tiny);

sub load {
    my ( $iConfig, $path ) = @_;
    my $object = $iConfig->read($path) or _croak( $iConfig->errstr() );
    %{$iConfig} = %{$object};
}

sub write { _croak('config file is READ-ONLY') }

sub write_string { _croak('config file is READ-ONLY') }

sub get {
    my ( $iConfig, $key ) = @_;
    if ( my ( $section, $real_key ) = $key =~ /\A(\w+)\.([\w\*]+)\z/xms ) {
        $real_key eq '*' and return $iConfig->{$section};
        exists $iConfig->{$section} or return '';
        return $iConfig->{$section}->{$real_key} || '';
    }
    $iConfig->{_}->{$key} || '';
}

sub get_int { Img0ch::Kernel::intval( $_[0]->get( $_[1] ) ) }

sub _croak {
    require Img0ch::Maple;
    Img0ch::Maple->throw_exception(@_);
}

1;
__END__

=head1 NAME

Img0ch::Config - img0chのシステム設定を管理するクラス

=head1 SYNOPSYS

  use Img0ch::Config

  my $iConfig = Img0ch::Config->new;
  $iConfig->load('file.conf');

  # BBSPathの値を取り出す
  my $bbspath = $iConfig->get('BBSPath');
  # ResMAXの値を数値として取り出す
  my $resmax = $iConfig->get_int('ResMAX');

=head1 DESCRIPTION

1つのini形式の設定ファイルを1つのオブジェクトとするクラスです。

=head2 new

=over 4

=item Arguments

none

=item Return Value

$iConfig (Img0ch::Config itself)

=back

I<Img0ch::Config>のオブジェクトを作成します。

=head2 load

=over 4

=item Arguments

$path_to_config_file

=item Return Value

%content_of_config_file

=back

I<$path_to_config_file>から設定を読み出します。
設定ファイルの形式は必ずini形式にする必要があります。

=head2 get

=over 4

=item Arguments

$key

=item Return Value

$string_or_array_reference_value

=back

指定された値を取り出します。I<$key>の後ろにI<.*>があった場合、
指定されたセクションの全ての値を配列のリファレンスとして返します。
未定義の場合ナルストリングが返されます。

=head2 get_int

=over 4

=item Arguments

$key

=item Return Value

$integer_value

=back

指定された値を数値として取り出します。指定された値に文字列が
含まれていたり、I<.*>があった場合0を返します。

=head1 AUTHOR

hkrn E<lt>hikarin@users.sourceforge.jpE<gt>

=cut
