package org.torikiri.jexpression;

import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;
import java.util.PropertyResourceBundle;
import java.util.ResourceBundle;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.torikiri.jexpression.expression.CompositeExpression;
import org.torikiri.jexpression.expression.StringValueExpression;


public class JECompiler {

	private static final Pattern PATTERN_1 = Pattern.compile("(^[0-9]*[a-zA-z][0-9]*)([a-zA-z][0-9]*)");

	private static final Pattern PATTERN_3 = Pattern.compile("(.*\\)\\s*)(\\(.*)");

	private static final Pattern PATTERN_2 = Pattern.compile("(^[0-9]+)([a-zA-Z])");

	private static final Map operations = new HashMap();

	static {
		ResourceBundle bundle = (PropertyResourceBundle) ResourceBundle.getBundle(
				Operation.class.getName(), Locale.getDefault(), JECompiler.class.getClassLoader());
		for (Enumeration enu = bundle.getKeys(); enu.hasMoreElements(); ) {
			String key = (String) enu.nextElement();
			try {
				operations.put(key, Class.forName(bundle.getString(key)));
			} catch (ClassNotFoundException e) {
				throw new RuntimeException(e);
			}
		}
	}

	private static final Pattern OPERATIONS_PATTERN;

	static {
		StringBuffer buff = new StringBuffer("\\(|\\)");
		for (Iterator it = operations.keySet().iterator(); it.hasNext();) {
			buff.append("|").append(JEUtils.escapeRegex((String) it.next()));
		}
		OPERATIONS_PATTERN = Pattern.compile(buff.toString());
	}

	private JECompiler() {}

	public static JExpression compile(String expr) {
		expr = format(expr);
		JExpression result = createExpression(splitExpression(expr));
		if (result instanceof CompositeExpression) {
			((CompositeExpression) result).optimize();
		}
		return result;
	}

	private static JExpression createExpression(String[] partsOfExpr) {
		if (partsOfExpr.length == 1) {
			return new StringValueExpression(partsOfExpr[0]);
		} else if ("(".equals(partsOfExpr[0])) {
			int endIndex = indexOfEndKakko(partsOfExpr);
			String[] array = JEUtils.subArray(partsOfExpr, 1, endIndex - 1);
			if (endIndex == partsOfExpr.length -1) {
				return createExpression(array);
			} else {
				String[] arrayOfAfter = JEUtils.subArray(partsOfExpr, endIndex + 2);
				Operation op = createOperation(partsOfExpr[endIndex + 1]);
				return new CompositeExpression(createExpression(array), createExpression(arrayOfAfter), op);
			}
		} else {
			String[] array = JEUtils.subArray(partsOfExpr, 1);
			StringValueExpression expr = new StringValueExpression(partsOfExpr[0]);
			return createCompositeExpression(expr, array);
		}
	}

	private static JExpression createCompositeExpression(JExpression pre, String[] arrayOfExpr) {
		Operation op = createOperation(arrayOfExpr[0]);
		String[] afterArray = JEUtils.subArray(arrayOfExpr, 1);
		if (isPrimaryOperation(arrayOfExpr[0])) {
			int index;
			JExpression post;
			if ("(".equals(afterArray[0])) {
				index = indexOfEndKakko(afterArray) + 1;
				post = createExpression(JEUtils.subArray(afterArray, 1, index - 2));
			} else {
				index = 1;
				post = new StringValueExpression(afterArray[0]);
			}
			JExpression expr = new CompositeExpression(pre, post, op);
			String[] array = JEUtils.subArray(afterArray, index);
			if (array.length == 0) {
				return expr;
			}
			return createCompositeExpression(expr, array);
		} else {
			return new CompositeExpression(pre, createExpression(afterArray), op);
		}
	}

	private static boolean isPrimaryOperation(String op) {
		return "*".equals(op) || "/".equals(op) || "^".equals(op);
	}

	private static int indexOfEndKakko(String[] array) {
		int offset = 0;
		for (int i=1; i<array.length; i++) {
			String str = array[i];
			if (str.indexOf('(') >= 0) {
				offset++;
			} else if (str.indexOf(")") >= 0) {
				if (offset == 0) {
					return i;
				} else {
					offset--;
				}
			}
		}
		throw new RuntimeException("Unmatch \"()\" : " + JEUtils.toString(array));
	}

	private static String[] splitExpression(String expr) {
		Matcher m = OPERATIONS_PATTERN.matcher(expr);
		String[] result = new String[0];
		int start = 0;
		while (m.find(start)) {
			String group = m.group().trim();
			if (m.start() > start) {
				String before = expr.substring(start, m.start()).trim();
				if (before.length() == 0) {
					result = JEUtils.add(result, new String[] {group});
				} else {
					result = JEUtils.add(result, new String[] {expr.substring(start, m.start()).trim(), group});
				}
			} else {
				result = JEUtils.add(result, group);
			}
			start = m.end();
		}
		String after = expr.substring(start).trim();
		if (after.length() != 0) {
			result = JEUtils.add(result, after);
		}
		return result;
	}

	private static String format(String expr) {
		Matcher m = PATTERN_1.matcher(expr);
		int start = 0;
		StringBuffer buff = new StringBuffer();
		while (m.find(start)) {
			buff.append(m.group(1)).append("*").append(m.group(2));
			start = m.end();
		}
		String result;
		if (buff.length() > 0) {
			result = buff.toString();
		} else {
			result = expr;
		}
		Matcher m2 = PATTERN_2.matcher(result);
		if (m2.find()) {
			result = m2.group(1).concat("*").concat(m2.group(2));
		}
		Matcher m3 = PATTERN_3.matcher(result);
		if (m3.find()) {
			result = m3.group(1).concat("*").concat(m3.group(2));
		}
		return result;
	}

	private static Operation createOperation(String op) {
		Class type = (Class) operations.get(op);
		if (type == null) {
			throw new RuntimeException("No define operation. [" + op + "]");
		}
		try {
			return (Operation) ((Class) operations.get(op)).newInstance();
		} catch (InstantiationException e) {
			throw new RuntimeException(e);
		} catch (IllegalAccessException e) {
			throw new RuntimeException(e);
		}
	}
}
