/*
 * @(#)NowAddress.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.net;

import java.util.ArrayList;

import org.maachang.commons.def.BaseDef;
import org.maachang.commons.exception.BaseException;
import org.maachang.commons.exception.ExecutionException;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.exception.SettingException;
import org.maachang.commons.sys.cmd.NativeCmd;
import org.maachang.commons.thread.ExecutionThread;
import org.maachang.commons.thread.LoopThread;
import org.maachang.commons.thread.Synchronized;
import org.maachang.commons.util.CharTable;
import org.maachang.commons.util.UtilCom;


/**
 * 現在のIPアドレス情報スレッド取得.
 * <BR><BR>
 * 現在のIPアドレス情報をスレッドにより取得します.
 *  
 * @version 1.0.0 2004/09/03
 * @author  masahito suzuki
 * @since   JRcCommons 1.00
 */
class NowAddressThread extends ExecutionThread
{
    
    /**
     * 実施インターバル : デフォルト値.
     * 60000msec -> 60秒.
     */
    private static final int DEF_INTERVAL = 60000 ;
    
    /**
     * 送信インターバル : 最小値.
     * 5000msec -> 5秒.
     */
    private static final int MIN_INTERVAL = 5000 ;
    
    /**
     * 送信インターバル : 最大値.
     * 300000sec -> 300秒.
     */
    private static final int MAX_INTERVAL = 300000 ;
    
    /**
     * スレッド実施タイミング.
     */
    private static final int CHECK_TIMING = 150 ;
    
    /**
     * 実施時間ウェイト.
     */
    private static final int BEFORE_EXEC_TIME = 2500 ;
    
    
    
    /**
     * インターバル値.
     */
    private int m_interval = 0 ;
    
    /**
     * 前回実施時間.
     */
    private long m_beforeTime = 0L ;
    
    /**
     * 実施カウント.
     */
    private int m_count = 0 ;
    
    /**
     * java実施コマンド.
     */
    private String[] m_execCmd = null ;
    
    /**
     * 対象IPアドレス.
     */
    private String m_ipAddr = null ;
    
    /**
     * 対象ホスト名.
     */
    private String m_host = null ;
    
    /**
     * 複数IPアドレス.
     */
    private String[] m_ipAddrs = null ;
    
    /**
     * 即時実行フラグ.
     */
    private boolean m_flushFlg = false ;
    
    /**
     * 複数IPアドレスに対するドメイン名.
     */
    private final CharTable m_domainTable = new CharTable() ;
    
    /**
     * ループスレッド.
     */
    private final LoopThread m_thread = new LoopThread() ;
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    /**
     * コンストラクタ.
     */
    public NowAddressThread()
    {
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * <BR>
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * デフォルト条件で情報を生成します.
     */
    public final void create()
    {
        try{
            this.create( NowAddressThread.DEF_INTERVAL ) ;
        }catch( Exception t ){
        }
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * インターバル値を設定して情報を生成します.
     * <BR>
     * @param interval 取得間隔のタイミング値を設定します.
     *                 設定可能な最小値は[5000]です.<BR>
     *                 設定可能な最大値は[300000]です.<BR>
     *                 また、この設定単位はミリ秒です.
     * @exception InputException 入力例外.
     */
    public final void create( int interval )
        throws InputException
    {
        int state ;
        NowAddressCore na = null ;
        
        if(
            interval < NowAddressThread.MIN_INTERVAL ||
            interval > NowAddressThread.MAX_INTERVAL
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.clear() ;
        m_sync.create() ;
        
        na = new NowAddressCore() ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                m_interval = interval ;
                m_execCmd = BaseDef.getInstance().getExecJavaCommand(
                    null,
                    na.getClass().getName()
                ) ;
                
                m_thread.create( this ) ;
                m_thread.startThread() ;
                
            }
            
            
            // スレッド実行待ち.
            for( ;; ){
                
                synchronized( m_sync.get() ){
                    state = m_thread.getState() ;
                }
                
                if(
                    state == ExecutionThread.STATE_NOT ||
                    state == ExecutionThread.STATE_EXECUTION
                )
                {
                    if( state == ExecutionThread.STATE_NOT ){
                        this.clear() ;
                        throw new SettingException(
                            "スレッド実行に失敗しました"
                        ) ;
                    }
                    break ;
                }
                
                UtilCom.cpuCreate() ;
                
            }
            
        }catch( NullPointerException nul ){
            this.clear() ;
        }catch( InputException in ){
            this.clear() ;
            throw in ;
        }catch( SettingException st ){
            this.clear() ;
            throw new InputException( st ) ;
        }finally{
            na = null ;
        }
        
    }
    
    /**
     * 情報をクリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public final void clear()
    {
        
        m_sync.clear() ;
        m_thread.clear() ;
        m_domainTable.clear() ;
        
        m_interval = 0 ;
        m_beforeTime = 0L ;
        m_count = 0 ;
        m_execCmd = null ;
        m_ipAddr = null ;
        m_host = null ;
        m_ipAddrs = null ;
        m_flushFlg = false ;
        
    }
    
    /**
     * 即時実行.
     * <BR><BR>
     * コマンド処理をインターバルに関係なく即時実行します.
     */
    public final void flush()
    {
        try{
            synchronized( m_sync.get() ){
                m_flushFlg = true ;
            }
        }catch( Exception t ){
        }
    }
    
    /**
     * インターバル時間の設定.
     * <BR><BR>
     * チェックタイミングのインターバル時間を設定します.
     * <BR>
     * @param interval 取得間隔のタイミング値を設定します.
     *                 設定可能な最小値は[5000]です.<BR>
     *                 設定可能な最大値は[300000]です.<BR>
     *                 また、この設定単位はミリ秒です.
     * @exception InputException 入力例外.
     */
    public final void setInterval( int interval )
        throws InputException
    {
        if(
            interval < NowAddressThread.MIN_INTERVAL ||
            interval > NowAddressThread.MAX_INTERVAL
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            synchronized( m_sync.get() ){
                
                m_interval = interval ;
                
            }
            
        }catch( NullPointerException nul ){
        }
        
    }
    
    /**
     * 設定インターバル時間を取得.
     * <BR><BR>
     * 設定されているインターバル時間を取得します.
     * <BR>
     * @return int 設定されているインターバル時間が返されます.<BR>
     *             また、設定されている単位はミリ秒です.
     */
    public final int getInterval()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_interval ;
            }
        }catch( Exception t ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * 現在有効情報の取得.
     * <BR><BR>
     * 現在有効となっている情報を取得します.
     * <BR>
     * @oaram code 取得対象のコード情報を設定します.
     * @return String 現在有効となっている情報を取得します.
     */
    public final Object getNow( int code )
    {
        int cnt ;
        Object ret = null ;
        
        try{
            
            for( ;; ){
                
                switch( code ){
                    case NowAddressCore.CODE_NOW_IPADDR :
                        
                        synchronized( m_sync.get() ){
                            cnt = m_count ;
                            ret = m_ipAddr ;
                        }
                        if( ret == null ){
                            ret = NetDef.NOT_ADDR ;
                        }
                        break ;
                        
                    case NowAddressCore.CODE_NOW_DOMAIN :
                        
                        synchronized( m_sync.get() ){
                            cnt = m_count ;
                            ret = m_host ;
                        }
                        if( ret == null ){
                            ret = NetDef.NOT_ADDR ;
                        }
                        break ;
                        
                    case NowAddressCore.CODE_ALL_IPADDR :
                        
                        synchronized( m_sync.get() ){
                            cnt = m_count ;
                            ret = m_ipAddrs ;
                        }
                        if( ret == null ){
                            ret = new String[ 1 ] ;
                            ( ( String[] )ret )[ 0 ] = NetDef.NOT_ADDR ;
                        }
                        break ;
                        
                    case NowAddressCore.CODE_ALL_DOMAIN :
                        
                        synchronized( m_sync.get() ){
                            cnt = m_count ;
                            ret = m_domainTable ;
                        }
                        break ;
                        
                    default :
                        
                        cnt = 1 ;
                        ret = NetDef.NOT_ADDR ;
                        break ;
                }
                
                if( this.isThread() == false || cnt != 0 ){
                    break ;
                }
                
                UtilCom.cpuCreate() ;
            }
            
        }catch( Exception t ){
            
            switch( code ){
                case NowAddressCore.CODE_NOW_IPADDR :
                case NowAddressCore.CODE_NOW_DOMAIN :
                    
                    ret = NetDef.NOT_ADDR ;
                    break ;
                    
                case NowAddressCore.CODE_ALL_IPADDR :
                    
                    ret = new String[ 1 ] ;
                    ( ( String[] )ret )[ 0 ] = NetDef.NOT_ADDR ;
                    break ;
                    
                case NowAddressCore.CODE_ALL_DOMAIN :
                    
                    ret = null ;
                    break ;
                    
                default :
                    
                    ret = NetDef.NOT_ADDR ;
                    break ;
            }
        }
        
        return ret ;
    }
    
    /**
     * 動作チェック.
     * <BR><BR>
     * このオブジェクトのスレッドが動作しているかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、スレッドは動作中です.<BR>
     *                 [false]が返された場合、スレッドは停止中です.
     */
    public final boolean isThread()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_thread.isThread() ;
            }
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 情報生成フラグ.
     * <BR><BR>
     * オブジェクト情報が生成されているかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.
     */
    public final boolean isCreate()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = true ;
            }
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 実行初期化処理をサポートします.
     * <BR><BR>
     * 実行初期化処理をサポートします.<BR>
     * この処理は、スレッド処理が開始された時に呼び出されます.
     * <BR>
     * @param obj 実行開始時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void init( Object obj )
        throws ExecutionException
    {
        try{
            
            synchronized( m_sync.get() ){
                
                if( m_execCmd == null ){
                    throw new ExecutionException(
                        "スレッド実施に対する条件が不足しています",
                        ExecutionException.LEVEL_STOP
                    ) ;
                }
                
            }
            
        }catch( NullPointerException nul ){
            throw new ExecutionException( nul,ExecutionException.LEVEL_STOP ) ;
        }catch( ExecutionException ee ){
            throw ee ;
        }
    }
    
    /**
     * 実行終了化処理をサポートします.
     * <BR><BR>
     * 実行終了化処理をサポートします.<BR>
     * この処理は、スレッド処理が終了された時に呼び出されます.
     * <BR>
     * @param obj 実行終了時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void exit( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * ストップ処理をサポートします。
     * <BR><BR>
     * ストップ処理をサポートします。<BR>
     * この処理は、スレッドでのストップ処理に対して呼び出し実行されます.
     * <BR>
     * @param obj ストップ時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void stop( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行処理をサポートします。
     * <BR><BR>
     * 実行処理をサポートします。<BR>
     * この処理は、スレッドでの実行処理に対して呼び出し実行されます.
     * <BR>
     * @param obj 実行時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void execution( Object obj )
        throws ExecutionException
    {
        int i ;
        int len ;
        int interval ;
        long timeout ;
        long beforeTime ;
        boolean flushFlg ;
        
        String ipAddr = null ;
        String hostName = null ;
        String[] addrs = null ;
        String[] cmd = null ;
        String[] retCmd = null ;
        ArrayList buf = null ;
        CharTable tbl = null ;
        
        NativeCmd execCmd = null ;
        ArrayList retCmdList = null ;
        ArrayList domainNames = null ;
        
        try{
            
            UtilCom.idleTime() ;
            
            synchronized( m_sync.get() ){
                
                interval = m_interval ;
                timeout = m_beforeTime ;
                cmd = m_execCmd ;
                flushFlg = m_flushFlg ;
                m_domainTable.clear() ;
                tbl = m_domainTable ;
                
            }
            
            beforeTime = timeout ;
            timeout += ( long )( interval & 0x00000000ffffffffL ) ;
            
            ////////////////////////////
            // 条件実施が検知された場合.
            // 即時実行フラグがONの場合.
            ////////////////////////////
            if( flushFlg == true || System.currentTimeMillis() >= timeout ){
                
                // 前回実施間隔が短い場合.
                if( System.currentTimeMillis() - beforeTime <= NowAddressThread.BEFORE_EXEC_TIME ){
                    
                    // 処理しない.
                    synchronized( m_sync.get() ){
                        m_flushFlg = false ;
                    }
                    
                    UtilCom.sleep( NowAddressThread.CHECK_TIMING ) ;
                    return ;
                    
                }
                
                // 対象の媒体名を更新.
                NetConfig.getInstance().flush() ;
                
                // コマンド生成.
                execCmd = new NativeCmd() ;
                
                len = cmd.length ;
                
                if( len > 0 ){
                    
                    // コマンド生成.
                    buf = new ArrayList() ;
                    for( i = 0 ; i < len ; i ++ ){
                        buf.add( cmd[ i ] ) ;
                    }
                    buf.add( NowAddressCore.ALL_DATA ) ;
                    
                    // コマンド実行.
                    retCmd = execCmd.execution( true,buf ) ;
                    
                    // 結果情報解析.
                    retCmdList = UtilCom.pauseString( retCmd[ 0 ],NowAddressCore.ALL_DATA_PAUSE ) ;
                    
                    // 解析対象情報が存在しない場合.
                    if( retCmdList == null || ( len = retCmdList.size() ) != 3 ){
                        
                        // 条件をデフォルトでセット.
                        ipAddr = NetDef.NOT_ADDR ;
                        hostName = NetDef.NOT_ADDR ;
                        addrs = new String[ 1 ] ;
                        addrs[ 0 ] = NetDef.NOT_ADDR ;
                        
                    }
                    else{
                        
                        // 取得情報をセット.
                        ipAddr = ( String )retCmdList.get( 0 ) ;
                        hostName = ( String )retCmdList.get( 1 ) ;
                        retCmd[ 0 ] = ( String )retCmdList.get( 2 ) ;
                        
                        // IPアドレス群を解析.
                        retCmdList.clear() ;
                        retCmdList = null ;
                        retCmdList = UtilCom.pauseString( retCmd[ 0 ],NowAddressCore.ALL_PAUSE ) ;
                        
                        // 解析結果が存在しない場合.
                        if( retCmdList == null ){
                            
                            // デフォルトをセット.
                            addrs = new String[ 1 ] ;
                            addrs[ 0 ] = NetDef.NOT_ADDR ;
                            
                        }
                        else{
                            
                            // 解析結果をセット.
                            len = retCmdList.size() ;
                            addrs = new String[ len ] ;
                            for( i = 0 ; i < len ; i ++ ){
                                
                                if( domainNames != null ){
                                    domainNames.clear() ;
                                }
                                domainNames = null ;
                                domainNames = UtilCom.pauseString(
                                    ( String )retCmdList.get( i ),NowAddressCore.DOMAIN_PAUSE
                                ) ;
                                
                                // 対象アドレスを取得.
                                addrs[ i ] = ( ( String )domainNames.get( 0 ) ).trim() ;
                                
                                // ドメイン名が有効である場合.
                                if( domainNames.size() == 2 ){
                                    
                                    tbl.add( addrs[ i ],( ( String )domainNames.get( 1 ) ).trim() ) ;
                                    
                                }
                                
                            }
                            
                        }
                        
                    }
                }
                else{
                    
                    // 条件をデフォルトでセット.
                    ipAddr = NetDef.NOT_ADDR ;
                    hostName = NetDef.NOT_ADDR ;
                    addrs = new String[ 1 ] ;
                    addrs[ 0 ] = NetDef.NOT_ADDR ;
                    
                }
                
                // 取得条件をセット.
                synchronized( m_sync.get() ){
                    
                    m_beforeTime = System.currentTimeMillis() ;
                    m_count = ( m_count > 999 ) ? 1 : m_count + 1 ;
                    m_ipAddr = ipAddr ;
                    m_host = hostName ;
                    m_ipAddrs = addrs ;
                    m_flushFlg = false ;
                    
                }
            }
            ///////////////////////////////////
            // 条件実施が検知されていない場合.
            ///////////////////////////////////
            else{
                
                UtilCom.sleep( NowAddressThread.CHECK_TIMING ) ;
                
            }
            
        }catch( ExecutionException ee ){
            throw ee ;
        }catch( BaseException be ){
            throw new ExecutionException( be,ExecutionException.LEVEL_STOP ) ;
        }catch( Exception t ){
            throw new ExecutionException( t,ExecutionException.LEVEL_STOP ) ;
        }finally{
            
            if( execCmd != null ){
                execCmd.clear() ;
            }
            if( retCmdList != null ){
                retCmdList.clear() ;
            }
            if( domainNames != null ){
                domainNames.clear() ;
            }
            if( buf != null ){
                buf.clear() ;
            }
            
            cmd = null ;
            ipAddr = null ;
            hostName = null ;
            addrs = null ;
            retCmd = null ;
            
            execCmd = null ;
            retCmdList = null ;
            buf = null ;
            tbl = null ;
            
        }
        
    }
    
    

}

