/*
 * @(#)MgcSvRcvThread.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.net.mgc ;

import java.net.InetAddress;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.commons.exception.ExecutionException;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.net.BaseMultiCast;
import org.maachang.commons.net.ConnectAddress;
import org.maachang.commons.net.ConnectTimeoutException;
import org.maachang.commons.thread.ExecutionThread;
import org.maachang.commons.thread.LoopThread;
import org.maachang.commons.thread.Synchronized;
import org.maachang.commons.util.UtilCom;


/**
 * MGC-Server受信スレッド.
 *  
 * @version 1.0.0 2005/07/29
 * @author  masahito suzuki
 * @since   JRcCommons 1.00
 */
class MgcSvRcvThread extends ExecutionThread
{
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( MgcSvRcvThread.class ) ;
    
    /**
     * 再送時間 : デフォルト値.
     * 150sec.
     */
    private static final int DEF_RESEND = MgcSvThread.DEF_RESEND ;
    
    /**
     * 再送時間 : 最小値.
     * 30sec.
     */
    private static final int MIN_RESEND = MgcSvThread.MIN_RESEND ;
    
    /**
     * デフォルト受信時間.
     */
    private static final int DEF_RCVTIME = 250 ;
    
    /**
     * 電文送信回数 : 最大値.
     */
    private static final int MAX_SEND = 3 ;
    
    
    
    /**
     * このマシンを示す要素.
     */
    private MgcValueImple m_thisValue = null ;
    
    /**
     * マルチキャストオブジェクト.
     */
    private BaseMultiCast m_mcast = null ;
    
    /**
     * サーバテーブル.
     */
    private MgcTable m_svTable = null ;
    
    /**
     * 再送タイミング.
     */
    private int m_resend = -1 ;
    
    /**
     * 暗号表(256).
     */
    private byte[] m_codeTable = null ;
    
    /**
     * 受信アドレス格納オブジェクト.
     */
    private ConnectAddress m_addr = null ;
    
    /**
     * 前回送信時間.
     */
    private long m_befTime = 0L ;
    
    /**
     * マルチキャストバインドアドレス.
     */
    private InetAddress m_bindAddr = null ;
    
    
    
    /**
     * ループスレッド.
     */
    private final LoopThread m_thread = new LoopThread() ;
    
    /**
     * 同期処理.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    
    /**
     * コンストラクタ.
     */
    private MgcSvRcvThread(){}
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象の条件を設定してオブジェクトを生成します.
     * <BR>
     * @param thisValue このマシンを示す要素を設定します.
     * @param mcast マルチキャストオブジェクトを設定します.
     * @param sv 対象のMgcサーバテーブルを設定します.
     * @param table 対象の暗号表(256byte)を設定します.<BR>
     *              設定可能なサイズは[256]byteです.<BR>
     *              また、[null]を設定した場合、デフォルト条件になります.
     * @exception InputException 入力例外.
     */
    public MgcSvRcvThread( MgcValueImple thisValue,BaseMultiCast mcast,MgcTable sv,byte[] table )
        throws InputException
    {
        this( thisValue,mcast,sv,table,DEF_RESEND ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象の条件を設定してオブジェクトを生成します.
     * <BR>
     * @param thisValue このマシンを示す要素を設定します.
     * @param mcast マルチキャストオブジェクトを設定します.
     * @param sv 対象のMgcサーバテーブルを設定します.
     * @param table 対象の暗号表(256byte)を設定します.<BR>
     *              設定可能なサイズは[256]byteです.<BR>
     *              また、[null]を設定した場合、デフォルト条件になります.
     * @param resend 再送タイムを設定します.<BR>
     *               設定可能な最小値は30秒です.<BR>
     *               設定単位はミリ秒です.
     * @exception InputException 入力例外.
     */
    public MgcSvRcvThread( MgcValueImple thisValue,BaseMultiCast mcast,MgcTable sv,byte[] table,int resend )
        throws InputException
    {
        
        if(
            thisValue == null || thisValue.isUse() == false ||
            mcast == null || mcast.isOpen() == false || mcast.isGroup() == false ||
            sv == null
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        resend = ( resend <= MIN_RESEND ) ? MIN_RESEND : resend ;
        table = ( table == null || table.length != MgcCommon.MAX_RAND_CODE ) ?
            null : table ;
        
        m_sync.create() ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                // 基本条件設定.
                m_thisValue = thisValue ;
                m_mcast = mcast ;
                m_svTable = sv ;
                m_codeTable = table ;
                m_resend = resend ;
                
                m_befTime = 0L ;
                m_addr = new ConnectAddress() ;
                
                if(
                    ( m_bindAddr = mcast.getInterface() ) == null ||
                    m_bindAddr.getHostAddress().equals( "0.0.0.0" ) == true ||
                    m_bindAddr.getHostAddress().equals( "127.0.0.1" ) == true
                )
                {
                    m_bindAddr = InetAddress.getLocalHost() ;
                }
                
                // スレッドを生成.
                m_thread.create( this ) ;
                m_thread.startThread() ;
                
            }
            
        }catch( Exception e ){
            this.destroy() ;
            throw new InputException( e ) ;
        }
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.destroy() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクト情報を破棄します.
     */
    public final void destroy()
    {
        try{
            synchronized( m_sync.get() ){
                
                m_thread.clear() ;
                
                m_thisValue = null ;
                m_mcast = null ;
                m_svTable = null ;
                m_befTime = 0L ;
                m_addr = null ;
                m_bindAddr = null ;
                m_codeTable = null ;
                
            }
        }catch( Exception e ){
        }
        
        m_sync.clear() ;
        
        m_thisValue = null ;
        m_mcast = null ;
        m_svTable = null ;
        m_befTime = 0L ;
        m_addr = null ;
        m_bindAddr = null ;
        m_codeTable = null ;
        
    }
    
    /**
     * 終了通知電文を送信.
     * <BR><BR>
     * 終了通知電文を送信します.
     * <BR>
     * @param count 送信する回数を設定します.
     */
    public final void sendMgcServerEnd( int count )
    {
        try{
            synchronized( m_sync.get() ){
                // シャットダウン電文を送信.
                this.send(
                    false,count,MgcCommon.HEADER_TYPE_END
                ) ;
            }
        }catch( Exception e ){
        }
    }
    
    /**
     * オブジェクト利用可能チェック.
     * <BR><BR>
     * オブジェクト情報が利用可能かチェックします.
     * <BR>
     * @return boolean オブジェクトが利用可能かチェックします.
     */
    public final boolean isUse()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_thread.isThread() ;
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    
    
    /**
     * 実行初期化処理をサポートします.
     * <BR><BR>
     * 実行初期化処理をサポートします.<BR>
     * この処理は、スレッド処理が開始された時に呼び出されます.
     * <BR>
     * @param obj 実行開始時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void init( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行終了化処理をサポートします.
     * <BR><BR>
     * 実行終了化処理をサポートします.<BR>
     * この処理は、スレッド処理が終了された時に呼び出されます.
     * <BR>
     * @param obj 実行終了時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void exit( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * ストップ処理をサポートします。
     * <BR><BR>
     * ストップ処理をサポートします。<BR>
     * この処理は、スレッドでのストップ処理に対して呼び出し実行されます.
     * <BR>
     * @param obj ストップ時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void stop( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行処理をサポートします。
     * <BR><BR>
     * 実行処理をサポートします。<BR>
     * この処理は、スレッドでの実行処理に対して呼び出し実行されます.
     * <BR>
     * @param obj 実行時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void execution( Object obj )
        throws ExecutionException
    {
        int i ;
        int cnt ;
        long befTime ;
        
        MgcValueImple thisValue = null ;
        BaseMultiCast mcast = null ;
        MgcTable svTable = null ;
        ConnectAddress addr = null ;
        InetAddress bindAddr = null ;
        byte[] rcv = null ;
        
        try{
            
            synchronized( m_sync.get() ){
                thisValue = m_thisValue ;
                mcast = m_mcast ;
                svTable = m_svTable ;
                
                befTime = m_resend + m_befTime ;
                addr = m_addr ;
                bindAddr = m_bindAddr ;
            }
            
            ///////////////////////////
            // マルチキャストから受信.
            ///////////////////////////
            try{
                // 受信処理.
                rcv = mcast.receive( addr,DEF_RCVTIME ) ;
                
            }catch( ConnectTimeoutException ct ){
                rcv = null ;
            }
            
            ////////////////////////////
            // 受信情報が存在する場合.
            ////////////////////////////
            if( rcv != null ){
                
                //<><><><><><><><><><><><><><><><><><><><><><><><><><>
                // 受信された電文タイプが「クライアント送信」の場合.
                //<><><><><><><><><><><><><><><><><><><><><><><><><><>
                if( MgcCommon.isMgcClient( rcv ) == true ){
                    
                    // クライアントから要求された更新回数を取得.
                    cnt = MgcCommon.getMgcClient( rcv ) ;
                    
                    // リトライ送信を要求している場合.
                    if( cnt > 0 ){
                        
                        // 電文を送信.
                        this.send( true,cnt,MgcCommon.HEADER_TYPE_DEFAULT ) ;
                        
                    }
                    
                }
                //<><><><><><><><><><><><><><><><><><><><><><><><>
                // 受信先がマルチキャストアドレスと同じで無い場合.
                // 受信された電文タイプが「サーバ送信」の場合.
                //<><><><><><><><><><><><><><><><><><><><><><><><>
                else if(
                    bindAddr.equals( addr.getAddress() ) == false &&
                    MgcCommon.isMgcServer( rcv ) == true
                )
                {
                    
                    // 戻り要求電文が受信された場合.
                    if( MgcCommon.isMgcServerReturn( rcv ) == true ){
                        
                        // 通常電文を送信.
                        this.send( false,1,MgcCommon.HEADER_TYPE_DEFAULT ) ;
                        
                    }
                    
                    // サーバテーブルに登録.
                    svTable.set( rcv ) ;
                    
                }
                
                rcv = null ;
                
            }
            ///////////////////////////////////
            // 送信タイマーが条件を満たす場合.
            ///////////////////////////////////
            else if( befTime <= System.currentTimeMillis() ){
                
                // 戻り要求電文を送信.
                this.send( true,1,MgcCommon.HEADER_TYPE_RETURN ) ;
                
            }
            /////////////////////////////
            // 処理条件が存在しない場合.
            /////////////////////////////
            else{
                
                // 一定期間待機.
                UtilCom.idleTime() ;
                
            }
            
        }catch( NullPointerException nul ){
            throw new ExecutionException(
                nul,ExecutionException.LEVEL_STOP
            ) ;
        }catch( ExecutionException ee ){
            throw ee ;
        }catch( InputException in ){
            LOG.error( "エラーが発生しました",in ) ;
        }catch( Exception e ){
            e.printStackTrace() ;
        }finally{
            thisValue = null ;
            mcast = null ;
            svTable = null ;
            addr = null ;
            rcv = null ;
        }
        
    }
    
    /**
     * このマシンの電文を送信.
     */
    private final void send( boolean update,int cnt,int mode )
    {
        int i ;
        BaseMultiCast mcast = null ;
        MgcValueImple val = null ;
        byte[] table = null ;
        byte[] bin = null ;
        
        cnt = ( cnt <= 0 ) ? 1 : cnt ;
        cnt = ( cnt >= MAX_SEND ) ? MAX_SEND : cnt ;
        
        try{
            
            synchronized( m_sync.get() ){
                mcast = m_mcast ;
                val = m_thisValue ;
                table = m_codeTable ;
            }
            
            // このマシンを表す電文を更新.
            synchronized( val ){
                bin = MgcCommon.createMgcServer( mode,val,table ) ;
            }
            
            // 指定回数送信.
            for( i = 0 ; i < cnt ; i ++ ){
                mcast.send( bin ) ;
            }
            
            // 前回送信時間を更新.
            if( update == true ){
                synchronized( m_sync.get() ){
                    m_befTime = System.currentTimeMillis() ;
                }
            }
            
        }catch( Exception e ){
            LOG.error( "エラーが発生しました",e ) ;
        }finally{
            mcast = null ;
            val = null ;
            table = null ;
            bin = null ;
        }
        
    }
    
}

