/*
 * @(#)BinResourceOutputStream.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.resource ;

import java.io.IOException;
import java.io.OutputStream;

import org.maachang.commons.exception.InputException;


/**
 * バイナリリソースOutputStream.
 * <BR><BR>
 * [org.maachang.commons.resource.BinResource]をOutputStreamで利用可能にしたオブジェクトです.
 *
 * @version 1.00, 2005/04/02
 * @author  Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class BinResourceOutputStream extends OutputStream
{
    
    /**
     * 増加単位.
     */
    private static final int ADD_SIZE = 1024 ;
    
    /**
     * バイナリリソース.
     */
    private BinResource m_resource = null ;
    
    /**
     * 書き込み位置.
     */
    private int m_seek = 0 ;
    
    /**
     * クローズモード.
     */
    private boolean m_closeMode = false ;
    
    /**
     * コンストラクタ.
     */
    private BinResourceOutputStream()
    {
        super() ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象のオブジェクトを設定します.
     * <BR>
     * @param resource 書き込み対象のバイナリリソースを設定します.
     * @exception InputException 入力例外.
     */
    public BinResourceOutputStream( BinResource resource )
        throws InputException
    {
        this( false,resource,0 ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象のオブジェクトを設定します.
     * <BR>
     * @param closeMode クローズ処理時に、渡されるリソースをクローズするか設定します.<BR>
     *                  [true]の場合、クローズ時に破棄されます.<BR>
     *                  [false]の場合、クローズ時に継続されます.
     * @param resource 書き込み対象のバイナリリソースを設定します.
     * @exception InputException 入力例外.
     */
    public BinResourceOutputStream( boolean closeMode,BinResource resource )
        throws InputException
    {
        this( closeMode,resource,0 ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象のオブジェクトを設定します.
     * <BR>
     * @param closeMode クローズ処理時に、渡されるリソースをクローズするか設定します.<BR>
     *                  [true]の場合、クローズ時に破棄されます.<BR>
     *                  [false]の場合、クローズ時に継続されます.
     * @param resource 書き込み対象のバイナリリソースを設定します.
     * @param offset 書き込み位置を設定します.
     * @exception InputException 入力例外.
     */
    public BinResourceOutputStream( boolean closeMode,BinResource resource,int offset )
        throws InputException
    {
        super() ;
        
        if( resource == null || resource.isUse() == false ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        m_resource = resource ;
        m_closeMode = closeMode ;
        m_seek = ( offset <= 0 ) ? 0 : offset ;
    }
    
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.close() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報クローズ.
     * <BR><BR>
     * 情報をクローズします.
     */
    public final void close()
    {
        if( m_closeMode == true ){
            if( m_resource != null ){
                m_resource.clear() ;
            }
        }
        m_resource = null ;
        m_seek = 0 ;
        m_closeMode = false ;
    }
    
    /**
     * 情報を強制的に書き込み.
     * <BR><BR>
     * 情報を強制的に書き込みます.
     * <BR>
     * throws IOException IO例外.
     */
    public final void flush() throws IOException
    {
        super.flush() ;
    }
    
    /**
     * 情報を書き込み.
     * <BR><BR>
     * 情報を書き込みます.
     * <BR>
     * @param b 書き込み対象の情報を設定します.<BR>
     *          また書き込み有効な条件は下位8ビットです.
     * @exception IOException IO例外.
     */
    public final void write( int b ) throws IOException
    {
        try{
            m_resource.set( m_seek,( byte )( b & 0x000000ff ) ) ;
            m_seek ++ ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報を書き込み.
     * <BR><BR>
     * 情報を書き込みます.
     * <BR>
     * @param b 書き込み対象の情報を設定します.
     * @exception IOException IO例外.
     */
    public final void write( byte[] b )
        throws IOException
    {
        this.write( b,0,b.length ) ;
    }
    
    /**
     * 情報を書き込み.
     * <BR><BR>
     * 情報を書き込みます.
     * <BR>
     * @param binary 書き込み対象の情報を設定します.
     * @param offset 書き込み対象のオフセット値を設定します.
     * @param length 書き込み対象のデータ長を設定します.
     * @exception IOException IO例外.
     */
    public final void write( byte[] binary,int offset,int length )
        throws IOException
    {
        
        if(
            (
                offset | length | ( offset + length ) |
                ( binary.length - ( offset + length ) )
            ) < 0
        )
        {
            throw new IndexOutOfBoundsException( "引数は不正です" ) ;
        }
        else if ( length == 0 ){
            return ;
        }
        
        try{
            m_resource.setBinary( m_seek,binary,offset,length ) ;
            m_seek += length ;
        }catch( Exception e ){
            throw new IOException( e.getMessage() ) ;
        }
    }
    
    /**
     * 格納バイナリリソースを取得.
     * <BR><BR>
     * 格納されているバイナリリソースを取得します.
     * <BR>
     * @return BinResource 格納されているバイナリリソースが返されます.
     */
    public final BinResource getBinResource()
    {
        return m_resource ;
    }
}

