/*
 * @(#)ThreadPool.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.thread;

import org.maachang.commons.exception.AccessException;
import org.maachang.commons.exception.InputException;

/**
 * スレッドプール.
 * <BR><BR>
 * スレッドを一定量管理します.
 *
 * @version     1.00, 2004/10/06
 * @author      Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class ThreadPool
{
    
    /**
     * スレッドプール管理数 : デフォルト値.
     */
    private static final int DEF_MANLENGTH = 15 ;
    
    /**
     * スレッドプール管理数 : 最小値.
     */
    private static final int MIN_MANLENGTH = 5 ;
    
    /**
     * スレッドプール管理数 : 最大値.
     */
    private static final int MAX_MANLENGTH = 499 ;
    
    
    /**
     * シーケンスID.
     */
    private int m_seq = 0 ;
    
    /**
     * スレッドプール管理.
     */
    private ObjectThread[] m_manager = null ;
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    
    /**
     * コンストラクタ.
     */
    public ThreadPool()
    {
        this.create() ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * スレッドプール管理を生成します.
     * <BR>
     * @param length スレッドプール管理サイズを設定します.<BR>
     *               設定可能な最小値は[5]です.<BR>
     *               設定可能な最大値は[499]です.
     * @exception InputException 入力例外.
     */
    public ThreadPool( int length )
        throws InputException
    {
        this.create( length ) ;
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * スレッドプール管理生成.
     * <BR><BR>
     * スレッドプール管理を生成します.
     */
    public final void create()
    {
        try{
            this.create( ThreadPool.DEF_MANLENGTH ) ;
        }catch( Exception t ){
            this.clear() ;
        }
    }
    
    /**
     * スレッドプール管理生成.
     * <BR><BR>
     * スレッドプール管理を生成します.
     * <BR>
     * @param length スレッドプール管理サイズを設定します.<BR>
     *               設定可能な最小値は[5]です.<BR>
     *               設定可能な最大値は[499]です.
     * @exception InputException 入力例外.
     */
    public final void create( int length )
        throws InputException
    {
        int i ;
        
        ObjectThread[] man = null ;
        
        if(
            length < ThreadPool.MIN_MANLENGTH ||
            length > ThreadPool.MAX_MANLENGTH
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.clear() ;
        m_sync.create() ;
        
        try{
            
            man = new ObjectThread[ length ] ;
            for( i = 0 ; i < length ; i ++ ){
                man[ i ] = new ObjectThread() ;
            }
            
            synchronized( m_sync.get() ){
                m_manager = man ;
            }
            
        }catch( Exception t ){
            this.clear() ;
        }finally{
            man = null ;
        }
        
    }
    
    /**
     * スレッドプールクリア.
     * <BR><BR>
     * スレッドプールをクリアします.
     */
    public final void clear()
    {
        int i ;
        int len ;
        
        ObjectThread[] man = null ;
        
        m_sync.clear() ;
        
        try{
            
            man = m_manager ;
            len = man.length ;
            
            for( i = 0 ; i < len ; i ++ ){
                
                try{
                    man[ i ].clear() ;
                }catch( Exception tt ){
                }finally{
                    man[ i ] = null ;
                }
                
            }
            
        }catch( Exception t ){
        }finally{
            m_manager = null ;
            man = null ;
        }
        
        m_seq = 0 ;
        
    }
    
    /**
     * スレッド実行.
     * <BR><BR>
     * 対象のスレッド処理を実施します.
     * <BR>
     * @param exec スレッド実施対象オブジェクトを設定します.
     * @param value スレッド実施時のパラメータを設定します.
     * @return boolean スレッド処理結果が返されます.<BR>
     *                 [true]が返された場合、対象スレッドは実施されました.<BR>
     *                 [false]が返された場合、対象スレッドは実施されませんでした.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final boolean execThread( ExecutionThread exec,Object value )
        throws InputException,AccessException
    {
        int i ;
        int len ;
        int seq ;
        boolean ret ;
        
        ObjectThread[] man = null ;
        
        if( exec == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        ret = false ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                if( ( man = m_manager ) == null ){
                    this.clear() ;
                    return false ;
                }
                
                len = man.length ;
                
            }
            
            for( i = 0,ret = false ; i < len ; i ++ ){
                
                synchronized( m_sync.get() ){
                    
                    seq = m_seq ;
                    m_seq = ( man.length <= seq ) ? 0 : seq + 1 ;
                    
                    if( man[ seq ].isExec() == true ){
                        continue ;
                    }
                    else if( man[ seq ].isCreate() == false ){
                        man[ seq ].create() ;
                    }
                    
                    man[ seq ].execThread( exec,value ) ;
                    ret = true ;
                    break ;
                    
                }
                
            }
            
        }catch( NullPointerException nul ){
            this.clear() ;
        }catch( Exception t ){
            this.clear() ;
            throw new AccessException( t ) ;
        }finally{
            man = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * スレッドプールサイズを取得.
     * <BR><BR>
     * スレッドプールサイズを取得します.
     * <BR>
     * @return int スレッドプールサイズが返されます.
     */
    public final int getLength()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_manager.length ;
            }
        }catch( Exception t ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * 利用可能なスレッドプールサイズを取得.
     * <BR><BR>
     * 現在利用可能なスレッドプールサイズを取得します.
     * <BR>
     * @return int 利用可能なスレッドプールサイズが返されます.
     */
    public final int getUseLength()
    {
        int i ;
        int len ;
        int ret ;
        
        ObjectThread[] man = null ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                man = m_manager ;
                len = man.length ;
                
                for( i = 0,ret = 0 ; i < len ; i ++ ){
                    
                    if( man[ i ].isExec() == false ){
                        ret ++ ;
                    }
                    
                }
                
            }
            
        }catch( Exception t ){
            ret = 0 ;
        }finally{
            man = null ;
        }
        
        return ret ;
    }
    
    
    
}

