/*
 * @(#)SequenceIDEx.java
 *
 * Copyright (c) 2004 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.util;

import org.maachang.commons.exception.InputException;
import org.maachang.commons.serialize.InitSerialize;
import org.maachang.commons.serialize.SerializeUtil;
import org.maachang.commons.thread.Synchronized;


/**
 * ID発番オブジェクト.
 * <BR><BR>
 * IDを発番するオブジェクトです.<BR>
 * このオブジェクトは、[org.maachang.commons.util.SequenceID]より、大きな値のIDを管理
 * する場合に利用することを推奨します.<BR>
 * また、SequenceIDオブジェクトとSequenceIDExオブジェクトの単位の幅は以下の
 * ようになっています.<BR>
 * <PRE>
 *   SequenceID     :   0   -   2147483647
 *   SequenceIDEx   :   0   -   9223372036854775807
 * </PRE>
 * また、速度を優先する場合は SequenceIDオブジェクトを利用します.<BR>
 * また、情報桁幅を多く取りたい場合などはSequenceIDExオブジェクトを利用します.
 *
 * @version     1.00, 2004/07/06
 * @author      Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class SequenceIDEx implements InitSerialize
{
    
    static {
        serialVersionUID = SerializeUtil.serialVersionUID(
            SequenceIDEx.class.getName()
        ) ;
    }
    
    /**
     * シリアライズUID.
     */
    private static final long serialVersionUID ;
    
    /**
     * 取得IDなし.
     */
    public static final long NOT_ID = 0x8000000000000000L ;
    
    /**
     * MAX-ID.
     */
    private long m_maxID = 0L ;
    
    /**
     * 開始値.
     */
    private long m_startID = 0L ;
    
    /**
     * 管理ID.
     */
    private long m_id = 0L ;
    
    /**
     * 折り返しタイムアウト値.
     */
    private long m_returnTimeout = 0L ;
    
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 任意の最大値を用いて情報を生成します.
     */
    public SequenceIDEx()
    {
        this.create() ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 情報を生成します.
     * <BR>
     * @param start 対象の開始値を設定します.
     * @param max 管理を行うIDの幅(MAX値)を設定します.
     */
    public SequenceIDEx( long start,long max )
    {
        
        try{
            this.create( start,max ) ;
        }catch( InputException in ){
        }
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        try{
            this.clear() ;
        }catch( Exception t ){
        }
    }
    
    /**
     * 初期化処理.
     * <BR><BR>
     * 初期化処理を行うメソッドです.<BR>
     * 基本的には、このインターフェイスを継承したオブジェクトは、
     * 初期化に必要な処理を実装することで、
     * [org.maachang.commons.serialize.SerializeCom.getSerialize()]からの、
     * オブジェクトロードの際に、このメソッドを呼び出してくれるので、
     * ロード後のオブジェクトを、円滑に利用する事が出来ます.
     */
    public void initSerializable()
    {
        m_sync.create() ;
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 情報を生成します.
     */
    public void create()
    {
        try{
            this.create( 0,Long.MAX_VALUE ) ;
        }catch( InputException in ){
        }
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 情報を生成します.
     * <BR>
     * @param start 対象の開始値を設定します.
     * @param max 管理を行うIDの幅(MAX値)を設定します.
     * @exception InputException 入力例外.
     */
    public final void create( long start,long max )
        throws InputException
    {
        
        if( start < 0L || max <= 1L || start >= ( max - 1L ) || max > Long.MAX_VALUE ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.clear() ;
        m_sync.create() ;
        
        try{
            synchronized( m_sync ){
                m_maxID = max ;
                m_startID = start ;
                m_id = start ;
            }
        }catch( NullPointerException nul ){
        }
        
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public final void clear()
    {
        
        m_sync.clear() ;
        m_sync.create() ;
        
        try{
            synchronized( m_sync.get() ){
                m_id = m_startID ;
                m_returnTimeout = System.currentTimeMillis() ;
            }
        }catch( NullPointerException nul ){
            m_id = m_startID ;
            m_returnTimeout = System.currentTimeMillis() ;
        }
        
    }
    
    /**
     * 一意なID項番を取得.
     * <BR><BR>
     * 一意なID項番を取得します.
     * <BR>
     * @return long 一意なID項番が返されます.
     */
    public final long getID()
    {
        long ret ;
        
        try{
            synchronized( m_sync.get() ){
                
                ret = m_id ;
                
                if( ret >= m_maxID ){
                    m_returnTimeout = System.currentTimeMillis() ;
                    m_id = m_startID ;
                }
                else{
                    m_id = ret + 1 ;
                }
                
            }
        }catch( NullPointerException nul ){
            ret = SequenceIDEx.NOT_ID ;
        }
        
        return ret ;
    }
    
    /**
     * 次に発行するID情報を設定.
     * <BR><BR>
     * 次に発行するID情報を設定します.
     * <BR>
     * @param id 次に発行されるID位置を設定します.
     */
    public final void setNextID( long id )
    {
        try{
            synchronized( m_sync.get() ){
                m_id = ( id <= m_startID || id >= m_maxID ) ?
                    ( ( id <= m_startID ) ? m_startID : m_maxID ) : id ;
            }
        }catch( NullPointerException nul ){
        }
        
    }
    
    /**
     * 設定ID開始番号の取得.
     * <BR><BR>
     * 設定されているID開始番号を取得します.
     * <BR>
     * @return long 設定されているID開始番号が返されます.
     */
    public final long getStartID()
    {
        long ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_startID ;
            }
        }catch( NullPointerException nul ){
            ret = SequenceIDEx.NOT_ID ;
        }
        
        return ret ;
    }
    
    /**
     * 設定ID幅(MAX値)を取得します.
     * <BR><BR>
     * 設定されているID幅(MAX値)を取得します.
     * <BR>
     * @return long 設定されているID幅(MAX値)が返されます.
     */
    public final long getMaxID()
    {
        long ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_maxID ;
            }
        }catch( NullPointerException nul ){
            ret = SequenceIDEx.NOT_ID ;
        }
        
        return ret ;
    }
    
    /**
     * カウント折り返し時の時間情報を取得.
     * <BR><BR>
     * カウントが設定ID幅(MAX値)を折り返したときの時間を取得します.
     * <BR>
     * @return long 折り返し時間が返されます.
     */
    public final long getReturnIDByTime()
    {
        long ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_returnTimeout ;
            }
        }catch( NullPointerException nul ){
            ret = 0L ;
        }
        
        return ret ;
    }

}
