/*
 * @(#)ConvertToIndexName.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.util.convert;

import java.math.BigDecimal;
import java.util.Date;

import org.maachang.commons.def.BaseDef;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.util.DateTimeFormat;


/**
 * 情報コンバート実装.
 * <BR><BR>
 * 情報をコンバートする場合の実装内容を提供します.
 * 取得方法として、インデックス名を用いて情報取得します.
 *
 * @version 1.00, 2003/11/04
 * @author  Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public abstract class ConvertToIndexName
{
    /**
     * DateTime日付変換用.
     */
    private final DateTimeFormat m_format = new DateTimeFormat() ;
    
    /**
     * コンストラクタ.
     */
    protected ConvertToIndexName()
    {}
    
    /**
     * 対象情報取得処理.
     * <BR><BR>
     * 取得対象の処理を実施します.
     * この処理は、取得対象の処理で渡される処理を
     * 定義します.
     * <BR>
     * @param name 取得対象の情報名を指定します.
     * @return String 取得対象の情報が返されます.
     * @exception InputException 入力例外.
     */
    protected abstract Object getStringByName( String name )
        throws InputException ;
    
    /**
     * 対象情報設定処理.
     * <BR><BR>
     * 設定対象の処理を実施します.
     * この処理は、設定対象の処理で渡される処理を
     * 定義します.
     * <BR>
     * @param name 設定対象の情報名を指定します.
     * @param stream 設定対象の内容を指定します.
     * @exception InputException 入力例外.
     */
    protected abstract void setStringByName( String name,String stream )
        throws InputException ;
    
    /**
     * 設定情報取得.
     * <BR><BR>
     * 指定情報を文字情報で取得します.
     * <BR>
     * @param name 設定対象の情報名を指定します.
     * @return String 取得情報に対する内容が返されます.
     */
    public String getString( String name )
    {
        String ret = null ;
        
        try{
            ret = ( String )this.getStringByName( name ) ;
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 設定情報取得.
     * <BR><BR>
     * 指定情報を日付・時間情報で取得します.
     * <BR>
     * @param name 設定対象の情報名を指定します.
     * @param format 読み込み対象の日付フォーマットを設定します.
     * @return Date 取得対象の日付オブジェクトが返されます.
     */
    public Date getDate( String name,String format )
    {
        Date ret = null ;
        
        try{
            
            synchronized( m_format ){
                m_format.create( format ) ;
                m_format.set( ( String )this.getStringByName( name ) ) ;
                ret = new Date( m_format.get().getTime() ) ;
            }
            
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 設定情報取得.
     * <BR><BR>
     * 指定情報をBigDecimal情報で取得します.
     * <BR>
     * @param name 設定対象の情報名を指定します.
     * @return BigDecimal 取得情報に対する内容が返されます.
     */
    public BigDecimal getBigDecimal( String name )
    {
        BigDecimal ret = null ;
        
        try{
            ret = new BigDecimal( ( String )this.getStringByName( name ) ) ;
            
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 設定情報取得.
     * <BR><BR>
     * 指定情報を整数情報で取得します.
     * <BR>
     * @param name 設定対象の情報名を指定します.
     * @return Short 取得情報に対する内容が(shortで)返されます.
     */
    public Short getShort( String name )
    {
        Short ret = null ;
        
        try{
            ret = new Short( ( String )this.getStringByName( name ) ) ;
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 設定情報取得.
     * <BR><BR>
     * 指定情報を整数情報で取得します.
     * <BR>
     * @param name 設定対象の情報名を指定します.
     * @return Integer 取得情報に対する内容が(intで)返されます.
     */
    public Integer getInteger( String name )
    {
        Integer ret = null ;
        
        try{
            ret = new Integer( ( String )this.getStringByName( name ) ) ;
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 設定情報取得.
     * <BR><BR>
     * 指定情報を整数情報で取得します.
     * <BR>
     * @param name 設定対象の情報名を指定します.
     * @return Long 取得情報に対する内容が(longで)返されます.
     */
    public Long getLong( String name )
    {
        Long ret = null ;
        
        try{
            ret = new Long( ( String )this.getStringByName( name ) ) ;
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 設定情報取得.
     * <BR><BR>
     * 指定情報を小数点を含む数値情報で取得します.
     * <BR>
     * @param name 設定対象の情報名を指定します.
     * @return Float 取得情報に対する内容が返されます.
     */
    public Float getFloat( String name )
    {
        Float ret = null ;
        
        try{
            ret = new Float( ( String )this.getStringByName( name ) ) ;
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 設定情報取得.
     * <BR><BR>
     * 指定情報を小数点を含む数値情報で取得します.
     * <BR>
     * @param name 設定対象の情報名を指定します.
     * @return Double 取得情報に対する内容が返されます.
     */
    public Double getDouble( String name )
    {
        Double ret = null ;
        
        try{
            ret = new Double( ( String )this.getStringByName( name ) ) ;
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 設定情報取得.
     * <BR><BR>
     * 指定情報を判定を行うブール変数で取得します.
     * <BR>
     * @param name 設定対象の情報名を指定します.
     * @return Boolean 取得情報に対する内容が返されます.
     */
    public Boolean getBoolean( String name )
    {
        String code = null ;
        Boolean ret = null ;
        
        try{
            
            code = ( String )this.getStringByName( name ) ;
            
            if( code.equalsIgnoreCase( BaseDef.BOOLEAN_TRUE ) == true ){
                ret = new Boolean( true ) ;
            }else if( code.equalsIgnoreCase( BaseDef.BOOLEAN_FALSE ) == true ){
                ret = new Boolean( false ) ;
            }else{
                ret = null ;
            }
            
        }catch( Exception e ){
            ret = null ;
        }finally{
            code = null ;
        }
        
        return ret ;
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 情報を文字情報で設定します.
     * <BR>
     * @param name 設定対象の情報名を指定します.
     * @param in 設定対象の文字情報を設定します.
     */
    public void setString( String name,String in )
    {
        try{
            this.setStringByName( name,in ) ;
        }catch( Exception e ){
        }
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 情報を日付・時間情報で設定します.
     * <BR>
     * @param name 設定対象の情報名を指定します.
     * @param format 対象の日付フォーマットを指定します.
     * @param in 設定対象の日付・時間情報を設定します.
     */
    public void setDate( String name,String format,Date in )
    {
        try{
            
            synchronized( m_format ){
                m_format.create( format ) ;
                m_format.set( in.getTime() ) ;
                this.setStringByName( name,m_format.toString() ) ;
            }
            
        }catch( Exception e ){
        }
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 情報をBigDecimal情報で設定します.
     * <BR>
     * @param name 設定対象の情報名を指定します.
     * @param in 設定対象のBigDecimal情報を設定します.
     */
    public void setBigDecimal( String name,BigDecimal in )
    {
        try{
            this.setStringByName( name,in.toString() ) ;
        }catch( Exception e ){
        }
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 情報を整数情報で設定します.
     * <BR>
     * @param name 設定対象の情報名を指定します.
     * @param in 設定対象の整数情報を設定します.
     */
    public void setShort( String name,short in )
    {
        try{
            this.setStringByName( name,String.valueOf( in ) ) ;
        }catch( Exception e ){
        }
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 情報を整数情報で設定します.
     * <BR>
     * @param name 設定対象の情報名を指定します.
     * @param in 設定対象の整数情報を設定します.
     */
    public void setShort( String name,Short in )
    {
        try{
            this.setStringByName( name,in.toString() ) ;
        }catch( Exception e ){
        }
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 情報を整数情報で設定します.
     * <BR>
     * @param name 設定対象の情報名を指定します.
     * @param in 設定対象の整数情報を設定します.
     */
    public void setInteger( String name,int in )
    {
        try{
            this.setStringByName( name,String.valueOf( in ) ) ;
        }catch( Exception e ){
        }
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 情報を整数情報で設定します.
     * <BR>
     * @param name 設定対象の情報名を指定します.
     * @param in 設定対象の整数情報を設定します.
     */
    public void setInteger( String name,Integer in )
    {
        try{
            this.setStringByName( name,in.toString() ) ;
        }catch( Exception e ){
        }
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 情報を整数情報で設定します.
     * <BR>
     * @param name 設定対象の情報名を指定します.
     * @param in 設定対象の整数情報を設定します.
     */
    public void setLong( String name,long in )
    {
        try{
            this.setStringByName( name,String.valueOf( in ) ) ;
        }catch( Exception e ){
        }
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 情報を整数情報で設定します.
     * <BR>
     * @param name 設定対象の情報名を指定します.
     * @param in 設定対象の整数情報を設定します.
     */
    public void setLong( String name,Long in )
    {
        try{
            this.setStringByName( name,in.toString() ) ;
        }catch( Exception e ){
        }
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 情報を小数点情報で設定します.
     * <BR>
     * @param name 設定対象の情報名を指定します.
     * @param in 設定対象の整数情報を設定します.
     */
    public void setFloat( String name,float in )
    {
        try{
            this.setStringByName( name,String.valueOf( in ) ) ;
        }catch( Exception e ){
        }
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 情報を小数点情報で設定します.
     * <BR>
     * @param name 設定対象の情報名を指定します.
     * @param in 設定対象の整数情報を設定します.
     */
    public void setFloat( String name,Float in )
    {
        try{
            this.setStringByName( name,in.toString() ) ;
        }catch( Exception e ){
        }
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 情報を小数点情報で設定します.
     * <BR>
     * @param name 設定対象の情報名を指定します.
     * @param in 設定対象の整数情報を設定します.
     */
    public void setDouble( String name,double in )
    {
        try{
            this.setStringByName( name,String.valueOf( in ) ) ;
        }catch( Exception e ){
        }
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 情報を小数点情報で設定します.
     * <BR>
     * @param name 設定対象の情報名を指定します.
     * @param in 設定対象の整数情報を設定します.
     */
    public void setDouble( String name,Double in )
    {
        try{
            this.setStringByName( name,in.toString() ) ;
        }catch( Exception e ){
        }
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 情報をBoolean情報で設定します.
     * <BR>
     * @param name 設定対象の情報名を指定します.
     * @param in 設定対象の整数情報を設定します.
     */
    public void setBoolean( String name,boolean in )
    {
        try{
            this.setStringByName( name,String.valueOf( in ) ) ;
        }catch( Exception e ){
        }
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 情報をBoolean情報で設定します.
     * <BR>
     * @param name 設定対象の情報名を指定します.
     * @param in 設定対象の整数情報を設定します.
     */
    public void setBoolean( String name,Boolean in )
    {
        try{
            this.setStringByName( name,in.toString() ) ;
        }catch( Exception e ){
        }
    }

}

