/*
 * @(#)UnderReceiveBean.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.connect.table ;

import java.net.InetAddress;

import org.maachang.connect.thread.ProtocolDef;

import org.maachang.commons.exception.InputException;
import org.maachang.commons.resource.BinResource;
import org.maachang.commons.resource.Resource;
import org.maachang.commons.resource.ResourceType;
import org.maachang.commons.util.box.SimpleFlagBox;

/**
 * 1つの受信中データを表すBean.
 *
 * @version 2006/12/22
 * @author  Masahito Suzuki
 * @since   MaachangConnect 1.00
 */
public class UnderReceiveBean implements BaseBean {
    
    /**
     * コネクションID.
     */
    private long id = -1L ;
    
    /**
     * 生成時間.
     */
    private long createTime = -1L ;
    
    /**
     * 再送カウント.
     */
    private int retryCount = 0 ;
    
    /**
     * コネクションアドレス.
     */
    private InetAddress address = null ;
    
    /**
     * コネクションポート番号.
     */
    private int port = -1 ;
    
    /**
     * 暗号ワード.
     */
    private String cb32Word = null ;
    
    /**
     * 受信データ電文.
     */
    private BinResource receiveData = null ;
    
    /**
     * 受信パケットサイズ.
     */
    private int receivePacketSize = -1 ;
    
    /**
     * パケット管理テーブル.
     */
    private SimpleFlagBox packetTable = null ;
    
    /**
     * コンストラクタ.
     */
    private UnderReceiveBean() {
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * １つの受信中Beanを生成します.
     * <BR>
     * @param resType 対象のリソースタイプを設定します.
     * @param id 対象のIDを設定します.
     * @param address 受信コネクションアドレスを設定します.
     * @param port 受信コネクションポート番号を設定します.
     * @param cb32Word 対象の暗号ワードを設定します.
     * @param size 受信データ長を設定します.
     * @exception InputException 入力例外.
     */
    public UnderReceiveBean(
        ResourceType resType,long id,InetAddress address,
        int port,String cb32Word,int size )
        throws InputException {
        if( address == null || size <= 0 ) {
            throw new InputException( "引数は不正です" ) ;
        }
        this.id = id ;
        this.createTime = System.currentTimeMillis() ;
        this.retryCount = 0 ;
        this.address = address ;
        this.port = port ;
        this.cb32Word = cb32Word ;
        this.receiveData = Resource.createBinResource( resType,size ) ;
        this.receivePacketSize = ( size / ProtocolDef.SEND_PACKET_DATA_LENGTH ) +
            ( ( ( size % ProtocolDef.SEND_PACKET_DATA_LENGTH ) != 0 ) ? 1 : 0 ) ;
        this.packetTable = new SimpleFlagBox( this.receivePacketSize ) ;
    }
    
    /**
     * 終了化処理.
     * <BR><BR>
     * 終了化処理を行います.
     */
    protected void finalize() throws Exception {
        id = -1L ;
        createTime = -1L ;
        retryCount = 0 ;
        address = null ;
        port = -1 ;
        cb32Word = null ;
        receiveData = null ;
        receivePacketSize = -1 ;
        packetTable = null ;
    }
    
    /**
     * 指定パケットを、受信データに設定.
     * <BR><BR>
     * 指定パケットを、受信データに設定します.
     * <BR>
     * @param packetNo 対象のパケット項番を設定します.
     * @param length 対象データ長を設定します.
     * @param packet 対象のパケット情報を設定します.
     */
    public synchronized void putPacketData( int packetNo,int length,byte[] packet ) {
        
        if( packetNo < 0 || packetNo >= packetTable.getMaxSize() ||
            length <= 0 || length > ProtocolDef.SEND_PACKET_DATA_LENGTH ||
            packet == null || packet.length <= 0 ) {
            return ;
        }
        
        receiveData.setBinary(
            packetNo * ProtocolDef.SEND_PACKET_DATA_LENGTH,
            packet,
            ProtocolDef.SEND_PACKET_HEADER_LENGTH,
            length ) ;
        
        createTime = System.currentTimeMillis() ;
        retryCount = 0 ;
        packetTable.add( true,packetNo ) ;
        
    }
    
    /**
     * パケット未到達ID群を取得.
     * <BR><BR>
     * パケット未到達ID群を取得します.
     * <BR>
     * @return int[] パケット未到達ID群が返されます.<BR>
     *               [null]が返された場合、未到達パケット情報は存在しません.
     */
    public synchronized int[] getRetryPacketIds() {
        return packetTable.useSearchSectors() ;
    }

    /**
     * パケット未到達ID群を取得.
     * <BR><BR>
     * パケット未到達ID群を取得します.
     * <BR>
     * @param size パケットデータ取得長を設定します.
     * @return int[] パケット未到達ID群が返されます.<BR>
     *               [null]が返された場合、未到達パケット情報は存在しません.
     */
    public synchronized int[] getRetryPacketIds( int size ) {
        return packetTable.useSearchSectors( size ) ;
    }
    
    /**
     * 受信完了した場合、受信完了データとして生成.
     * <BR><BR>
     * 受信完了した場合、受信完了データとして生成します.<BR>
     * また、完了していない場合は、情報を取得できません.
     * <BR>
     * @return CompletionBean 受信完了Beanが返されます.
     */
    public synchronized CompletionBean getCompletionBean() {
        if( packetTable.isSpace() == false ) {
            return new CompletionBean( address,port,cb32Word,receiveData ) ;
        }
        return null ;
    }
    
    /**
     * 生成時間を更新.
     * <BR><BR>
     * 生成時間を更新します.
     */
    public synchronized void updateTime() {
        createTime = System.currentTimeMillis() ;
        retryCount = 0 ;
    }
    
    /**
     * 再送カウントを１インクリメント.
     * <BR><BR>
     * 再送カウントを１インクリメントします.
     */
    public synchronized void retryCountIncrement() {
        createTime = System.currentTimeMillis() ;
        retryCount ++ ;
    }

    /**
     * createTime を取得.
     * <BR><BR>
     * @return createTime が返されます.
     */
    public synchronized long getCreateTime() {
        return createTime;
    }

    /**
     * retryCount を取得.
     * <BR><BR>
     * @return retryCount が返されます.
     */
    public synchronized int getRetryCount() {
        return retryCount;
    }

    /**
     * id を取得.
     * <BR><BR>
     * @return id が返されます.
     */
    public synchronized long getId() {
        return id;
    }

    /**
     * address を取得.
     * <BR><BR>
     * @return address が返されます.
     */
    public InetAddress getAddress() {
        return address;
    }

    /**
     * port を取得.
     * <BR><BR>
     * @return port が返されます.
     */
    public int getPort() {
        return port;
    }
    
    /**
     * cb32Word を取得.
     * <BR><BR>
     * @return cb32Word が返されます.
     */
    public String getCb32Word() {
        return cb32Word;
    }

    /**
     * receiveData を取得.
     * <BR><BR>
     * @return receiveData が返されます.
     */
    public synchronized BinResource getReceiveData() {
        return receiveData;
    }

    /**
     * packetTable を取得.
     * <BR><BR>
     * @return packetTable が返されます.
     */
    public synchronized SimpleFlagBox getPacketTable() {
        return packetTable;
    }

    /**
     * receivePacketSize を取得.
     * <BR><BR>
     * @return receivePacketSize が返されます.
     */
    public synchronized int getReceivePacketSize() {
        return receivePacketSize;
    }

    /**
     * パケット情報が完了したかチェック.
     * <BR><BR>
     * パケット情報が完了しているかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、受信情報として、完了しています.
     *                 [false]が返された場合、受信情報として、完了していません.
     */
    public synchronized boolean isCompletion() {
        return ( packetTable.isSpace() == false ) ? true : false ;
    }

}

