/*
 * @(#)TableTimeoutThread.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.connect.thread ;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.connect.table.ConnectTable;
import org.maachang.connect.table.MaachangConnectSync;
import org.maachang.connect.table.MaachangConnectTableFactory;
import org.maachang.connect.table.SendTable;
import org.maachang.connect.table.UnderReceiveTable;

import org.maachang.commons.exception.BaseException;
import org.maachang.commons.exception.ExecutionException;
import org.maachang.commons.thread.ExecutionThread;
import org.maachang.commons.thread.LoopThread;
import org.maachang.commons.thread.Synchronized;
import org.maachang.commons.util.UtilCom;

/**
 * テーブルデータ内タイムアウトチェックスレッド.
 *
 * @version 2006/12/23
 * @author  Masahito Suzuki
 * @since   MaachangConnect 1.00
 */
public class TableTimeoutThread extends ExecutionThread {
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( TableTimeoutThread.class ) ;
    
    
    
    /**
     * テーブルファクトリ.
     */
    private MaachangConnectTableFactory tableFactory = null ;
    
    /**
     * 処理同期.
     */
    private MaachangConnectSync execSynchronized = null ;
    
    /**
     * ループスレッド.
     */
    private final LoopThread thread = new LoopThread() ;
    
    /**
     * 同期処理.
     */
    private final Synchronized sync = new Synchronized() ;
    
    
    /**
     * コンストラクタ.
     */
    private TableTimeoutThread(){
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * タイムアウト監視対象のテーブルFactoryを設定します.
     * <BR>
     * @param tableFactory 対象のテーブルFactoryを設定します.
     * @param execSynchronized 実行同期を設定します.
     */
    public TableTimeoutThread( MaachangConnectTableFactory tableFactory,MaachangConnectSync execSynchronized ){
        
        sync.create() ;
        
        try{
            
            this.tableFactory = tableFactory ;
            this.execSynchronized = execSynchronized ;
            
            thread.create( this ) ;
            thread.startThread() ;
            
        }catch( Exception e ){
            this.destroy() ;
        }
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * <BR>
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.destroy() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    public final void destroy()
    {
        sync.clear() ;
        thread.clear() ;
        tableFactory = null ;
        execSynchronized = null ;
    }
    
    /**
     * スレッド状態を取得.
     * <BR><BR>
     * スレッド状態を取得します.
     * <BR>
     * @return boolean スレッド状態が返されます.<BR>
     *                 [true]が返された場合、スレッドは実行中です.<BR>
     *                 [false]が返された場合、スレッドは停止中です.
     */
    public final boolean isThread()
    {
        boolean ret ;
        
        try{
            synchronized( sync.get() ){
                ret = thread.isThread() ;
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    
    
    /**
     * 実行初期化処理をサポートします.
     * <BR><BR>
     * 実行初期化処理をサポートします.<BR>
     * この処理は、スレッド処理が開始された時に呼び出されます.
     * <BR>
     * @param obj 実行開始時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void init( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行終了化処理をサポートします.
     * <BR><BR>
     * 実行終了化処理をサポートします.<BR>
     * この処理は、スレッド処理が終了された時に呼び出されます.
     * <BR>
     * @param obj 実行終了時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void exit( Object obj )
        throws ExecutionException
    {
        // スレッド終了に合わせて、全体終了フラグをONに設定.
        tableFactory.setExitThreadFlag( true ) ;
    }
    
    /**
     * ストップ処理をサポートします。
     * <BR><BR>
     * ストップ処理をサポートします。<BR>
     * この処理は、スレッドでのストップ処理に対して呼び出し実行されます.
     * <BR>
     * @param obj ストップ時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void stop( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行処理をサポートします。
     * <BR><BR>
     * 実行処理をサポートします。<BR>
     * この処理は、スレッドでの実行処理に対して呼び出し実行されます.
     * <BR>
     * @param obj 実行時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void execution( Object obj )
        throws ExecutionException
    {
        int i ;
        int len ;
        
        MaachangConnectTableFactory factory = null ;
        MaachangConnectSync execSync = null ;
        
        try{
            
            UtilCom.idleTime() ;
            
            synchronized( sync.get() ) {
                factory = this.tableFactory ;
                execSync = this.execSynchronized ;
            }
            
            // スレッド終了通知がONの場合.
            if( factory.isExitThreadFlag() == true ) {
                LOG.error( "スレッド終了通知を検知しました" ) ;
                throw new ExecutionException( "スレッド終了通知を検知",
                    ExecutionException.LEVEL_STOP ) ;
            }
            
            UtilCom.idleTime() ;
            
            // コネクションタイムアウト監視.
            ConnectTable conn = factory.getConnectTable() ;
            if( conn != null ) {
                len = conn.size() ;
                for( i = len-1 ; i >= 0 ; i -- ) {
                    try {
                        synchronized( execSync.getConnectSync().get() ) {
                            if( conn.exitConnectByTimeout( i ) == true ) {
                                LOG.debug( "[TableTimeoutThread]コネクションテーブルから削除:" + i ) ;
                            }
                        }
                    } catch( Exception ee ) {
                        //LOG.warn( "[Connect]処理例外-"+i,ee ) ;
                    }
                    UtilCom.idleTime() ;
                }
            }
            else {
                factory.setExitThreadFlag( true ) ;
                throw new ExecutionException( "コネクションテーブル異常",
                    ExecutionException.LEVEL_STOP ) ;
            }
            conn = null ;
            
            UtilCom.idleTime() ;
            
            // 送信電文タイムアウト監視.
            SendTable snd = factory.getSendTable() ;
            if( snd != null ) {
                len = snd.size() ;
                for( i = len-1 ; i >= 0 ; i -- ) {
                    try {
                        synchronized( execSync.getSendSync().get() ) {
                            if( snd.removeSendDataByTimeout( i ) == true ) {
                                LOG.debug( "[TableTimeoutThread]送信テーブルから削除:" + i ) ;
                            }
                        }
                    } catch( Exception ee ) {
                        //LOG.warn( "[Send]処理例外-"+i,ee ) ;
                    }
                    UtilCom.idleTime() ;
                }
            }
            else {
                factory.setExitThreadFlag( true ) ;
                throw new ExecutionException( "送信テーブル異常",
                    ExecutionException.LEVEL_STOP ) ;
            }
            snd = null ;
            
            UtilCom.idleTime() ;
            
            // 受信中電文タイムアウト監視.
            UnderReceiveTable rcv = factory.getUnderReceiveTable() ;
            if( rcv != null ) {
                len = rcv.size() ;
                for( i = len-1 ; i >= 0 ; i -- ) {
                    try {
                        synchronized( execSync.getUnderReceiveSync().get() ) {
                            if( rcv.removeUnderReceiveaByTimeout( i ) == true ) {
                                LOG.debug( "[TableTimeoutThread]受信待ちテーブルから削除:" + i ) ;
                            }
                        }
                    } catch( Exception ee ) {
                        //LOG.warn( "[UnderReceive]処理例外-"+i,ee ) ;
                    }
                    UtilCom.idleTime() ;
                }
            }
            else {
                factory.setExitThreadFlag( true ) ;
                throw new ExecutionException( "受信待ちテーブル異常",
                    ExecutionException.LEVEL_STOP ) ;
            }
            rcv = null ;
            
        }catch( NullPointerException nul ){
            throw new ExecutionException(
                nul,ExecutionException.LEVEL_STOP
            ) ;
        }catch( ExecutionException ex ) {
            LOG.error( "実行処理例外",ex ) ;
            throw ex ;
        }catch( BaseException be ){
            LOG.error( "エラーが発生しました", be ) ;
        }catch( Exception e ){
            LOG.error( "エラーが発生しました", e ) ;
        }finally{
            factory = null ;
        }
        
    }
    
}

