/*
 * @(#)ConfigService.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.config.service ;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.io.IOCom;
import org.maachang.commons.sys.NamingManager;
import org.maachang.queue.config.MqConfig;
import org.maachang.queue.config.MqDefine;
import org.maachang.queue.config.SSLConfig;

/**
 * コンフィグサービス.
 *  
 * @version 2006/12/31
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
public class ConfigService {
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( ConfigService.class ) ;
    
    /**
     * ネーミングマネージャ登録名.
     */
    public static final String NAMING_MANAGER = "CONFIG@maachangq.config.service" ;
    
    /**
     * 同期オブジェクト.
     */
    private static final Object SYNC = new Object() ;
    
    /**
     * コンストラクタ.
     */
    public ConfigService() { }
    
    
    /**
     * 初期化処理.
     * <BR><BR>
     * 初期化処理を行います.
     */
    public void init() {
        this.init( MqDefine.EMV_NAME+MqDefine.CONF_DIR,
            MqDefine.CONF_FILE,
            MqDefine.SSL_CONF_FILE ) ;
    }
    
    /**
     * 初期化処理.
     * <BR><BR>
     * 初期化処理を行います.
     * <BR>
     * @param baseDir 対象の基本ディレクトリ名を設定します.
     */
    public void init( String baseDir ) {
        this.init( baseDir,
            MqDefine.CONF_FILE,
            MqDefine.SSL_CONF_FILE ) ;
    }
    
    /**
     * 初期化処理.
     * <BR><BR>
     * 初期化処理を行います.
     * <BR>
     * @param baseDir 対象の基本ディレクトリ名を設定します.
     * @param baseFile 対象の基本コンフィグファイル名を設定します.
     * @param sslFile 対象のSSLコンフィグファイル名を設定します.
     */
    public void init( String baseDir,String baseFile,String sslFile ) {
        
        LOG.info( "#### MaachangQコンフィグサービス生成処理 ####" ) ;
        LOG.info( "# コンフィグディレクトリ:" + baseDir ) ;
        LOG.info( "# 基本コンフィグファイル名:" + baseFile ) ;
        LOG.info( "# SSLコンフィグファイル名:" + sslFile ) ;
        
        if( baseDir == null || ( baseDir = baseDir.trim() ).length() <= 0 ) {
            throw new InputException( "指定コンフィグディレクトリ名は不正です" ) ;
        }
        else if( IOCom.isDirExists( baseDir ) == false ) {
            throw new InputException( "指定コンフィグディレクトリは存在しません" ) ;
        }
        
        ConfigChild ch = new ConfigChild() ;
        
        MqConfigImple conf = new MqConfigImple() ;
        conf.create( baseDir+"/"+baseFile ) ;
        
        SSLConfigImple ssl = new SSLConfigImple() ;
        ssl.create( baseDir+"/"+sslFile ) ;
        
        ch.setMqConfig( conf ) ;
        ch.setSslConfig( ssl ) ;
        
        this.destroy( false ) ;
        
        // ネーミングマネージャに登録.
        synchronized( SYNC ) {
            
            NamingManager.add( NAMING_MANAGER,ch ) ;
            
        }
        
    }
    
    /**
     * 終了化処理.
     * <BR><BR>
     * 終了化処理を行います.
     */
    public void destroy() {
        this.destroy( true ) ;
    }
    
    /**
     * 終了化処理.
     * <BR><BR>
     * 終了化処理を行います.
     * <BR>
     * @param mode ログ表示モードを設定します.
     */
    public void destroy( boolean mode ) {
        
        if( mode == true ) {
            LOG.info( "#### MaachangQコンフィグサービス破棄処理 ####" ) ;
        }
        
        // ネーミングマネージャから削除.
        synchronized( SYNC ) {
            
            NamingManager.remove( NAMING_MANAGER ) ;
            
        }
        
    }
    
    /**
     * 基本コンフィグ情報を取得.
     * <BR><BR>
     * 基本コンフィグ情報を取得します.
     * <BR>
     * @return MqConfig 基本コンフィグ情報が返されます.
     */
    public static final MqConfig getMqConfig() {
        
        ConfigChild ch = null ;
        
        synchronized( SYNC ) {
            
            ch = ( ConfigChild )NamingManager.get( NAMING_MANAGER ) ;
            
        }
        
        if( ch != null ) {
            return ch.getMqConfig() ;
        }
        
        return null ;
    }
    
    /**
     * SSLコンフィグ情報を取得.
     * <BR><BR>
     * SSLコンフィグ情報を取得します.
     * <BR>
     * @return SSLConfig SSLコンフィグ情報が返されます.
     */
    public static final SSLConfig getSSLConfig() {
        
        ConfigChild ch = null ;
        
        synchronized( SYNC ) {
            
            ch = ( ConfigChild )NamingManager.get( NAMING_MANAGER ) ;
            
        }
        
        if( ch != null ) {
            return ch.getSslConfig() ;
        }
        
        return null ;
    }
    
}

class ConfigChild {
    
    private MqConfig mqConfig = null ;
    private SSLConfig sslConfig = null ;
    
    /**
     * mqConfigを取得.
     * <BR><BR>
     * @return mqConfig が返されます。
     */
    public MqConfig getMqConfig() {
        return mqConfig;
    }
    
    /**
     * mqConfigを設定.
     * <BR><BR>
     * @param mqConfig 対象のmqConfigを設定.
     */
    public void setMqConfig(MqConfig mqConfig) {
        this.mqConfig = mqConfig;
    }
    
    /**
     * sslConfigを取得.
     * <BR><BR>
     * @return sslConfig が返されます。
     */
    public SSLConfig getSslConfig() {
        return sslConfig;
    }
    
    /**
     * sslConfigを設定.
     * <BR><BR>
     * @param sslConfig 対象のsslConfigを設定.
     */
    public void setSslConfig(SSLConfig sslConfig) {
        this.sslConfig = sslConfig;
    }
    
}

