/*
 * @(#)ExecutionAdminChannel.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.connect.admin ;

import org.maachang.queue.access.MaachangQAccessDefine;
import org.maachang.queue.access.MaachangQErrorCode;
import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.access.net.ConnectObject;
import org.maachang.queue.access.protocol.admin.AdminChannelBean;
import org.maachang.queue.access.protocol.admin.AdminChannelProtocol;
import org.maachang.queue.access.status.ChannelStatus;
import org.maachang.queue.connect.ExecutionAdmin;
import org.maachang.queue.connect.admin.login.LoginSession;
import org.maachang.queue.connect.common.Telegram;
import org.maachang.queue.main.channel.Channel;
import org.maachang.queue.main.channel.ChannelFactory;
import org.maachang.queue.main.channel.ChannelKey;
import org.maachang.queue.main.channel.SendChannel;
import org.maachang.queue.main.service.SaveAndLoadService;


/**
 * チャネル電文処理.
 *
 * @version 2007/01/21
 * @author  Masahito Suzuki
 * @since   MaachangQ 1.00
 */
public class ExecutionAdminChannel
    extends AbstractExecutionAdmin
    implements ExecutionAdmin {
    
    /**
     * コンストラクタ.
     */
    public ExecutionAdminChannel() {
        
    }
    
    /**
     * 管理者実行処理IDを取得.
     * <BR><BR>
     * 設定されている管理者実行処理IDを取得します.
     * <BR>
     * @return int 管理者実行処理IDが返されます.
     */
    public int getId() {
        return AdminChannelProtocol.CATEGORY_TYPE_ADMIN_CHANNEL ;
    }
    
    /**
     * 実行処理.
     * <BR><BR>
     * 各管理者用実行処理を実装するためのメソッド.
     * <BR>
     * @param telegramType 処理対象の実行タイプを格納します.
     * @param connect コネクションオブジェクトが設定されます.
     * @param session 対象のログインセッションが設定されます.
     * @param id 対象の電文IDが設定されます.
     * @param telegram 受信された電文が設定されます.
     * @return boolean 処理結果が返されます.
     * @exception Exception 例外.
     */
    public boolean execution( int[] telegramType,LoginSession session,
        ConnectObject connect,int id,Telegram telegram )
        throws Exception {
        
        byte[] binary = telegram.getTelegram() ;
        telegram.clear() ;
        
        // 受信バイナリを解析.
        AdminChannelBean bean = new AdminChannelBean() ;
        AdminChannelProtocol.analysisProtocol( bean,binary ) ;
        telegramType[ 0 ] = bean.getType() ;
        bean.setId( id ) ;
        binary = null ;
        
        switch( bean.getType() ) {
            case AdminChannelProtocol.TYPE_ADD_CHANNEL :
                this.addChannel( session,connect,bean ) ;
                return true ;
            case AdminChannelProtocol.TYPE_SET_CHANNEL :
                this.setChannel( session,connect,bean ) ;
                return true ;
            case AdminChannelProtocol.TYPE_REMOVE_CHANNEL :
                this.removeChannel( session,connect,bean ) ;
                return true ;
            case AdminChannelProtocol.TYPE_STOP_CHANNEL :
                this.stopChannel( session,connect,bean ) ;
                return true ;
            case AdminChannelProtocol.TYPE_START_CHANNEL :
                this.startChannel( session,connect,bean ) ;
                return true ;
            case AdminChannelProtocol.TYPE_GET_CHANNEL :
                this.getChannel( session,connect,bean ) ;
                return true ;
            case AdminChannelProtocol.TYPE_GET_CHANNEL_NAMES :
                this.getChannelNames( session,connect,bean ) ;
                return true ;
            case AdminChannelProtocol.TYPE_GET_CHANNEL_SIZE :
                this.getChannelSize( session,connect,bean ) ;
                return true ;
            case AdminChannelProtocol.TYPE_IS_CHANNEL :
                this.isChannel( session,connect,bean ) ;
                return true ;
        }
        
        return false ;
        
    }
    
    /**
     * チャネル追加.
     */
    public final void addChannel(
        LoginSession session,ConnectObject connect,AdminChannelBean bean )
        throws Exception {
        
        super.checkLogin( true,session,bean ) ;
        
        synchronized( ChannelFactory.getSync() ) {
            
            // チャネルが存在する場合.
            if( ChannelFactory.get( bean.getChannelType(),bean.getName() ) != null ) {
                throw new MaachangQException(
                    this.stringChannel( bean ) + "は既に存在します",
                    MaachangQErrorCode.ERROR_USE_CHANNEL ) ;
            }
            
            try {
                
                // 送信チャネル追加.
                if( bean.getChannelType() == MaachangQAccessDefine.TYPE_SEND ) {
                    ChannelFactory.createSendChannel(
                        bean.getName(),bean.getQueueManager(),
                        bean.getInetAddress(),bean.getPort(),
                        bean.getConnectName(),bean.getUseCb32Word() ) ;
                }
                // 受信チャネル追加.
                else {
                    ChannelFactory.createReveiveChannel(
                        bean.getName() ) ;
                }
                
                // 状態セーブ.
                SaveAndLoadService.save( SaveAndLoadService.SAVE_TYPE_CHANNEL ) ;
                
            } catch( Exception e ) {
                LOG.error( "error",e ) ;
                throw new MaachangQException(
                    this.stringChannel( bean ) + "の追加に失敗",
                    MaachangQErrorCode.ERROR_ADD_CHANNEL ) ;
            }
            
        }
        
        super.sendResult( connect,bean,
            MaachangQErrorCode.SUCCESS,"チャネル追加" ) ;
        
    }
    
    /**
     * チャネル設定.
     */
    public final void setChannel(
        LoginSession session,ConnectObject connect,AdminChannelBean bean )
        throws Exception {
        
        super.checkLogin( true,session,bean ) ;
        
        synchronized( ChannelFactory.getSync() ) {
            
            Channel channel = this.getChannel( bean ) ;
            
            if( channel != null && ( channel instanceof SendChannel ) == true ) {
                
                try {
                    
                    SendChannel send = ( SendChannel )channel ;
                    
                    if( send.isCb32Word( bean.getConnectName(),bean.getUseCb32Word() ) == true ) {
                        send.setConnectName( bean.getConnectName() ) ;
                        send.setUseCb32Word( bean.getUseCb32Word() ) ;
                    }
                    
                    send.setQueueManager( bean.getQueueManager() ) ;
                    
                    if( bean.getHost() != null && bean.getHost().trim().length() > 0 ) {
                        send.setInetAddress( bean.getInetAddress() ) ;
                    }
                    
                    if( bean.getPort() > 0 ) {
                        send.setPort( bean.getPort() ) ;
                    }
                    
                    // 状態セーブ.
                    SaveAndLoadService.save( SaveAndLoadService.SAVE_TYPE_CHANNEL ) ;
                    
                } catch( Exception e ) {
                    LOG.error( "error",e ) ;
                    throw new MaachangQException(
                        this.stringChannel( bean ) + "の設定に失敗",
                        MaachangQErrorCode.ERROR_ADD_CHANNEL ) ;
                }
            }
            
        }
        
        super.sendResult( connect,bean,
            MaachangQErrorCode.SUCCESS,"チャネル設定" ) ;
        
    }
    
    /**
     * チャネル削除.
     */
    public final void removeChannel(
        LoginSession session,ConnectObject connect,AdminChannelBean bean )
        throws Exception {
        
        super.checkLogin( true,session,bean ) ;
        
        synchronized( ChannelFactory.getSync() ) {
            
            this.getChannel( bean ) ;
            
            try {
                // チャネル削除.
                ChannelFactory.remove(
                    bean.getChannelType(),bean.getName() ) ;
                
                // 状態セーブ.
                SaveAndLoadService.save( SaveAndLoadService.SAVE_TYPE_CHANNEL ) ;
            } catch( Exception e ) {
                LOG.error( "error",e ) ;
                throw new MaachangQException(
                    this.stringChannel( bean ) + "の削除に失敗",
                    MaachangQErrorCode.ERROR_REMOVE_QUEUE_MANAGER ) ;
            }
            
        }
        
        super.sendResult( connect,bean,
            MaachangQErrorCode.SUCCESS,"チャネル削除" ) ;
        
    }
    
    /**
     * チャネル停止.
     */
    public final void stopChannel(
        LoginSession session,ConnectObject connect,AdminChannelBean bean )
        throws Exception {
        
        super.checkLogin( true,session,bean ) ;
        
        synchronized( ChannelFactory.getSync() ) {
            
            Channel channel = this.getChannel( bean ) ;
            
            // チャネルステータスを計画停止.
            if( channel.getState() == ChannelStatus.STATE_SUCCESS ) {
                try {
                    channel.setState( ChannelStatus.STATE_PLANNED_STOP ) ;
                } catch( Exception e ) {
                    LOG.error( "error",e ) ;
                    throw new MaachangQException(
                        this.stringChannel( bean ) + "の停止に失敗",
                        MaachangQErrorCode.ERROR_STOP_CHANNEL ) ;
                }
            }
            else {
                throw new MaachangQException(
                    this.stringChannel( bean ) + "のステータスが正常以外(" +
                    ChannelStatus.getStateByString( channel.getState() ) +
                    ")のため、停止に失敗",
                    MaachangQErrorCode.ERROR_NOT_STOP_CHANNEL ) ;
            }
            
        }
        
        super.sendResult( connect,bean,
            MaachangQErrorCode.SUCCESS,"チャネル停止" ) ;
        
    }
    
    /**
     * チャネル停止から復元.
     */
    public final void startChannel(
        LoginSession session,ConnectObject connect,AdminChannelBean bean )
        throws Exception {
        
        super.checkLogin( true,session,bean ) ;
        
        synchronized( ChannelFactory.getSync() ) {
            
            Channel channel = this.getChannel( bean ) ;
            
            // チャネルステータスを計画停止から復元.
            if( channel.getState() == ChannelStatus.STATE_PLANNED_STOP ) {
                try {
                    channel.setState( ChannelStatus.STATE_STARTUP ) ;
                } catch( Exception e ) {
                    LOG.error( "error",e ) ;
                    throw new MaachangQException(
                        this.stringChannel( bean ) + "の開始に失敗",
                        MaachangQErrorCode.ERROR_RESTART_CHANNEL ) ;
                }
            }
            else {
                throw new MaachangQException(
                    this.stringChannel( bean ) + "の開始に失敗しました(" +
                    ChannelStatus.getStateByString( channel.getState() ) +
                    ")",
                    MaachangQErrorCode.ERROR_NOT_RESTART_CHANNEL ) ;
            }
            
        }
        
        super.sendResult( connect,bean,
            MaachangQErrorCode.SUCCESS,"チャネル開始" ) ;
        
    }
    
    /**
     * チャネル情報取得.
     */
    public final void getChannel(
        LoginSession session,ConnectObject connect,AdminChannelBean bean )
        throws Exception {
        
        super.checkLogin( false,session,bean ) ;
        
        synchronized( ChannelFactory.getSync() ) {
            
            Channel channel = this.getChannel( bean ) ;
            
            try {
                
                if( channel instanceof SendChannel ) {
                    
                    SendChannel send = ( SendChannel )channel ;
                    
                    bean.setQueueManager( send.getQueueManager() ) ;
                    bean.setHost( send.getInetAddress().getHostName() ) ;
                    bean.setPort( send.getPort() ) ;
                    bean.setConnectName( send.getConnectName() ) ;
                    bean.setUseCb32Word( send.getUseCb32Word() ) ;
                    
                }
                
                bean.setState( channel.getState() ) ;
                
            } catch( Exception e ) {
                LOG.error( "error",e ) ;
                throw new MaachangQException(
                    this.stringChannel( bean ) + "の取得に失敗",
                    MaachangQErrorCode.ERROR_GET_CHANNEL ) ;
            }
            
        }
        
        this.sendChannel( session,connect,bean ) ;
        
    }
    
    /**
     * チャネル名群取得.
     */
    public final void getChannelNames(
        LoginSession session,ConnectObject connect,AdminChannelBean bean )
        throws Exception {
        
        super.checkLogin( false,session,bean ) ;
        
        synchronized( ChannelFactory.getSync() ) {
            
            try {
                
                ChannelKey[] keys = ChannelFactory.getChannekKeys() ;
                
                if( keys != null && keys.length > 0 ) {
                    
                    int len = keys.length ;
                    String[] names = new String[ len ] ;
                    int[] nums = new int[ len ] ;
                    
                    for( int i = 0 ; i < len ; i ++ ) {
                        names[ i ] = keys[ i ].getName() ;
                        nums[ i ] = keys[ i ].getChannelType() ;
                    }
                    
                    bean.setParams( names ) ;
                    bean.setNumbers( nums ) ;
                }
                else {
                    bean.setParams( null ) ;
                    bean.setNumbers( null ) ;
                }
                
            } catch( Exception e ) {
                LOG.error( "error",e ) ;
                throw new MaachangQException(
                    "チャネル一覧取得に失敗",
                    MaachangQErrorCode.ERROR_GET_LIST_CHANNEL ) ;
            }
            
        }
        
        this.sendChannel( session,connect,bean ) ;
        
    }
    
    /**
     * チャネル数取得.
     */
    public final void getChannelSize(
        LoginSession session,ConnectObject connect,AdminChannelBean bean )
        throws Exception {
        
        super.checkLogin( false,session,bean ) ;
        
        synchronized( ChannelFactory.getSync() ) {
            
            try {
                
                int size = ChannelFactory.size() ;
                bean.setSize( size ) ;
                
            } catch( Exception e ) {
                LOG.error( "error",e ) ;
                throw new MaachangQException(
                    "チャネル数取得に失敗",
                    MaachangQErrorCode.ERROR_GET_SIZE_CHANNEL ) ;
            }
            
        }
        
        this.sendChannel( session,connect,bean ) ;
        
    }
    
    /**
     * チャネル存在チェック.
     */
    public final void isChannel(
        LoginSession session,ConnectObject connect,AdminChannelBean bean )
        throws Exception {
        
        super.checkLogin( false,session,bean ) ;
        
        synchronized( ChannelFactory.getSync() ) {
            
            try {
                this.getChannel( bean ) ;
            } catch( MaachangQException fa ) {
                throw fa ;
            } catch( Exception e ) {
                LOG.error( "error",e ) ;
                throw new MaachangQException(
                    "チャネル(" + bean.getName() +
                    ")の存在確認に失敗",
                    MaachangQErrorCode.ERROR_IS_CHANNEL ) ;
            }
            
        }
        
        super.sendResult( connect,bean,
            MaachangQErrorCode.SUCCESS,"チャネル存在" ) ;
    }
    
    /**
     * 戻り電文送信.
     */
    private final void sendChannel(
        LoginSession session,ConnectObject connect,AdminChannelBean bean )
        throws Exception {
        
        byte[] binary = AdminChannelProtocol.createProtocol(
            bean.getId(),session.isOwner(),bean ) ;
        
        connect.send( binary ) ;
        
    }
    
    /**
     * チャネル取得.
     */
    private final Channel getChannel( AdminChannelBean bean )
        throws Exception {
        
        Channel ch = ChannelFactory.get(
            bean.getChannelType(),bean.getName() ) ;
        
        // チャネルが存在しない場合.
        if( ch == null ) {
            throw new MaachangQException(
                this.stringChannel( bean ) + "は存在しません",
                MaachangQErrorCode.ERROR_NOT_CHANNEL ) ;
        }
        
        return ch ;
    }
    
    /**
     * チャネル情報から、チャネル名を取得.
     */
    private final String stringChannel( AdminChannelBean bean ) {
        
        return new StringBuffer().
            append( "チャネル(name:" ).
            append( bean.getName() ).
            append( " type:" ).
            append(
                ( ( bean.getChannelType() == MaachangQAccessDefine.TYPE_SEND ) ?
                    "送信" : "受信" ) ).
            append( ")" ).
            toString() ;
    }
}

