/*
 * @(#)ClientConnectThread.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.connect.common ;

import java.net.Socket;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.commons.exception.BaseException;
import org.maachang.commons.exception.ExecutionException;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.thread.ExecutionThread;
import org.maachang.commons.thread.LoopThread;
import org.maachang.commons.thread.Synchronized;
import org.maachang.commons.util.CharTable;
import org.maachang.commons.util.UtilCom;
import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.access.net.ConnectObject;
import org.maachang.queue.access.net.ReceiveTimeoutException;
import org.maachang.queue.access.protocol.CommonProtocol;
import org.maachang.queue.connect.admin.login.LoginSession;

/**
 * クライアントコネクションスレッド.
 *
 * @version 2007/01/14
 * @author  Masahito Suzuki
 * @since   MaachangQ 1.00
 */
class ClientConnectThread extends ExecutionThread {
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( ClientConnectThread.class ) ;
    
    /**
     * デフォルトタイムアウト.
     */
    private static final long DEFAULT_TIMEOUT = 60000L ;
    
    /**
     * 未操作タイムアウト.
     */
    private long noWorkTimeout = -1L ;
    
    /**
     * コネクションオブジェクト.
     */
    private ConnectObject connect = null ;
    
    /**
     * クライアント接続処理.
     */
    private ExecutionReceiveClient client = null ;
    
    /**
     * サーバソケットスレッド.
     */
    private ServerConnectThread parent = null ;
    
    /**
     * 受信電文.
     */
    private final Telegram telegram = new Telegram() ;
    
    /**
     * ループスレッド.
     */
    private final LoopThread thread = new LoopThread() ;
    
    /**
     * 同期処理.
     */
    private final Synchronized sync = new Synchronized() ;
    
    /**
     * ログインセッション.
     */
    private LoginSession session = new LoginSession() ;
    
    /**
     * トランザクション管理.
     */
    private final CharTable tranMan = new CharTable() ;
    
    /**
     * 受信ID.
     */
    private final int[] id = new int[ 1 ] ;
    
    /**
     * 電文実行タイプ.
     */
    private final int[] telegramType = new int[ 1 ] ;
    
    /**
     * 最終処理実行時間.
     */
    private long lastAccess = -1L ;
    
    
    /**
     * コンストラクタ.
     */
    private ClientConnectThread(){
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * クライアント接続オブジェクトを生成します.
     * <BR>
     * @param noWorkTimeout 未操作時のタイムアウト値を設定します.
     * @param parent 上位スレッドオブジェクトを設定します.
     * @param socket 対象のサーバソケットを設定します.
     * @param client 実行オブジェクトを設定します.
     */
    public ClientConnectThread(
        long noWorkTimeout,ServerConnectThread parent,Socket socket,
        ExecutionReceiveClient client ){
        
        if( parent == null || parent.isThread() == false ||
            socket == null || socket.isClosed() == true || client == null ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        sync.create() ;
        
        try{
            
            this.noWorkTimeout = noWorkTimeout ;
            this.connect = new ConnectObject( socket ) ;
            this.client = client ;
            this.parent = parent ;
            this.lastAccess = System.currentTimeMillis() ;
            
            thread.create( false,this,null ) ;
            thread.startThread() ;
            
        }catch( Exception e ){
            this.destroy() ;
        }
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * <BR>
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.destroy() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    public final void destroy()
    {
        this.exit( null ) ;
        
        sync.clear() ;
        thread.clear() ;
        
        if( client != null ) {
            try {
                client.exit( session,tranMan ) ;
            } catch( Exception e ) {
            }
        }
        
        if( connect != null ) {
            try {
                connect.destroy() ;
            } catch( Exception e ) {
            }
        }
        
        tranMan.clear() ;
        connect = null ;
        client = null ;
    }
    
    /**
     * 平均通信速度を取得.
     * <BR><BR>
     * 平均通信速度を取得します.
     * <BR>
     * @return long 平均通信速度が返されます.
     */
    public final long getSpeed() {
        long ret = 0 ;
        
        try {
            synchronized( sync.get() ) {
                ret = connect.getSecondByReceiveByte() ;
            }
        } catch( Exception e ) {
            ret = 0L ;
        }
        
        return ret ;
    }
    
    /**
     * スレッド状態を取得.
     * <BR><BR>
     * スレッド状態を取得します.
     * <BR>
     * @return boolean スレッド状態が返されます.<BR>
     *                 [true]が返された場合、スレッドは実行中です.<BR>
     *                 [false]が返された場合、スレッドは停止中です.
     */
    public final boolean isThread()
    {
        boolean ret ;
        
        try{
            synchronized( sync.get() ){
                ret = thread.isThread() ;
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 実行初期化処理をサポートします.
     * <BR><BR>
     * 実行初期化処理をサポートします.<BR>
     * この処理は、スレッド処理が開始された時に呼び出されます.
     * <BR>
     * @param obj 実行開始時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void init( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行終了化処理をサポートします.
     * <BR><BR>
     * 実行終了化処理をサポートします.<BR>
     * この処理は、スレッド処理が終了された時に呼び出されます.
     * <BR>
     * @param obj 実行終了時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void exit( Object obj )
        throws ExecutionException
    {
        
        // クライアント固有終了処理.
        try {
            client.exit( session,tranMan ) ;
        } catch( Exception e ) {
        }
        
        // このオブジェクトを破棄.
        try {
            parent.remove( this ) ;
        } catch( Exception e ) {
        }
        
        // コネクションを破棄.
        try {
            connect.destroy() ;
        } catch( Exception e ) {
        }
        
    }
    
    /**
     * ストップ処理をサポートします。
     * <BR><BR>
     * ストップ処理をサポートします。<BR>
     * この処理は、スレッドでのストップ処理に対して呼び出し実行されます.
     * <BR>
     * @param obj ストップ時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void stop( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行処理をサポートします。
     * <BR><BR>
     * 実行処理をサポートします。<BR>
     * この処理は、スレッドでの実行処理に対して呼び出し実行されます.
     * <BR>
     * @param obj 実行時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void execution( Object obj )
        throws ExecutionException
    {
        long lastAccess = -1L ;
        ConnectObject connect = null ;
        ExecutionReceiveClient client = null ;
        LoginSession session = null ;
        byte[] binary = null ;
        
        try{
            
            //UtilCom.idleTime() ;
            
            synchronized( sync.get() ) {
                lastAccess = this.lastAccess + this.noWorkTimeout ;
                connect = this.connect ;
                client = this.client ;
                session = this.session ;
            }
            
            if( System.currentTimeMillis() >= lastAccess ) {
                throw new ExecutionException(
                    "未操作タイムアウトによる処理クローズ",
                    ExecutionException.LEVEL_STOP
                ) ;
            }
            
            try {
                binary = connect.receive( DEFAULT_TIMEOUT ) ;
            } catch( ReceiveTimeoutException tm ) {
                binary = null ;
            } catch( Exception e2 ) {
                try {
                    connect.destroy() ;
                } catch( Exception e3 ) {
                }
                throw new ExecutionException(
                    e2,ExecutionException.LEVEL_STOP
                ) ;
            }
            
            if( binary != null ) {
                
                int type ;
                
                try {
                    
                    type = CommonProtocol.protocolByCategory( binary ) ;
                    binary = CommonProtocol.analysisProtocol( null,id,binary ) ;
                    
                    telegram.setTelegram( binary ) ;
                    binary = null ;
                    
                } catch( Exception cl ) {
                    LOG.error( "基本電文の解析に失敗",cl ) ;
                    return ;
                }
                
                synchronized( sync.get() ) {
                    this.lastAccess = System.currentTimeMillis() ;
                }
                
                try {
                    telegramType[ 0 ] = -1 ;
                    client.execution( telegramType,session,tranMan,
                        connect,id[ 0 ],type,telegram ) ;
                } catch( Exception cl ) {
                    if( cl instanceof NullPointerException ) {
                        LOG.error( "error",cl ) ;
                    }
                    else if( cl instanceof MaachangQException ) {
                        LOG.warn( new StringBuffer().
                            append( "msg:" ).
                            append( cl.getMessage() ).
                            append( "(code:" ).
                            append( ( ( MaachangQException )cl ).getErrorCode() ).
                            append( " addr:" ).
                            append( connect.getRemoteAddress().getHostAddress() ).
                            append( " port:" ).
                            append( connect.getRemotePort() ).
                            append( ")" ).
                            toString() ) ;
                    }
                    else {
                        LOG.warn( new StringBuffer().
                            append( "msg:" ).
                            append( cl.getMessage() ).
                            append( "(addr:" ).
                            append( connect.getRemoteAddress().getHostAddress() ).
                            append( " port:" ).
                            append( connect.getRemotePort() ).
                            append( ")" ).
                            toString(),cl ) ;
                    }
                    
                    this.resultError( id[ 0 ],telegramType[ 0 ],connect,cl ) ;
                    
                }
            }
            else {
                UtilCom.idleTime() ;
            }
            
        }catch( OutOfMemoryError mem ) {
            LOG.error( "OutOfMemoryError",mem ) ;
        }catch( NullPointerException nul ){
            //LOG.error( "スレッドストップを検知",nul ) ;
            throw new ExecutionException(
                nul,ExecutionException.LEVEL_STOP
            ) ;
        }catch( ExecutionException ee ) {
            //LOG.info( "スレッドストップを検知" ) ;
            throw ee ;
        }catch( BaseException be ){
            LOG.error( "エラーが発生しました", be ) ;
        }catch( Exception e ){
            LOG.error( "エラーが発生しました", e ) ;
        }finally{
            
        }
        
    }
    
    /**
     * エラー発生時にその内容を接続先に返す.
     */
    private final void resultError( int id,int telegramType,
        ConnectObject conn,Exception ex ) {
        
        ServerConnectThread.resultError(
            id,telegramType,null,null,-1,conn,ex ) ;
        
    }
    
}

