/*
 * @(#)SendQueueDataThread.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.channel.service.send ;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.commons.exception.BaseException;
import org.maachang.commons.exception.ExecutionException;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.resource.BinResource;
import org.maachang.commons.resource.cache.CacheDef;
import org.maachang.commons.thread.ExecutionThread;
import org.maachang.commons.thread.LoopThread;
import org.maachang.commons.thread.Synchronized;
import org.maachang.commons.util.UtilCom;
import org.maachang.queue.access.status.ChannelStatus;
import org.maachang.queue.access.status.QueueManagerStatus;
import org.maachang.queue.access.status.QueueStatus;
import org.maachang.queue.main.channel.Channel;
import org.maachang.queue.main.channel.ChannelFactory;
import org.maachang.queue.main.channel.SendChannel;
import org.maachang.queue.main.channel.protocol.ChannelProtocol;
import org.maachang.queue.main.channel.protocol.ProtocolData;
import org.maachang.queue.main.connect.Connect;
import org.maachang.queue.main.connect.ConnectFactory;
import org.maachang.queue.main.manager.QueueManager;
import org.maachang.queue.main.manager.QueueManagerFactory;
import org.maachang.queue.main.queue.QueueDef;
import org.maachang.queue.main.queue.QueueMessage;
import org.maachang.queue.main.queue.SendMqOption;
import org.maachang.queue.main.queue.base.BaseQueue;
import org.maachang.queue.main.queue.base.BaseQueueFactory;
import org.maachang.queue.main.queue.base.core.QArrayChild;

/**
 * コミット済み送信キューデータ送信スレッド.
 *
 * @version 2006/12/21
 * @author  Masahito Suzuki
 * @since   MaachangQ 1.00
 */
public class SendQueueDataThread extends ExecutionThread {
    
    /**
     * 1パケット長.
     */
    private static final int ONE_PACKET = CacheDef.SECTOR_LENGTH / 2 ;
    
    /**
     * デフォルト待機回数.
     */
    private static final int DEF_WAIT_COUNT = 64 ;
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( SendQueueDataThread.class ) ;
    
    
    
    /**
     * スレッド項番.
     */
    private int threadNum = -1 ;
    
    /**
     * コミット済み送信キュー管理テーブル.
     */
    private SendQueueManage sendQueueManage = null ;
    
    /**
     * ループスレッド.
     */
    private final LoopThread thread = new LoopThread() ;
    
    /**
     * 同期処理.
     */
    private final Synchronized sync = new Synchronized() ;
    
    /**
     * プロトコルオブジェクト.
     */
    private final ProtocolData protocol = new ProtocolData() ;
    
    
    /**
     * コンストラクタ.
     */
    private SendQueueDataThread(){
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * コミット済み送信キュー内のデータを送信する処理です.
     * <BR>
     * @param threadNum スレッド項番を設定します.
     * @param sendQueueManage コミット済み送信キュー管理テーブルを設定します.
     */
    public SendQueueDataThread( int threadNum,SendQueueManage sendQueueManage )
        throws InputException {
        
        if( sendQueueManage == null ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            this.threadNum = threadNum ;
            this.sendQueueManage = sendQueueManage ;
            
            thread.create( this ) ;
            thread.startThread() ;
            
        }catch( Exception e ){
            this.destroy() ;
        }
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * <BR>
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.destroy() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    public final void destroy()
    {
        sync.clear() ;
        thread.clear() ;
        threadNum = -1 ;
        sendQueueManage = null ;
        
    }
    
    /**
     * スレッド項番を取得.
     * <BR><BR>
     * スレッド項番を取得します.
     * <BR>
     * @return int スレッド項番が返されます.
     */
    public int getThreadNum() {
        return this.threadNum ;
    }
    
    /**
     * スレッド状態を取得.
     * <BR><BR>
     * スレッド状態を取得します.
     * <BR>
     * @return boolean スレッド状態が返されます.<BR>
     *                 [true]が返された場合、スレッドは実行中です.<BR>
     *                 [false]が返された場合、スレッドは停止中です.
     */
    public final boolean isThread()
    {
        boolean ret ;
        
        try{
            synchronized( sync.get() ){
                ret = thread.isThread() ;
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 実行処理をサポートします。
     * <BR><BR>
     * 実行処理をサポートします。<BR>
     * この処理は、スレッドでの実行処理に対して呼び出し実行されます.
     * <BR>
     * @param obj 実行時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void execution( Object obj )
        throws ExecutionException {
        SendQueueManage sendQueueManage = null ;
        
        try{
            synchronized( sync.get() ) {
                sendQueueManage = this.sendQueueManage ;
            }
            
            // キュー数を取得.
            int len = BaseQueueFactory.getQueueTypeByElementLength(
                QueueDef.TYPE_SEND ) ;
            
            // 送信キューロックオブジェクトを取得.
            LockSendQueue lock = sendQueueManage.getLock() ;
            
            // キュー数が存在する場合.
            if( len > 0 ) {
                
                boolean idleFlag = false ;
                
                // キュー数に合わせて処理.
                for( int i = len-1 ; i >= 0 ; i -- ) {
                    
                    // アイドルタイムを行う必要がある場合.
                    if( idleFlag == true ) {
                        UtilCom.idleTime() ;
                    }
                    
                    idleFlag = false ;
                    
                    // キュー情報を取得.
                    BaseQueue queue = BaseQueueFactory.get( QueueDef.TYPE_SEND,i ) ;
                    
                    try {
                        
                        // キューが処理不可の場合.
                        if( this.checkQueue( queue ) == false ||
                            this.checkQueueByState( queue ) == false ||
                            this.checkQueueManager(
                                queue.getState().getParentManagerName()
                            ) == false ||
                            queue.isCommitMessage() == false ) {
                            
                            UtilCom.idleTime() ;
                            continue ;
                        }
                        
                        synchronized( lock ) {
                            
                            // キューがロックされているかチェック.
                            if( lock.isLock( queue ) == true ) {
                                idleFlag = true ;
                                continue ;
                            }
                            
                            // キューがロックされていない場合はロック処理.
                            lock.lock( queue,threadNum ) ;
                            
                            
                        }
                        
                        // 件数未定でコミット条件のデータを取得.
                        QArrayChild ch = null ;
                        
                        // 送信キュー内の情報が存在しなくなるまで処理.
                        for( ;; ) {
                            
                            boolean exec = false ;
                            
                            // 対象のキューステータスが送信不可の場合.
                            if( this.checkQueueByState( queue ) == false ) {
                                // 処理しない.
                                UtilCom.idleTime() ;
                                break ;
                            }
                            
                            try {
                                
                                // １つの要素を取得.
                                ch = queue.getQArrayChild( true,0 ) ;
                                if( ch == null || ch.isUse() == false ) {
                                    // 送信可能なデータが存在しない場合は、処理しない.
                                    UtilCom.idleTime() ;
                                    break ;
                                }
                                
                                // 処理終了チェック.
                                if( sendQueueManage.checkExitOrStop( queue,ch ) == true ) {
                                    UtilCom.idleTime() ;
                                    continue ;
                                }
                                
                                boolean sendOK = false ;
                                
                                // 送信条件を取得.
                                try {
                                    sendOK = sendQueueManage.isResend( queue,ch ) ;
                                } catch( Exception e ) {
                                    sendOK = false ;
                                    UtilCom.idleTime() ;
                                }
                                
                                // 送信条件の場合.
                                if( sendOK == true ) {
                                    
                                    // 通信処理開始.
                                    exec = true ;
                                    
                                    // チャネル毎に送信処理.
                                    this.sendToChannel( queue,ch,sendQueueManage ) ;
                                    
                                    int sendSize = 0 ;
                                    
                                    // 送信データ長を取得.
                                    if( ch != null && ch.getValue() != null && 
                                        ch.getValue() instanceof QueueMessage ) {
                                        
                                        sendSize = ( ( QueueMessage )ch.getValue() ).getLength() ;
                                        
                                    }
                                    
                                    // 送信完了まで待機.
                                    int waitLen = ( ( sendSize / ONE_PACKET ) * 32 ) + DEF_WAIT_COUNT ;
                                    
                                    // 受信完了まで一定期間待機.
                                    for( int j = 0 ; j < waitLen ; j ++ ) {
                                        if( sendQueueManage.isSendSuccess( queue,ch ) == true ) {
                                            break ;
                                        }
                                        UtilCom.idleTime() ;
                                    }
                                    
                                }
                                else {
                                    UtilCom.idleTime() ;
                                    break ;
                                }
                                
                            
                            } finally {
                                // 処理が行われていた場合のみ、リリース.
                                if( exec == true ) {
                                    sendQueueManage.releaseExecution( ch ) ;
                                }
                            }
                            
                        }
                        
                    } finally {
                        // 送信キューアンロック.
                        lock.unlock( queue ) ;
                    }
                    
                }
                
            }
            else {
                UtilCom.idleTime() ;
            }
            
        } catch( OutOfMemoryError me ) {
            LOG.error( "OutOfMemoryError",me ) ;
        }catch( NullPointerException nul ){
            throw new ExecutionException(
                nul,ExecutionException.LEVEL_STOP
            ) ;
        }catch( BaseException be ){
            LOG.error( "エラーが発生しました[" + threadNum + "]", be ) ;
        }catch( Exception e ){
            LOG.error( "エラーが発生しました[" + threadNum + "]", e ) ;
        }
        
    }
    
    /**
     * チャネル毎に送信処理.
     */
    private void sendToChannel( BaseQueue queue,QArrayChild ch,
        SendQueueManage sendQueueManage )
        throws Exception {
        
        SendQueueParam param = ( SendQueueParam )ch.getOption() ;
        int[] channels = param.getSendChannelId() ;
        int len = channels.length ;
        
        for( int i = 0 ; i < len ; i ++ ) {
            
            // 対象チャネルを取得.
            SendChannel channel = ( SendChannel )ChannelFactory.getToID( channels[ i ] ) ;
            
            // 送信条件でない場合.
            if( channel == null || ( ( Channel )channel ).isChannel() == false ) {
                continue ;
            }
            // チャネルステータスが送信不可能な状態の場合.
            else if( ( ( Channel )channel ).getState() != ChannelStatus.STATE_SUCCESS ) {
                // キューステータスを送信不可に設定.
                queue.getState().setState( QueueStatus.STATE_ERROR ) ;
                break ;
            }
            
            // コネクションを取得.
            Connect conn = ConnectFactory.get( channel.getConnectName() ) ;
            
            // コネクションが存在しない場合は処理しない.
            if( conn == null ) {
                continue ;
            }
            
            // データ送信.
            try {
                
                // 送信処理.
                this.send( param.getSendId(),queue,
                	ch,( Channel )channel,conn,sendQueueManage ) ;
                
            } catch( ExecutionException ee ) {
                throw ee ;
            } catch( Exception e ) {
                LOG.error( "## データ送信処理に失敗[man:" + 
                    queue.getState().getParentManagerName() +
                    " q:" + ( ( SendChannel )channel ).getQueueManager() +
                    " sq:" + queue.getState().getName() +
                    " ch:" + ( ( Channel )channel ).getName() +
                    " dataID:" + ch.getID() + "]",e ) ;
            }
            
        }
        
    }
    
    /**
     * 送信処理.
     */
    private void send( long id,BaseQueue queue,QArrayChild ch,Channel channel,
        Connect conn,SendQueueManage sendQueueManage )
        throws Exception {
    	
    	SendChannel send = ( SendChannel )channel ;
        
        // プロトコルデータに変換.
        protocol.create( id,
        	send.getQueueManager(),
            queue.getState().getParentManagerName(),
            queue.getState().getName(),channel.getName(),ch,
            conn.getPort(),channel.getId(),conn.getMacAddress() ) ;
        
        // 電文を生成.
        BinResource bin = ChannelProtocol.getProtocolDataByTelegram(
            protocol,conn.getResourceType(),false ) ;
        
        // 送信処理.
        conn.sendByResource(
            send.getUseCb32Word(),send.getInetAddress(),
            send.getPort(),bin ) ;
    }
    
    /**
     * キューマネージャの状態が処理不可であるかチェック.
     */
    private final boolean checkQueueManager( String name ) {
        
        // キューマネージャを取得.
        QueueManager man = QueueManagerFactory.get( name ) ;
        
        // キューマネージャステータスが正常以外の場合.
        if( man == null || man.getState() != QueueManagerStatus.STATE_SUCCESS ) {
            return false ;
        }
        
        return true ;
    }
    
    /**
     * キューオブジェクトの状態が処理不可であるかチェック.
     */
    private final boolean checkQueue( BaseQueue queue ) {
        
        // 対象キューに対して、チャネル設定が行われていない場合.
        if( queue.isQueue() == false ||
            queue.getOption() == null ||
            ( ( SendMqOption )queue.getOption() ).size() <= 0 ) {
            return false ;
        }
        
        return true ;
    }
    
    /**
     * 現在のキューステータスが送信不可であるかチェック.
     */
    private final boolean checkQueueByState( BaseQueue queue ) {
        
        // 対象キューのステータスが「正常」、「警告」、
        // 「キュー満杯」以外の場合.
        if( queue.getState().getState() != QueueStatus.STATE_SUCCESS &&
            queue.getState().getState() != QueueStatus.STATE_WARNING &&
            queue.getState().getState() != QueueStatus.STATE_FULL ) {
            return false ;
        }
        
        return true ;
    }
    
}

