/*
 * @(#)Connect.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.connect ;

import java.net.InetAddress;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.commons.exception.ExecutionException;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.exception.TimeoutException;
import org.maachang.commons.net.ConnectAddress;
import org.maachang.commons.net.NetAdapter;
import org.maachang.commons.net.NetConfig;
import org.maachang.commons.net.NetDef;
import org.maachang.commons.resource.BinResource;
import org.maachang.commons.resource.ResourceType;
import org.maachang.commons.resource.cache.CacheDef;
import org.maachang.commons.util.UtilCom;
import org.maachang.connect.Cb32Bean;
import org.maachang.connect.MaachangConnect;
import org.maachang.queue.main.cache.MqCache;
import org.maachang.queue.main.cache.MqCacheFactory;

/**
 * コネクションオブジェクト.
 *  
 * @version 2006/08/30
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
class ConnectImple implements Connect {
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( ConnectImple.class ) ;
    
    /**
     * デフォルト送受信バッファ最低値.
     */
    public static final int DEF_BUFFER = 1048576 ;
    
    /**
     * デフォルトスレッドサイズ.
     */
    public static final int DEF_THREAD_SIZE = 2 ;
    
    /**
     * デフォルトパケットサイズ.
     */
    public static final int DEF_PACKET_SIZE = CacheDef.SECTOR_LENGTH ;
    
    /**
     * デフォルト送信電文ロールサイズ.
     */
    public static final int DEF_SEND_TELEGRAM_ROLL = 128 ;
    
    /**
     * デフォルト受信電文ロールサイズ.
     */
    public static final int DEF_RECV_TELEGRAM_ROLL = 512 ;
    
    /**
     * デフォルト受信データ削除時間.
     */
    public static final int DEF_RECV_DELETE_TIME = 500 ;
    
    /**
     * 送受信バッファ最低値.
     */
    public static final int MIN_BUFFER = CacheDef.MAX_SECTOR ;
    
    
    /**
     * コネクション間通信オブジェクト.
     */
    private MaachangConnect maachangConnect = null ;
    
    /**
     * 通信オブジェクト名.
     */
    private String name = null ;
    
    /**
     * MQキャッシュ名.
     */
    private String mqCacheName = null ;
    
    /**
     * データバッファ長.
     */
    private int buffer = -1 ;
    
    /**
     * マックアドレス.
     */
    private byte[] macAddress = null ;
    
    /**
     * バインド時のInetAddress.
     */
    private InetAddress bindInetAddress = null ;
    
    /**
     * コンストラクタ.
     */
    private ConnectImple() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 条件を設定して通信条件を生成します.
     * <BR>
     * @param name 通信オブジェクト名を設定します.
     * @param mqCacheName MqCache名を設定します.
     * @param port バインドポートを設定します.
     * @param bindAddr バインド先アドレスを設定します.
     * @param buffer 送受信バッファ長を設定します.
     * @param cb32 対象の暗号フラグを設定します.
     * @param cb32Word 対象の暗号ワード群を設定します.
     * @exception InputException 入力例外.
     */
    public ConnectImple( String name,String mqCacheName,int port,InetAddress bindAddr,
        int buffer,boolean cb32,String[] cb32Word )
        throws InputException {
        
        MqCache cache = null ;
        
        if( mqCacheName == null || mqCacheName.length() <= 0 ) {
            throw new InputException( "指定MQキャッシュ名(" + mqCacheName + ")は不正です" ) ;
        }
        else if( ( cache = MqCacheFactory.get( mqCacheName ) ) == null ) {
            throw new InputException( "指定MQキャッシュ名(" + mqCacheName + ")は存在しません" ) ;
        }
        
        ResourceType resType = cache.getResourceType() ;
        
        buffer = ( buffer <= MIN_BUFFER ) ? MIN_BUFFER : buffer ;
        
        Cb32Bean bean = new Cb32Bean() ;
        bean.setCb32( cb32 ) ;
        bean.putWords( cb32Word ) ;
        
        MaachangConnect maachangConnect = null ;
        maachangConnect = new MaachangConnect( bean,port,bindAddr,buffer,resType ) ;
        
        byte[] macAddr = ConnectImple.getMacAddress( maachangConnect ) ;
        
        LOG.info( "name:" + name + " macAddress:" +
            ConnectImple.convertMacBinaryByString( macAddr ) ) ;
        
        this.maachangConnect = maachangConnect ;
        this.name = name ;
        this.mqCacheName = mqCacheName ;
        this.buffer = buffer ;
        this.macAddress = macAddr ;
        this.bindInetAddress = bindAddr ;
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.destroy() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * このオブジェクトを破棄します.
     */
    public void destroy() {
        if( maachangConnect != null ) {
            try {
                maachangConnect.close() ;
            } catch( Exception e ) {
            }
        }
        maachangConnect = null ;
        name = null ;
        buffer = -1 ;
        bindInetAddress = null ;
        macAddress = null ;
    }
    
    /**
     * 送信処理.
     * <BR><BR>
     * 指定アドレスに対して電文を送信します.<BR>
     * また、送信対象の内容はバイナリです.
     * <BR>
     * @param cb32Word 対象の暗号ワードを設定します.
     * @param addr 送信先のアドレスを設定します.
     * @param binary 送信対象の電文を設定します.
     */
    public void sendByBinary( String cb32Word,InetAddress addr,byte[] binary ) {
        this.sendByBinary( cb32Word,addr,DEF_PORT,binary ) ;
    }
    
    /**
     * 送信処理.
     * <BR><BR>
     * 指定アドレスに対して電文を送信します.<BR>
     * また、送信対象の内容はバイナリです.
     * <BR>
     * @param cb32Word 対象の暗号ワードを設定します.
     * @param addr 送信先のアドレスを設定します.
     * @param port 送信先の受信アドレスを設定します.
     * @param binary 送信対象の電文を設定します.
     */
    public void sendByBinary( String cb32Word,InetAddress addr,int port,byte[] binary ) {
        
        if( maachangConnect != null ) {
            maachangConnect.send( addr,port,cb32Word,binary ) ;
        }
        
    }
    
    /**
     * 送信処理.
     * <BR><BR>
     * 指定アドレスに対して電文を送信します.<BR>
     * また、送信対象の内容はバイナリリソースです.
     * <BR>
     * @param cb32Word 対象の暗号ワードを設定します.
     * @param addr 送信先のアドレスを設定します.
     * @param binary 送信対象の電文を設定します.
     */
    public void sendByResource( String cb32Word,InetAddress addr,BinResource binary ) {
        this.sendByResource( cb32Word,addr,DEF_PORT,binary ) ;
    }
    
    /**
     * 送信処理.
     * <BR><BR>
     * 指定アドレスに対して電文を送信します.<BR>
     * また、送信対象の内容はバイナリリソースです.
     * <BR>
     * @param cb32Word 対象の暗号ワードを設定します.
     * @param addr 送信先のアドレスを設定します.
     * @param port 送信先の受信アドレスを設定します.
     * @param binary 送信対象の電文を設定します.
     */
    public void sendByResource( String cb32Word,InetAddress addr,int port,BinResource binary ) {
        
        if( maachangConnect != null ) {
            maachangConnect.send( addr,port,cb32Word,binary ) ;
        }
        
    }
    
    /**
     * 受信処理.
     * <BR><BR>
     * 受け取った電文情報を受信します.<BR>
     * また受信対象はバイナリです.
     * <BR>
     * @param out 受信元もアドレス情報が格納されます.<BR>
     *            [null]を設定した場合、情報は受け取りません.
     * @param cb32Word 対象の暗号ワードが格納されます.<BR>
     *            [null]を設定した場合、情報は受け取りません.
     * @return byte[] 受信されたバイナリ情報が返されます.
     * @exception ExecutionException 実行例外.
     */
    public byte[] recvByBinary( ConnectAddress out,String[] cb32Word )
        throws ExecutionException {
        
        byte[] ret = null ;
        
        if( maachangConnect != null ) {
            
            BinResource res = maachangConnect.receive( out,cb32Word ) ;
            if( res != null ) {
                ret = res.getBinary() ;
            }
            
        }
        
        return ret ;
        
    }
    
    /**
     * 受信処理.
     * <BR><BR>
     * 受け取った電文情報を受信します.<BR>
     * また受信対象はバイナリリソースです.
     * <BR>
     * @param out 受信元もアドレス情報が格納されます.<BR>
     *            [null]を設定した場合、情報は受け取りません.
     * @param cb32Word 対象の暗号ワードが格納されます.<BR>
     *            [null]を設定した場合、情報は受け取りません.
     * @return BinResource 受信されたバイナリ情報が返されます.
     * @exception ExecutionException 実行例外.
     */
    public BinResource recvByResource( ConnectAddress out,String[] cb32Word )
        throws ExecutionException {
        
        BinResource ret = null ;
        
        if( maachangConnect != null ) {
            
            ret = maachangConnect.receive( out,cb32Word ) ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * 受信処理.
     * <BR><BR>
     * 受け取った電文情報を受信します.<BR>
     * また受信対象はバイナリです.
     * <BR>
     * @param out 受信元もアドレス情報が格納されます.<BR>
     *            [null]を設定した場合、情報は受け取りません.
     * @param cb32Word 対象の暗号ワードが格納されます.<BR>
     *            [null]を設定した場合、情報は受け取りません.
     * @param timeout タイムアウト値を設定します.<BR>
     *                [0]以下を設定した場合、無限に待ちます.
     * @return byte[] 受信されたバイナリ情報が返されます.
     * @exception TimeoutException 受信タイムアウト例外.
     * @exception ExecutionException 実行例外.
     */
    public byte[] recvByBinary( ConnectAddress out,String[] cb32Word,int timeout )
        throws TimeoutException,ExecutionException {
        
        byte[] ret = null ;
        
        if( maachangConnect != null ) {
            
            BinResource res = maachangConnect.receive( out,cb32Word,timeout ) ;
            if( res != null ) {
                ret = res.getBinary() ;
            }
            
        }
        
        return ret ;
        
    }
    
    /**
     * 受信処理.
     * <BR><BR>
     * 受け取った電文情報を受信します.<BR>
     * また受信対象はバイナリリソースです.
     * <BR>
     * @param out 受信元もアドレス情報が格納されます.<BR>
     *            [null]を設定した場合、情報は受け取りません.
     * @param cb32Word 対象の暗号ワードが格納されます.<BR>
     *            [null]を設定した場合、情報は受け取りません.
     * @param timeout タイムアウト値を設定します.<BR>
     *                [0]以下を設定した場合、無限に待ちます.
     * @return BinResource 受信されたバイナリ情報が返されます.
     * @exception TimeoutException 受信タイムアウト例外.
     * @exception ExecutionException 実行例外.
     */
    public BinResource recvByResource( ConnectAddress out,String[] cb32Word,int timeout )
        throws TimeoutException,ExecutionException {
        
        BinResource ret = null ;
        
        if( maachangConnect != null ) {
            
            ret = maachangConnect.receive( out,cb32Word,timeout ) ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * 通信オブジェクト名を取得.
     * <BR><BR>
     * 通信オブジェクト名を取得します.
     * <BR>
     * @return String 通信オブジェクト名が返されます.
     */
    public String getName() {
        
        return name ;
        
    }
    
    /**
     * リソースタイプを取得.
     * <BR><BR>
     * リソースタイプを取得します.
     * <BR>
     * @return ResourceType リソースタイプが返されます.
     */
    public ResourceType getResourceType() {
        
        MqCache cache = null ;
        if( ( cache = MqCacheFactory.get( mqCacheName ) ) == null ) {
            return null ;
        }
        
        return cache.getResourceType() ;
        
    }
    
    /**
     * キャッシュ名を取得.
     * <BR><BR>
     * キャッシュ名を取得します.
     * <BR>
     * @return String キャッシュ名が返されます.
     */
    public String getCacheName() {
        return mqCacheName ;
    }
    
    /**
     * 通信バッファ長を取得.
     * <BR><BR>
     * 通信バッファ長が返されます.
     * <BR>
     * @return int 通信バッファ長が返されます.
     */
    public int getBuffer() {
        return buffer ;
    }
    
    /**
     * バインドアドレスを取得.
     * <BR><BR>
     * バインドアドレスを取得します.
     * <BR>
     * @return InetAddress バインドアドレスが返されます.
     */
    public InetAddress getBindAddress() {
        
        InetAddress ret = null ;
        
        if( maachangConnect != null ) {
            
            ret = maachangConnect.getBindAddress() ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * バインドポート番号を取得.
     * <BR><BR>
     * バインドポート番号を取得します.
     * <BR>
     * @return int 送信ポート番号が返されます.
     */
    public int getPort() {
        
        int ret = -1 ;
        
        if( maachangConnect != null ) {
            
            ret = maachangConnect.getBindPort() ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * 暗号ワードを追加.
     * <BR><BR>
     * 暗号ワードを追加します.
     * <BR>
     * @param word 対象の暗号ワードを設定します.
     */
    public void addCb32Word( String word ) {
        maachangConnect.addWord( word ) ;
    }
    
    /**
     * 暗号ワードを削除.
     * <BR><BR>
     * 暗号ワードを削除します.
     * <BR>
     * @param word 削除対象の暗号ワードを設定します.
     */
    public void removeCb32Word( String word ) {
        maachangConnect.removeWord( word ) ;
    }
    
    /**
     * 暗号ワード群を取得.
     * <BR><BR>
     * 暗号ワード群を取得します.
     * <BR>
     * @return String[] 暗号ワードが返されます.
     */
    public String[] getCb32Word() {
        return maachangConnect.getWords() ;
    }
    
    /**
     * 暗号ワード数を取得.
     * <BR><BR>
     * 暗号ワード数を取得します.
     * <BR>
     * @return int 暗号ワード数が返されます.
     */
    public int getCb32WordSize() {
        return maachangConnect.getWordSize() ;
    }
    
    /**
     * コネクションBeanを取得.
     * <BR><BR>
     * コネクションBeanを取得します.
     * <BR>
     * @return ConnectBean コネクションBeanが返されます.
     */
    public ConnectBean getConnectBean() {
        
        ConnectBean ret = null ;
        
        if( maachangConnect != null ) {
            ret = new ConnectBean() ;
            if( this.bindInetAddress == null ) {
                ret.setBindAddress( null ) ;
            }
            else {
                ret.setBindAddress( this.bindInetAddress.getHostName() ) ;
            }
            ret.setBufferLength( this.buffer ) ;
            ret.setConnectObjectName( this.name ) ;
            ret.setMqCacheName( this.mqCacheName ) ;
            ret.setPort( this.maachangConnect.getBindPort() ) ;
            ret.setCb32( this.maachangConnect.isCb32() ) ;
            ret.setCb32Word( this.maachangConnect.getWords() ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * 暗号モードを設定.
     * <BR><BR>
     * 暗号モードを設定します.
     * <BR>
     * @param mode 暗号モードを設定します.
     */
    public void setCb32( boolean mode ) {
        if( maachangConnect != null ) {
            maachangConnect.setCb32( mode ) ;
        }
    }
    
    /**
     * バインドアドレスを取得.
     * <BR><BR>
     * バインド時に設定されたInetAddressが返されます.
     * <BR>
     * @return InetAddress バインドInetAddressが返されます.
     */
    public InetAddress getBindInetAddress() {
        return this.bindInetAddress ;
    }
    
    /**
     * バインド先のMACアドレスを取得.
     * <BR><BR>
     * バインド先のMACアドレスを取得します.
     * <BR>
     * @return byte[] MACアドレスが返されます.
     */
    public byte[] getMacAddress() {
        return macAddress ;
    }
    
    /**
     * 暗号モードを取得.
     * <BR><BR>
     * 暗号モードを取得します.
     * <BR>
     * @return boolean 暗号モードが返されます.<BR>
     *                 [true]が返された場合、暗号モードはONです.<BR>
     *                 [false]が返された場合、暗号モードはOFFです.
     */
    public boolean isCb32() {
        
        if( maachangConnect != null ) {
            return maachangConnect.isCb32() ;
        }
        
        return false ;
        
    }
    
    /**
     * オブジェクトが利用可能かチェック.
     * <BR><BR>
     * オブジェクトが利用可能かチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、オブジェクトは利用可能です.<BR>
     *                 [false]が返された場合、オブジェクトは利用不可能です.
     */
    public boolean isUseObject() {
        
        boolean ret = false ;
        
        if( maachangConnect != null ) {
            
            ret = maachangConnect.isOpen() ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * 通信元オブジェクトを取得.
     * <BR><BR>
     * 通信元オブジェクトを取得します.
     * <BR>
     * @return MaachangConnect 通信元オブジェクトが返されます.
     */
    public MaachangConnect getNativeConnect() {
        return maachangConnect ;
    }
    
    
    
    /**
     * Localhost.
     */
    private static final String LOCAL_ADDR = "127.0.0.1" ;
    
    /**
     * バインドIP指定なし.
     */
    private static final String NOT_BIND_ADDR = "0.0.0.0" ;
    
    /**
     * MACアドレスサイズ.
     */
    private static final int MAC_LENGTH = NetDef.MAC_ADDRESS_LENGTH ;
    
    /**
     * MACアドレスを取得.
     */
    private static final byte[] getMacAddress( MaachangConnect maachangConnect ) {
        
        if( maachangConnect == null || maachangConnect.getBindAddress() == null ) {
            return null ;
        }
        
        String mac = null ;
        
        // ネットワークアダプタを取得.
        NetAdapter adp = NetConfig.getInstance().getNetAdapter() ;
        
        // バインド情報指定なしの場合.
        String hostAddr = maachangConnect.getBindAddress().getHostAddress() ;
        
        // バインド情報がLocalHostの場合.
        if( LOCAL_ADDR.equals( hostAddr ) ) {
            
            // ff:ff:ff:ff:ff:ff を返す.
            return new byte[] {
                ( byte )0xff,( byte )0xff,( byte )0xff,
                ( byte )0xff,( byte )0xff,( byte )0xff,
            } ;
            
        }
        // バインド先が指定されていない場合.
        else if( NOT_BIND_ADDR.equals( hostAddr ) ) {
            
            // IPが存在するMACアドレスを正とする.
            String[] names = adp.getNames() ;
            if( names != null && names.length > 0 ) {
                int len = names.length ;
                for( int i = 0 ; i < len ; i ++ ) {
                    
                    int addrLen = adp.getElement(
                        names[ i ],NetAdapter.CATEGORY_IPADDR ) ;
                    
                    if( addrLen > 0 ) {
                        mac = adp.getCategory(
                            names[ i ],NetAdapter.CATEGORY_MACADDR,0 ) ;
                        if( mac != null && mac.length() > 0 ) {
                            return convertMacStringByBinary( mac ) ;
                        }
                    }
                    
                }
            }
            
        }
        // バインド先が指定されている場合.
        else {
            
            // IPが一致するMACアドレスを正とする.
            String[] names = adp.getNames() ;
            if( names != null && names.length > 0 ) {
                int len = names.length ;
                for( int i = 0 ; i < len ; i ++ ) {
                    
                    int lenJ = adp.getElement(
                        names[ i ],NetAdapter.CATEGORY_IPADDR ) ;
                    
                    for( int j = 0 ; j < lenJ ; j ++ ) {
                        
                        if( hostAddr.equals( adp.getCategory(
                                names[ i ],NetAdapter.CATEGORY_IPADDR,j ) ) ) {
                            
                            mac = adp.getCategory(
                                names[ i ],NetAdapter.CATEGORY_MACADDR,0 ) ;
                            
                            if( mac != null && mac.length() > 0 ) {
                                return convertMacStringByBinary( mac ) ;
                            }
                        }
                        
                    }
                }
            }
        }
        
        return null ;
    }
    
    /**
     * 取得したMACアドレスを解析.
     */
    private static final byte[] convertMacStringByBinary( String mac ) {
        
        byte[] ret = UtilCom.convert16StringToBinary( true,mac ) ;
        if( ret == null || ret.length != MAC_LENGTH ) {
            return null ;
        }
        return ret ;
    }
    
    /**
     * MACアドレスを文字列変換.
     */
    private static final String convertMacBinaryByString( byte[] bin ) {
        if( bin != null && bin.length == MAC_LENGTH ) {
            StringBuffer buf = new StringBuffer() ;
            for( int i = 0 ; i < MAC_LENGTH ; i ++ ) {
                if( i != 0 ) {
                    buf.append( ":" ) ;
                }
                buf.append( Integer.toHexString(
                    ( int )( bin[ i ] & 0x000000ff ) ) ) ;
            }
            
            return buf.toString() ;
        }
        return "" ;
    }
}

