/*
 * @(#)QueueManaegrFactory.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.manager ;

import org.maachang.commons.exception.AccessException;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.sys.NamingManager;
import org.maachang.commons.util.SearchPortion;

/**
 * キューマネージャファクトリ.
 *  
 * @version 2006/08/30
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
public class QueueManagerFactory
{
    
    /**
     * ネーミングマネージャ登録拡張子.
     */
    public static final String NAMING_PLUS = "@maachangq.manager.factory" ;
    
    /**
     * 同期オブジェクト.
     */
    private static final Object SYNC = new Object() ;
    
    /**
     * コンストラクタ.
     */
    public QueueManagerFactory() {
        
    }
    
    /**
     * キューマネージャを全て破棄.
     * <BR><BR>
     * キューマネージャ情報を全て破棄します.
     */
    public final void destroy() {
        
        synchronized( SYNC ) {
            
            int len ;
            String[] names = QueueManagerFactory.getNames() ;
            
            if( names != null && ( len = names.length ) > 0 ) {
                for( int i = 0 ; i < len ; i ++ ) {
                    try {
                        QueueManagerFactory.remove( false,names[ i ] ) ;
                    } catch( Exception e ) {
                    }
                }
            }
            
        }
    }
    
    /**
     * キューマネージャ生成.
     * <BR><BR>
     * 条件を設定してキューマネージャを生成します.
     * <BR>
     * @param bean 対象のQueueManagerBeanを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final void create( QueueManagerBean bean )
        throws InputException,AccessException {
        
        if( bean == null || bean.getQueueManagerName() == null ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        QueueManagerFactory.create(
            bean.getQueueManagerName(),bean.getCacheSize() ) ;
        
    }
    
    /**
     * キューマネージャ生成.
     * <BR><BR>
     * 条件を設定してキューマネージャを生成します.
     * <BR>
     * @param naming キューマネージャ名を設定します.
     * @param size キャッシュセクタ数を設定します.<BR>
     *             設定可能な最小値は[MqDefine.MIN_CACHE_SECTOR]です.<BR>
     *             設定可能な最大値は[MqDefine.MAX_CACHE_SECTOR]です.<BR>
     *             [-1]を設定した場合、キャッシュファイルは生成されません.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final void create( String naming,int size )
        throws InputException,AccessException {
        
        String key = null ;
        BaseQueueManagerImple imple = null ;
        
        if(
            naming == null || ( naming = naming.trim().toLowerCase() ).length() <= 0 ||
            ( key = QueueManagerFactory.createNaming( naming ) ) == null ) {
            throw new InputException( "引数は不正です" ) ;
        }
        else if( NamingManager.get( key ) != null ) {
            throw new InputException( "対象のキューマネージャ名[" + naming +
                "]は既に登録されています" ) ;
        }
        
        synchronized( SYNC ) {
            
            imple = new BaseQueueManagerImple( naming,size ) ;
            NamingManager.add( key,imple ) ;
            
        }
        
    }
    
    /**
     * キューマネージャ削除.
     * <BR><BR>
     * キューマネージャオブジェクトを削除します.
     * <BR>
     * @param naming 削除対象のネーミングを設定します.
     */
    public static final void remove( String naming ) {
        
        QueueManagerFactory.remove( false,naming ) ;
        
    }
    
    /**
     * キューマネージャ削除.
     * <BR><BR>
     * キューマネージャオブジェクトを削除します.
     * <BR>
     * @param delete 削除モードを設定します.<BR>
     *               [true]を設定した場合、実際のキャッシュファイルも削除します.<BR>
     *               [false]を設定した場合、実際のキャッシュファイルは削除されません.
     * @param naming 削除対象のネーミングを設定します.
     */
    public static final void remove( boolean delete,String naming ) {
        
        Object o = null ;
        
        if( ( naming = QueueManagerFactory.createNaming( naming ) ) == null ) {
            return ;
        }
        
        synchronized( SYNC ) {
            
            o = NamingManager.get( naming ) ;
            
            if( o != null && o instanceof BaseQueueManagerImple ) {
                
                NamingManager.remove( naming ) ;
                
                if( delete == true ) {
                    ( ( BaseQueueManagerImple )o ).deleteCacheToDestroy() ;
                }
                
            }
            
        }
        
    }
    
    /**
     * キューマネージャ取得.
     * <BR><BR>
     * キューマネージャオブジェクトを取得します.
     * <BR>
     * @param naming 取得対象のネーミングを設定します.
     * @return QueueManaegr キューマネージャ情報が返されます.
     */
    public static final QueueManager get( String naming ) {
        
        Object o = null ;
        QueueManager ret = null ;
        
        if( ( naming = QueueManagerFactory.createNaming( naming ) ) == null ) {
            return null ;
        }
        
        synchronized( SYNC ) {
            
            o = NamingManager.get( naming ) ;
            if( o != null && o instanceof QueueManager ) {
                ret = ( QueueManager )o ;
            }
            
        }
        
        return ret ;
        
    }
    
    /**
     * 登録されているキューマネージャ名一覧を取得.
     * <BR><BR>
     * 登録されているキューマネージャ名一覧を取得します.
     * <BR>
     * @return String[] キューマネージャ名一覧が返されます.
     */
    public static final String[] getNames() {
        
        int i ;
        int len ;
        int namingLen ;
        
        String[] ret = null ;
        
        synchronized( SYNC ) {
            
            ret = SearchPortion.searchString(
                NamingManager.getNames(),
                new StringBuffer().append( "*" ).append( NAMING_PLUS ).toString() ) ;
            
            if( ret != null && ( len = ret.length ) > 0 ) {
                namingLen = NAMING_PLUS.length() ;
                for( i = 0 ; i < len ; i ++ ) {
                    
                    ret[ i ] = ret[ i ].substring( 0,ret[ i ].length() - namingLen ) ;
                    
                }
            }
            
        }
        
        return ret ;
        
    }
    
    /**
     * 登録されている情報長を取得.
     * <BR><BR>
     * 登録されている情報長を取得します.
     * <BR>
     * @return int 登録されている情報長が返されます.
     */
    public static final int size() {
        
        int ret = 0 ;
        
        synchronized( SYNC ) {
            
            String[] names = SearchPortion.searchString(
                NamingManager.getNames(),
                new StringBuffer().append( "*" ).append( NAMING_PLUS ).toString() ) ;
            
            if( names != null ) {
                ret = names.length ;
            }
            
        }
        
        return ret ;
        
    }
    
    /**
     * 同期オブジェクトを取得.
     * <BR><BR>
     * 同期オブジェクトを取得します.
     * <BR>
     * @return Object 同期オブジェクトが返されます.
     */
    public static final Object getSync() {
        return SYNC ;
    }
    
    
    
    /**
     * キューマネージャファクトリ名生成.
     */
    private static final String createNaming( String name ) {
        
        if( name == null || ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            return null ;
        }
        
        return new StringBuffer().append( name ).append( NAMING_PLUS ).toString() ;
        
    }
}

