/*
 * @(#)QArray.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.queue.base.core ;

import org.maachang.commons.exception.InputException;
import org.maachang.commons.thread.Synchronized;
import org.maachang.commons.util.SequenceID;
import org.maachang.commons.util.UtilCom;

/**
 * キュー配列オブジェクト.
 * <BR><BR>
 * キュー配列を示します.<BR>
 * このオブジェクトは、キュー配列子要素を追加順に管理しますが、
 * 通常の配列とは違い、キュー配列子要素側で、クリアした場合、
 * キュー配列管理テーブルから、その情報を削除してくれます.
 *  
 * @version 2005/12/29
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
public class QArray
{
    
    /**
     * キュー配列を示すオブジェクト.
     */
    protected final QArrayChild m_table = new QArrayChild() ;
    
    /**
     * 最後の配列を示すオブジェクト.
     */
    protected QArrayChild m_last = null ;
    
    /**
     * 格納データ数.
     */
    protected int m_size = 0 ;
    
    /**
     * 削除件数.
     */
    protected int m_deleteSize = 0 ;
    
    /**
     * オートコミットフラグ.
     */
    protected boolean m_autoCommitFlag = false ;
    
    /**
     * シーケンスID.
     */
    protected final SequenceID m_seq = new SequenceID() ;
    
    /**
     * 同期オブジェクト.
     */
    protected Synchronized m_sync = null ;
    
    /**
     * コンストラクタ.
     */
    public QArray()
    {
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オブジェクト生成.
     * <BR><BR>
     * オブジェクト情報を生成します.
     * <BR>
     * @param sync 同期オブジェクトを設定します.
     * @exception InputException 入力例外.
     */
    public void create( Synchronized sync )
        throws InputException
    {
        this.create( sync,false ) ;
    }
    
    /**
     * オブジェクト生成.
     * <BR><BR>
     * オブジェクト情報を生成します.
     * <BR>
     * @param sync 同期オブジェクトを設定します.
     * @param autoCommit オートコミットフラグを設定します.<BR>
     *                   [true]を設定した場合、オートコミットはONとなります.<BR>
     *                   [false]を設定した場合、オートコミットはOFFとなります.
     * @exception InputException 入力例外.
     */
    public void create( Synchronized sync,boolean autoCommit )
        throws InputException
    {
        if( sync == null || sync.get() == null ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.clear() ;
        m_sync = sync ;
        
        try{
            synchronized( m_sync.get() ){
                m_seq.create() ;
                m_table.m_next = null ;
                m_table.m_before = null ;
                m_last = null ;
                m_size = 0 ;
                m_deleteSize = 0 ;
                m_autoCommitFlag = autoCommit ;
            }
        }catch( Exception e ){
        }
        
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 対象の情報をクリアします.
     */
    public void clear()
    {
        boolean flg = false ;
        QArrayChild tmp = null ;
        QArrayChild tmp2 = null ;
        
        try{
            synchronized( m_sync.get() ){
                
                tmp = m_table.m_next ;
                
                for( ;; ){
                    
                    tmp2 = tmp.m_next ;
                    
                    tmp.m_before = null ;
                    tmp.m_next = null ;
                    tmp.m_value[ 0 ] = null ;
                    tmp.m_value[ 1 ] = null ;
                    tmp.m_key[ 0 ] = null ;
                    tmp.m_key[ 1 ] = null ;
                    tmp.m_qarray = null ;
                    tmp.m_sync = null ;
                    
                    if( tmp2 == null ){
                        break ;
                    }
                    
                    tmp = tmp2 ;
                    tmp2 = null ;
                    
                }
                
                m_table.m_next = null ;
                m_table.m_before = null ;
                m_last = null ;
                m_size = 0 ;
                m_deleteSize = 0 ;
                m_autoCommitFlag = false ;
                m_sync = null ;
                flg = true ;
                
            }
        }catch( Exception e ){
        }finally{
            tmp = null ;
            tmp2 = null ;
        }
        
        if( flg == false ){
            m_table.m_next = null ;
            m_table.m_before = null ;
            m_last = null ;
            m_size = 0 ;
            m_deleteSize = 0 ;
            m_autoCommitFlag = false ;
            m_sync = null ;
        }
        
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 一番最後の配列に情報を追加します.
     * <BR>
     * @param value 対象の情報を追加します.
     * @param priority 対象の優先順位を設定します.<BR>
     *                 設定できる最小値は[0]です.<BR>
     *                 設定できる最大値は[9500]です.
     * @param expire 対象のExpire値を設定します.<BR>
     *               [0]以下を設定した場合、未設定となります.
     * @return QArrayChild 追加対象のキュー配列子要素が返されます.
     */
    public QArrayChild add( Object value,int priority,long expire )
    {
        QArrayChild tmp = null ;
        QArrayChild ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                
                // 追加情報を生成.
                ret = new QArrayChild( this,value,priority,expire ) ;
                
                // データが0件の場合.
                if( m_last == null ){
                    
                    // 初めの条件に設定.
                    m_table.m_next = ret ;
                    ret.m_before = m_table ;
                    m_last = ret ;
                    
                }
                // 最後に追加した情報より、指定プライオリティ値が
                // 同じか、小さい場合.
                else if( m_last.m_priority >= priority ){
                    
                    m_last.m_next = ret ;
                    ret.m_before = m_last ;
                    m_last = ret ;
                    
                }
                // 先頭情報が存在する場合.
                else if( ( tmp = m_table.m_next ) != null ){
                    
                    // 先頭情報よりもプライオリティ値が高い.
                    if( tmp.m_priority < priority ){
                        
                        // 送信中の場合.
                        if( tmp.m_sendID > 0L ) {
                            
                            // 未送信の情報の前に設定.
                            for( ;; ) {
                                
                                // 情報が存在する.
                                if( ( tmp = tmp.m_next ) != null ) {
                                    
                                    // 未送信情報の場合.
                                    if( tmp.m_sendID <= 0L ) {
                                        
                                        tmp.m_before = ret ;
                                        ret.m_next = tmp ;
                                        ret.m_before = m_table ;
                                        m_table.m_next = ret ;
                                        break ;
                                    }
                                    
                                }
                                // すべての情報が、送信対象の場合.
                                else {
                                    
                                    // 一番最後に追加.
                                    m_last.m_next = ret ;
                                    ret.m_before = m_last ;
                                    m_last = ret ;
                                    break ;
                                }
                                
                            }
                        }
                        // 先頭が未送信状態の場合.
                        else {
                            tmp.m_before = ret ;
                            ret.m_next = tmp ;
                            ret.m_before = m_table ;
                            m_table.m_next = ret ;
                        }
                        
                    }
                    // 先頭のプライオリティの方が、最後のプライオリティよりも、大きい場合.
                    else if( ( tmp.m_priority - priority ) > ( priority - m_last.m_priority ) ){
                        
                        // 一番最後から、プライオリティの当てはめられる
                        // 位置条件を探す.
                        tmp = m_last.m_before ;
                        
                        for( ;; ){
                            
                            // 情報が存在する場合.
                            if( tmp != null ) {
                                
                                // 対象条件が未送信で、条件よりも
                                // 指定プライオリティ値が同じか、小さい場合.
                                if( tmp.m_sendID <= 0L &&
                                    tmp.m_priority >= priority ){
                                    
                                    ret.m_next = tmp.m_next ;
                                    ret.m_before = tmp ;
                                    tmp.m_next.m_before = ret ;
                                    tmp.m_next = ret ;
                                    break ;
                                    
                                }
                                
                                // 前位置に遷移.
                                tmp = tmp.m_before ;
                                
                            }
                            // 情報が存在しない場合.
                            else {
                                // 一番最後に追加.
                                m_last.m_next = ret ;
                                ret.m_before = m_last ;
                                m_last = ret ;
                                break ;
                            }
                            
                        }
                        
                    }
                    // 最後のプライオリティ値の方が、大きい場合.
                    else{
                        
                        // 一番最初から、プライオリティの当てはめられる
                        // 位置条件を探す.
                        tmp = tmp.m_next ;
                        
                        for( ;; ){
                            
                            // 情報が存在する場合.
                            if( tmp != null ) {
                                
                                // 対象条件が未送信で、条件よりも
                                // 指定プライオリティ値が同じか、大きい場合.
                                if( tmp.m_sendID <= 0L &&
                                    tmp.m_priority < priority ){
                                    ret.m_before = tmp.m_before ;
                                    ret.m_next = tmp ;
                                    tmp.m_before.m_next = ret ;
                                    tmp.m_before = ret ;
                                    break ;
                                }
                                
                                // 次の位置に移動.
                                tmp = tmp.m_next ;
                                
                            }
                            // 情報が存在しない場合.
                            else {
                                // 一番最後に追加.
                                m_last.m_next = ret ;
                                ret.m_before = m_last ;
                                m_last = ret ;
                                break ;
                            }
                            
                        }
                        
                    }
                    
                }
                
                m_size ++ ;
                
            }
        }catch( Exception e ){
        }
        
        return ret ;
        
    }
    
    /**
     * 対象情報を削除.
     * <BR><BR>
     * 指定された項番の情報を削除します.
     * <BR>
     * @param no 削除対象の項番を設定します.
     * @return QArrayChild 削除された情報が返されます.<BR>
     *                     情報が存在しない場合[null]が返されます.
     */
    public QArrayChild remove( int no )
    {
        QArrayChild ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = this.get( no ) ;
                if( m_autoCommitFlag == true ){
                    ret.cut() ;
                }
                else{
                    ret.delete() ;
                }
            }
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象情報を取得.
     * <BR><BR>
     * 指定された項番の情報を取得します.
     * <BR>
     * @param no 取得対象の項番を設定します.
     * @return QArrayChild 指定項番に対する情報が返されます.<BR>
     *                     情報が存在しない場合[null]が返されます.
     */
    public QArrayChild get( int no )
    {
        int i,cnt ;
        int len ;
        int sz ;
        
        QArrayChild ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                
                if( m_autoCommitFlag == true ){
                    
                    len = m_size ;
                    
                    if( no < 0 || no >= len ){
                        ret = null ;
                    }
                    else if( no + 1 == len ){
                        ret = m_last ;
                    }
                    else{
                        
                        ret = m_table ;
                        
                        for( i = 0 ; i < len ; i ++ ){
                            
                            if( ( ret = ret.m_next ) == null || i >= no ){
                                break ;
                            }
                            
                        }
                        
                    }
                }
                else{
                    
                    sz = m_size - m_deleteSize ;
                    
                    if( no < 0 || no >= sz ){
                        ret = null ;
                    }
                    else if(
                        no + 1 == sz &&
                        (
                            m_last.m_state[ 0 ] == QArrayChild.STATE_CREATE ||
                            m_last.m_state[ 0 ] == QArrayChild.STATE_TO_COMMIT
                        )
                    )
                    {
                        ret = m_last ;
                    }
                    else{
                        
                        ret = m_table.m_next ;
                        len = m_size ;
                        
                        for( i = 0,cnt = 0 ; i < len ; i ++ ){
                            
                            if( ret == null ){
                                break ;
                            }
                            else if(
                                ret.m_state[ 0 ] == QArrayChild.STATE_CREATE ||
                                ret.m_state[ 0 ] == QArrayChild.STATE_TO_COMMIT
                            )
                            {
                                cnt ++ ;
                                if( cnt > no ){
                                    break ;
                                }
                            }
                            
                            ret = ret.m_next ;
                            
                        }
                        
                    }
                }
            }
            
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 現在格納数を取得.
     * <BR><BR>
     * 現在データが格納されているサイズを取得します.
     * <BR>
     * @return int 現在格納されているサイズが返されます.
     */
    public int size()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_size - m_deleteSize ;
            }
        }catch( Exception e ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * 新しいIDを取得.
     * <BR><BR>
     * 新しいID情報を取得します.
     * <BR>
     * @return long 新しいID情報が返されます.
     */
    protected long getID() {
        
        long ret = -1L ;
        
        try {
            synchronized( m_sync.get() ) {
                
                ret = (
                    ( ( m_seq.getID() & 0x000000007fffffffL ) << 32L ) |
                    ( UtilCom.getTimeByInteger() & 0x00000000ffffffffL )
                ) ;
                
            }
            
        } catch( Exception e ) {
            ret = -1L ;
        }
        
        return ret ;
        
    }
    
    /**
     * 文字列に変換.
     * <BR><BR>
     * 対象条件を文字列に変換します.
     * <BR>
     * @return String 変換された文字列が返されます.
     */
    public String toString()
    {
        int i ;
        QArrayChild ch = null ;
        
        StringBuffer buf = null ;
        String ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                
                buf = new StringBuffer() ;
                ch = m_table.m_next ;
                
                for( i = 0 ;; i ++ ){
                    
                    if( ch == null ){
                        break ;
                    }
                    
                    buf.append( " [" ) ;
                    buf.append( (i+1) ) ;
                    buf.append( "]:{" ) ;
                    buf.append( ch ) ;
                    buf.append( "}" ) ;
                    
                    ch = ch.m_next ;
                    
                }
                
            }
            
            ret = buf.toString() ;
            ret = ( ret == null || ret.length() <= 0 ) ? "null" : ret ;
            
        }catch( Exception e ){
            ret = "null" ;
        }finally{
            buf = null ;
        }
        
        return ret ;
    }
}

