/*
 * @(#)MonitorQueueStateThread.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.queue.service ;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.commons.exception.BaseException;
import org.maachang.commons.exception.ExecutionException;
import org.maachang.commons.thread.ExecutionThread;
import org.maachang.commons.thread.LoopThread;
import org.maachang.commons.thread.Synchronized;
import org.maachang.commons.util.UtilCom;
import org.maachang.queue.access.MaachangQAccessDefine;
import org.maachang.queue.access.status.ChannelStatus;
import org.maachang.queue.access.status.QueueStatus;
import org.maachang.queue.access.status.StatusDefine;
import org.maachang.queue.main.channel.Channel;
import org.maachang.queue.main.channel.ChannelFactory;
import org.maachang.queue.main.queue.SendMqOption;
import org.maachang.queue.main.queue.base.BaseQueue;
import org.maachang.queue.main.queue.base.BaseQueueFactory;

/**
 * キューステータス監視スレッド.
 *  
 * @version 2007/01/25
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
class MonitorQueueStateThread extends ExecutionThread {
    
    /**
     * 待機時間
     */
    private static final int WAIT_TIME = 250 ;
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( MonitorQueueStateThread.class ) ;
    
    /**
     * ループスレッド.
     */
    private final LoopThread thread = new LoopThread() ;
    
    /**
     * 同期処理.
     */
    private final Synchronized sync = new Synchronized() ;
    
    
    /**
     * コンストラクタ.
     */
    public MonitorQueueStateThread(){
        
        sync.create() ;
        
        try{
            
            thread.create( this ) ;
            thread.startThread() ;
            
        }catch( Exception e ){
            this.destroy() ;
        }
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * <BR>
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception {
        
        try{
            this.destroy() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    public final void destroy() {
        sync.clear() ;
        thread.clear() ;
    }
    
    /**
     * すべてのキューステータスを、停止中に設定.
     * <BR><BR>
     * すべてのキューステータスを、停止中に設定します.
     */
    public final void shutdownQueue() {
        
        int type = MaachangQAccessDefine.TYPE_RECEIVE ;
        
        for( int i = 0 ;; i ++ ) {
            
            BaseQueue queue = BaseQueueFactory.get( type,i ) ;
            
            if( queue == null ) {
                if( type == MaachangQAccessDefine.TYPE_RECEIVE ) {
                    type = MaachangQAccessDefine.TYPE_SEND ;
                    i = -1 ;
                    continue ;
                }
                break ;
            }
            else if( queue.isQueue() == false ) {
                continue ;
            }
            
            queue.getState().setState( QueueStatus.STATE_SHUTDOWN ) ;
            
        }
    }
    
    /**
     * スレッド状態を取得.
     * <BR><BR>
     * スレッド状態を取得します.
     * <BR>
     * @return boolean スレッド状態が返されます.<BR>
     *                 [true]が返された場合、スレッドは実行中です.<BR>
     *                 [false]が返された場合、スレッドは停止中です.
     */
    public final boolean isThread()
    {
        boolean ret ;
        
        try{
            synchronized( sync.get() ){
                ret = thread.isThread() ;
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    
    
    /**
     * 実行初期化処理をサポートします.
     * <BR><BR>
     * 実行初期化処理をサポートします.<BR>
     * この処理は、スレッド処理が開始された時に呼び出されます.
     * <BR>
     * @param obj 実行開始時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void init( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行終了化処理をサポートします.
     * <BR><BR>
     * 実行終了化処理をサポートします.<BR>
     * この処理は、スレッド処理が終了された時に呼び出されます.
     * <BR>
     * @param obj 実行終了時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void exit( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * ストップ処理をサポートします。
     * <BR><BR>
     * ストップ処理をサポートします。<BR>
     * この処理は、スレッドでのストップ処理に対して呼び出し実行されます.
     * <BR>
     * @param obj ストップ時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void stop( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行処理をサポートします。
     * <BR><BR>
     * 実行処理をサポートします。<BR>
     * この処理は、スレッドでの実行処理に対して呼び出し実行されます.
     * <BR>
     * @param obj 実行時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void execution( Object obj )
        throws ExecutionException {
        
        try{
            
            int i ;
            
            // 受信キューのステータスチェック.
            for( i = 0 ;; i ++ ) {
                
                UtilCom.idleSleep( WAIT_TIME ) ;
                
                // 受信キューを取得.s
                BaseQueue queue = BaseQueueFactory.get(
                    MaachangQAccessDefine.TYPE_RECEIVE,i ) ;
                
                if( queue == null ) {
                    break ;
                }
                else if( queue.isQueue() == false ) {
                    continue ;
                }
                
                // ステータスが「開始中」および「エラー」の場合.
                if( queue.getState().getState() == QueueStatus.STATE_STARTUP ||
                    queue.getState().getState() == QueueStatus.STATE_ERROR ) {
                    
                    // ステータスを正常に設定.
                    queue.getState().setState( QueueStatus.STATE_SUCCESS ) ;
                    queue.checkStatus() ;
                    
                }
                
            }
            
            // 送信キューのステータスチェック.
            for( i = 0 ;; i ++ ) {
                
                UtilCom.idleSleep( WAIT_TIME ) ;
                
                // 送信キューを取得.
                BaseQueue queue = BaseQueueFactory.get(
                    MaachangQAccessDefine.TYPE_SEND,i ) ;
                
                if( queue == null ) {
                    break ;
                }
                else if( queue.isQueue() == false ) {
                    continue ;
                }
                
                SendMqOption opt = ( SendMqOption )queue.getOption() ;
                
                if( opt == null ) {
                    continue ;
                }
                
                int len = opt.size() ;
                boolean channelError = false ;
                boolean end = false ;
                
                // 送信キュー内のチャネル状態チェック.
                if( len > 0 ) {
                    for( int j = 0 ; j < len ; j ++ ) {
                        
                        UtilCom.idleSleep( WAIT_TIME ) ;
                        
                        // キュー状態が削除状態の場合.
                        if( queue.isQueue() == false ) {
                            // このキューは処理しないようにする.
                            end = true ;
                            break ;
                        }
                        
                        UtilCom.idleTime() ;
                        
                        // １つの送信チャネルを取得.
                        Channel channel = ChannelFactory.get(
                            MaachangQAccessDefine.TYPE_SEND,opt.get( j ) ) ;
                        
                        // そのチャネルが存在しない場合.
                        if( channel == null || channel.isChannel() == false ) {
                            // そのチャネルを送信キューから削除.
                            opt.remove( opt.get( j ) ) ;
                            
                            // 削除結果0件の場合.
                            if( opt.size() == 0 ) {
                                // キューステータスをエラーに設定.
                                channelError = true ;
                                break ;
                            }
                            j -- ;
                        }
                        
                        // チャネルステータスが正常以外の場合.
                        if( channel.getState() != ChannelStatus.STATE_SUCCESS ) {
                            // キューステータスをエラーに設定.
                            channelError = true ;
                            break ;
                        }
                        
                    }
                    
                    // 対象送信キューは存在しない場合.
                    if( end == true ) {
                        // 処理しない.
                        continue ;
                    }
                    
                    try {
                        
                        // チャネルステータスがエラーで、
                        // キューステータスがエラー属性以外の場合.
                        if( channelError == true &&
                            ( queue.getState().getState() & StatusDefine.STATE_MASK_ERROR ) !=
                                StatusDefine.STATE_MASK_ERROR ) {
                            // ステータスをエラーに設定.
                            queue.getState().setState( QueueStatus.STATE_ERROR ) ;
                        }
                        // チャネルステータスが復帰した場合.
                        else if( channelError == false &&
                            ( queue.getState().getState() == QueueStatus.STATE_STARTUP ||
                                queue.getState().getState() == QueueStatus.STATE_ERROR  ) ) {
                            
                            // ステータスを正常に設定.
                            queue.getState().setState( QueueStatus.STATE_SUCCESS ) ;
                            queue.checkStatus() ;
                        }
                        
                    } catch( Exception e ) {
                        LOG.error( "キューチェック中にエラー発生",e ) ;
                    }
                }
                
            }
            
        }catch( NullPointerException nul ){
            throw new ExecutionException(
                nul,ExecutionException.LEVEL_STOP
            ) ;
        }catch( BaseException be ){
            LOG.error( "エラーが発生しました", be ) ;
        }catch( Exception e ){
            LOG.error( "エラーが発生しました", e ) ;
        }finally{
            
        }
        
    }
    
}

