/*
 * @(#)SaveAndLoad.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.service ;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.commons.exception.AccessException;
import org.maachang.commons.io.IOCom;
import org.maachang.commons.serialize.SerializeUtil;
import org.maachang.queue.config.MqDefine;
import org.maachang.queue.config.service.ConfigService;
import org.maachang.queue.main.cache.MqCache;
import org.maachang.queue.main.cache.MqCacheBean;
import org.maachang.queue.main.cache.MqCacheFactory;
import org.maachang.queue.main.channel.Channel;
import org.maachang.queue.main.channel.ChannelBean;
import org.maachang.queue.main.channel.ChannelFactory;
import org.maachang.queue.main.channel.ChannelKey;
import org.maachang.queue.main.connect.Connect;
import org.maachang.queue.main.connect.ConnectBean;
import org.maachang.queue.main.connect.ConnectFactory;
import org.maachang.queue.main.manager.QueueManager;
import org.maachang.queue.main.manager.QueueManagerBean;
import org.maachang.queue.main.manager.QueueManagerFactory;
import org.maachang.queue.main.queue.QueueBean;
import org.maachang.queue.main.queue.base.BaseQueue;
import org.maachang.queue.main.queue.base.BaseQueueBean;
import org.maachang.queue.main.queue.base.BaseQueueFactory;
import org.maachang.queue.main.queue.base.BaseQueueKey;

/**
 * MaachangQセーブ＆ロード.
 *  
 * @version 2006/12/28
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
class SaveAndLoad {
    
    /**
     * セーブタイプ : キューマネージャ.
     */
    public static final int SAVE_TYPE_MANAGER = 0 ;
    
    /**
     * セーブタイプ : キュー.
     */
    public static final int SAVE_TYPE_QUEUE = 1 ;
    
    /**
     * セーブタイプ : チャネル.
     */
    public static final int SAVE_TYPE_CHANNEL = 2 ;
    
    /**
     * セーブタイプ : コネクション.
     */
    public static final int SAVE_TYPE_CONNECT = 3 ;
    
    /**
     * セーブタイプ : キャッシュ.
     */
    public static final int SAVE_TYPE_CACHE = 4 ;
    
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( SaveAndLoad.class ) ;
    
    /**
     * キャッシュヘッダ名.
     */
    private static final String HEAD_CACHE = "cache" ;
    
    /**
     * コネクションヘッダ名.
     */
    private static final String HEAD_CONNECT = "connect" ;
    
    /**
     * チャネルヘッダ名.
     */
    private static final String HEAD_CHANNEL = "channel" ;
    
    /**
     * キューヘッダ名.
     */
    private static final String HEAD_QUEUE = "queue" ;
    
    /**
     * キューマネージャヘッダ名.
     */
    private static final String HEAD_MANAGER = "manager" ;
    
    /**
     * 区切り文字.
     */
    private static final String CUT = "." ;
    
    /**
     * デフォルト同期.
     */
    private static final Object SYNC = new Object() ;
    
    
    
    /**
     * 構成データ全て破棄.
     * <BR><BR>
     * 構成データを全て破棄します.
     * <BR>
     * @exception AccessException アクセス例外.
     */
    public static final void destroy()
        throws AccessException {
        
        Exception lastException = null ;
        
        long time = System.currentTimeMillis() ;
        LOG.info( "## 構成データを全て破棄:開始" ) ;
        
        synchronized( SYNC ) {
            
            try {
                LOG.info( " ## キューを全て破棄:開始" ) ;
                new BaseQueueFactory().destroy() ;
                //SaveAndLoad.deleteHeaderByQueue() ;
                LOG.info( " ## キューを全て破棄:成功" ) ;
            } catch( Exception e ) {
                lastException = e ;
                LOG.error( " ## キューの破棄に失敗",e ) ;
            }
            
            try {
                LOG.info( " ## チャネルを全て破棄:開始" ) ;
                new ChannelFactory().destroy() ;
                //SaveAndLoad.deleteHeaderByChannel() ;
                LOG.info( " ## チャネルを全て破棄:成功" ) ;
            } catch( Exception e ) {
                lastException = e ;
                LOG.error( " ## チャネルの破棄に失敗",e ) ;
            }
            
            try {
                LOG.info( " ## キューマネージャを全て破棄:開始" ) ;
                new QueueManagerFactory().destroy() ;
                //SaveAndLoad.deleteHeaderByQueueManager() ;
                LOG.info( " ## キューマネージャを全て破棄:成功" ) ;
            } catch( Exception e ) {
                lastException = e ;
                LOG.error( " ## キャッシュマネージャの破棄に失敗",e ) ;
            }
            
            try {
                LOG.info( " ## コネクションを全て破棄:開始" ) ;
                new ConnectFactory().destroy() ;
                //SaveAndLoad.deleteHeaderByConnect() ;
                LOG.info( " ## コネクションを全て破棄:成功" ) ;
            } catch( Exception e ) {
                lastException = e ;
                LOG.error( " ## コネクションの破棄に失敗",e ) ;
            }
            
            try {
                LOG.info( " ## キャッシュを全て破棄:開始" ) ;
                new MqCacheFactory().destroy() ;
                //SaveAndLoad.deleteHeaderByCache() ;
                LOG.info( " ## キャッシュを全て破棄:成功" ) ;
            } catch( Exception e ) {
                lastException = e ;
                LOG.error( " ## キャッシュの破棄に失敗",e ) ;
            }
            
        }
        
        LOG.info( "## 構成データを全て破棄:終了(" +
            ( System.currentTimeMillis()-time ) + "msec)" ) ;
        
        if( lastException != null ) {
            throw new AccessException( lastException ) ;
        }
    }
    
    /**
     * 構成データ保存処理.
     * <BR><BR>
     * 構成データを保存します.
     * <BR>
     * @exception AccessException アクセス例外.
     */
    public static final void save() {
        SaveAndLoad.save( -1,null ) ;
    }
    
    /**
     * 構成データ保存処理.
     * <BR><BR>
     * 構成データを保存します.
     * <BR>
     * @param thread サービススレッドを設定します.
     * @exception AccessException アクセス例外.
     */
    public static final void save( SaveThread thread )
        throws AccessException {
        SaveAndLoad.save( -1,thread ) ;
    }
    
    /**
     * 構成データ保存処理.
     * <BR><BR>
     * 構成データを保存します.
     * <BR>
     * @param type 対象のタイプを設定します.
     * @param thread サービススレッドを設定します.
     * @exception AccessException アクセス例外.
     */
    public static final void save( int type,SaveThread thread )
        throws AccessException {
        
        Exception lastException = null ;
        
        // キャッシュ情報のセーブ.
        long time = System.currentTimeMillis() ;
        LOG.info( "## 構成情報のセーブ開始" ) ;
        
        try {
            synchronized( SYNC ) {
                
                int count = 0 ;
                long tm = 0L ;
                
                if( type == -1 || type == SAVE_TYPE_CACHE ) {
                    
                    // コネクション情報のセーブ.
                    tm = System.currentTimeMillis() ;
                    LOG.info( "  ## キャッシュ構成のセーブ開始" ) ;
                    
                    try {
                        
                         count = SaveAndLoad.saveCache() ;
                        
                        LOG.info( "  ## キャッシュ構成のセーブ成功(" + count + "):" +
                            ( System.currentTimeMillis()-tm ) ) ;
                        
                    } catch( Exception e ) {
                        LOG.error( "  ## キャッシュ構成のセーブ失敗(" + count + ")",e ) ;
                        lastException = e ;
                    }
                }
                
                if( type == -1 || type == SAVE_TYPE_CONNECT ) {
                    
                    // コネクション情報のセーブ.
                    tm = System.currentTimeMillis() ;
                    LOG.info( "  ## コネクション構成のセーブ開始" ) ;
                    
                    try {
                        
                        count = SaveAndLoad.saveConnect() ;
                        
                        LOG.info( "  ## コネクション構成のセーブ成功:(" + count + "):" +
                            ( System.currentTimeMillis()-tm ) ) ;
                        
                    } catch( Exception e ) {
                        LOG.error( "  ## コネクション構成のセーブ失敗(" + count + ")",e ) ;
                        lastException = e ;
                    }
                    
                }
                
                if( type == -1 || type == SAVE_TYPE_CHANNEL ) {
                    
                    // チャネル情報のセーブ.
                    tm = System.currentTimeMillis() ;
                    LOG.info( "  ## チャネル構成のセーブ開始" ) ;
                    
                    try {
                        
                        count = SaveAndLoad.saveChannel() ;
                        
                        LOG.info( "  ## チャネル構成のセーブ成功:(" + count + "):" +
                            ( System.currentTimeMillis()-tm ) ) ;
                        
                    } catch( Exception e ) {
                        LOG.error( "  ## チャネル構成のセーブ失敗(" + count + ")",e ) ;
                        lastException = e ;
                    }
                    
                }
                
                if( type == -1 || type == SAVE_TYPE_QUEUE ) {
                    
                    // キュー情報のセーブ.
                    tm = System.currentTimeMillis() ;
                    LOG.info( "  ## キュー構成のセーブ開始" ) ;
                    
                    try {
                        
                        count = SaveAndLoad.saveQueue() ;
                        
                        LOG.info( "  ## キュー構成のセーブ成功:(" + count + "):" +
                            ( System.currentTimeMillis()-tm ) ) ;
                        
                    } catch( Exception e ) {
                        LOG.error( "  ## キュー構成のセーブ失敗(" + count + ")",e ) ;
                        lastException = e ;
                    }
                    
                }
                
                if( type == -1 || type == SAVE_TYPE_MANAGER ) {
                    
                    // キューマネージャ情報のセーブ.
                    tm = System.currentTimeMillis() ;
                    LOG.info( "  ## キューマネージャ構成のセーブ開始" ) ;
                    
                    try {
                        
                        count = SaveAndLoad.saveQueueManager() ;
                        
                        LOG.info( "  ## キューマネージャ構成のセーブ成功:(" + count + "):" +
                            ( System.currentTimeMillis()-tm ) ) ;
                        
                    } catch( Exception e ) {
                        LOG.error( "  ## キューマネージャ構成のセーブ失敗(" + count + ")",e ) ;
                        lastException = e ;
                    }
                    
                }
                
                // スレッド情報が設定されている場合.
                if( thread != null ) {
                    
                    // 次回処理時間を設定.
                    thread.setUpdateTime( System.currentTimeMillis() ) ;
                    
                }
            }
            
        } catch( Exception ee ) {
            lastException = ee ;
        }
        
        LOG.info( "## 構成情報のセーブ終了:" +
            ( System.currentTimeMillis()-time ) ) ;
        
        // エラー処理が発生している場合.
        if( lastException != null ) {
            if( lastException instanceof AccessException ) {
                throw ( AccessException )lastException ;
            }
            throw new AccessException( lastException ) ;
        }
        
    }
    
    /**
     * 構成情報を全てロード.
     * <BR><BR>
     * 構成情報を全てロードします.
     * <BR>
     * @param mode ロードする時のモード設定.<BR>
     *              [true]を設定した場合、ロードする前の現在の全ての構成をクリアします.
     *              [false]を設定した場合、構成情報が既に存在する場合、処理しません.
     * @exception AccessException アクセス例外.
     */
    public static final void load( boolean mode )
        throws AccessException {
        
        // 構成情報が既に存在する場合.
        if( MqCacheFactory.size() > 0 ||
            ChannelFactory.size() > 0 ||
            BaseQueueFactory.size() > 0 ||
            QueueManagerFactory.size() > 0 ) {
            
            // 現在の構成情報を破棄して、ロードする場合.
            if( mode == true ) {
                SaveAndLoad.destroy() ;
            }
            // 現在の構成情報が存在する場合は、何もしない場合.
            else {
                return ;
            }
        }
        
        // キャッシュ情報のセーブ.
        long time = System.currentTimeMillis() ;
        LOG.info( "## 構成情報のロード開始" ) ;
        
        try {
            synchronized( SYNC ) {
                
                int count = 0 ;
                
                // キャッシュ情報のロード.
                long tm = System.currentTimeMillis() ;
                LOG.info( " ## キャッシュ情報をロード:開始" ) ;
                try {
                    
                    count = SaveAndLoad.loadCache() ;
                    if( count == 0 ) {
                        LOG.info( " ## キャッシュ情報なし" ) ;
                    }
                    else {
                        LOG.info( " ## キャッシュ情報をロード:成功(" + count + "):" +
                            ( System.currentTimeMillis()-tm ) ) ;
                    }
                    
                } catch( Exception e ) {
                    LOG.error( " ## キャッシュ情報のロードに失敗(" + count + ")" ) ;
                    throw e ;
                }
                
                // コネクション情報のロード.
                tm = System.currentTimeMillis() ;
                LOG.info( " ## コネクション情報をロード:開始" ) ;
                try {
                    
                    count = SaveAndLoad.loadConnect() ;
                    if( count == 0 ) {
                        LOG.info( " ## コネクション情報なし" ) ;
                    }
                    else {
                        LOG.info( " ## コネクション情報をロード:成功(" + count + "):" +
                            ( System.currentTimeMillis()-tm ) ) ;
                    }
                    
                } catch( Exception e ) {
                    LOG.error( " ## コネクション情報のロードに失敗(" + count + ")" ) ;
                    throw e ;
                }
                
                // チャネル情報のロード.
                tm = System.currentTimeMillis() ;
                LOG.info( " ## チャネル情報をロード:開始" ) ;
                try {
                    
                    count = SaveAndLoad.loadChannel() ;
                    if( count == 0 ) {
                        LOG.info( " ## チャネル情報なし" ) ;
                    }
                    else {
                        LOG.info( " ## チャネル情報をロード:成功(" + count + "):" +
                            ( System.currentTimeMillis()-tm ) ) ;
                    }
                    
                } catch( Exception e ) {
                    LOG.error( " ## チャネル情報のロードに失敗(" + count + ")" ) ;
                    throw e ;
                }
                
                // キューマネージャ情報のロード.
                tm = System.currentTimeMillis() ;
                LOG.info( " ## キューマネージャ情報をロード:開始" ) ;
                try {
                    
                    count = SaveAndLoad.loadQueueManager() ;
                    if( count == 0 ) {
                        LOG.info( " ## キューマネージャ情報なし" ) ;
                    }
                    else {
                        LOG.info( " ## キューマネージャ情報をロード:成功(" + count + "):" +
                            ( System.currentTimeMillis()-tm ) ) ;
                    }
                    
                } catch( Exception e ) {
                    LOG.error( " ## キューマネージャ情報のロードに失敗(" + count + ")" ) ;
                    throw e ;
                }
                
                // キュー情報のロード.
                tm = System.currentTimeMillis() ;
                LOG.info( " ## キュー情報をロード:開始" ) ;
                try {
                    
                    count = SaveAndLoad.loadQueue() ;
                    if( count == 0 ) {
                        LOG.info( " ## キュー情報なし" ) ;
                    }
                    else {
                        LOG.info( " ## キュー情報をロード:成功(" + count + "):" +
                            ( System.currentTimeMillis()-tm ) ) ;
                    }
                    
                } catch( Exception e ) {
                    LOG.error( " ## キュー情報のロードに失敗(" + count + ")" ) ;
                    throw e ;
                }
                
            }
            
        } catch( AccessException ae ) {
            LOG.error( "## 構成情報のロード失敗" ) ;
            SaveAndLoad.destroy() ;
            throw ae ;
        } catch( Exception ee ) {
            LOG.error( "## 構成情報のロード失敗" ) ;
            SaveAndLoad.destroy() ;
            throw new AccessException( ee ) ;
        } finally {
            
            LOG.info( "## 構成情報のロード終了:" +
                ( System.currentTimeMillis()-time ) ) ;
            
        }
    }
    
    /**
     * 同期オブジェクトを取得.
     * <BR><BR>
     * 同期オブジェクトを取得します.
     * <BR>
     * @return Object 同期オブジェクトが返されます.
     */
    public static final Object getSync() {
        return SYNC ;
    }
    
    /**
     * キャッシュ情報を保存.
     */
    private static final int saveCache()
        throws Exception {
        
        int len ;
        int count = 0 ;
        
        MqCache cache = null ;
        MqCacheBean bean = null ;
        String[] names = MqCacheFactory.getNames() ;
        
        if( names != null && ( len = names.length ) > 0 ) {
            
            SerializeUtil.putSerialize(
                SaveAndLoad.getManagerFileName( HEAD_CACHE ),names ) ;
            
            for( int i = 0 ; i < len ; i ++ ) {
                
                if( ( cache = MqCacheFactory.get( names[ i ] ) ) == null ) {
                    continue ;
                }
                
                if( ( bean = cache.getMqCacheBean() ) == null ) {
                    continue ;
                }
                
                try {
                    SerializeUtil.putCb32Serialize(
                        false,null,SaveAndLoad.getFileName( HEAD_CACHE,names[ i ] ),
                        bean ) ;
                    count ++ ;
                } catch( Exception e ) {
                    LOG.error( "  %%-キャッシュ[" + names[ i ] +
                        "]の保存に失敗",e ) ;
                }
                
            }
            
        }
        else {
            SaveAndLoad.deleteHeaderByCache() ;
        }
        
        return count ;
        
    }
    
    /**
     * コネクション情報を保存.
     */
    private static final int saveConnect()
        throws Exception {
        
        int len ;
        int count = 0 ;
        
        Connect conn = null ;
        ConnectBean bean = null ;
        String[] names = ConnectFactory.getNames() ;
        
        if( names != null && ( len = names.length ) > 0 ) {
            
            SerializeUtil.putSerialize(
                SaveAndLoad.getManagerFileName( HEAD_CONNECT ),names ) ;
            
            for( int i = 0 ; i < len ; i ++ ) {
                
                if( ( conn = ConnectFactory.get( names[ i ] ) ) == null ) {
                    continue ;
                }
                
                if( ( bean = conn.getConnectBean() ) == null ) {
                    continue ;
                }
                
                try {
                    SerializeUtil.putCb32Serialize(
                        false,null,SaveAndLoad.getFileName( HEAD_CONNECT,names[ i ] ),
                        bean ) ;
                    count ++ ;
                } catch( Exception e ) {
                    LOG.error( "  %%-コネクション[" + names[ i ] +
                        "]の保存に失敗",e ) ;
                }
                
            }
            
        }
        else {
            SaveAndLoad.deleteHeaderByConnect() ;
        }
        
        return count ;
        
    }
    
    /**
     * チャネル情報を保存.
     */
    private static final int saveChannel()
        throws Exception {
        
        int len ;
        int count = 0 ;
        
        Channel channel = null ;
        ChannelBean bean = null ;
        ChannelKey[] keys = ChannelFactory.getChannekKeys() ;
        
        if( keys != null && ( len = keys.length ) > 0 ) {
            
            SerializeUtil.putSerialize(
                SaveAndLoad.getManagerFileName( HEAD_CHANNEL ),keys ) ;
            
            for( int i = 0 ; i < len ; i ++ ) {
                
                ChannelKey key = keys[ i ] ;
                
                if( key == null ) {
                    continue ;
                }
                
                if( ( channel = ChannelFactory.get(
                    key.getChannelType(),key.getName() ) ) == null ) {
                    continue ;
                }
                
                if( ( bean = channel.getChannelBean() ) == null ) {
                    continue ;
                }
                
                try {
                    SerializeUtil.putCb32Serialize(
                        false,null,SaveAndLoad.getFileName(
                            HEAD_CHANNEL,key.getChannelType()+CUT+key.getName() ),
                        bean ) ;
                    count ++ ;
                } catch( Exception e ) {
                    LOG.error( "  %%-チャネル[type:" + key.getChannelType() +
                        " name:" + key.getName() +
                        "]の保存に失敗",e ) ;
                }
                
            }
            
        }
        else {
            SaveAndLoad.deleteHeaderByChannel() ;
        }
        
        return count ;
        
    }
    
    /**
     * キュー情報を保存.
     */
    private static final int saveQueue()
        throws Exception {
        
        int len ;
        int count = 0 ;
        
        BaseQueue queue = null ;
        QueueBean bean = null ;
        BaseQueueKey[] keys = BaseQueueFactory.getKeys() ;
        
        if( keys != null && ( len = keys.length ) > 0 ) {
            
            SerializeUtil.putSerialize(
                SaveAndLoad.getManagerFileName( HEAD_QUEUE ),keys ) ;
            
            for( int i = 0 ; i < len ; i ++ ) {
                
                BaseQueueKey key = keys[ i ] ;
                
                if( key == null ) {
                    continue ;
                }
                
                if( ( queue = BaseQueueFactory.get( key ) ) == null ) {
                    continue ;
                }
                
                if( ( bean = ( BaseQueueBean )queue.getQueueBean() ) == null ) {
                    continue ;
                }
                
                try {
                    SerializeUtil.putCb32Serialize(
                        false,null,SaveAndLoad.getFileName( HEAD_QUEUE,
                            key.getQueueType()+CUT+
                            key.getParentManagerName()+CUT+
                            key.getName() ),
                        bean ) ;
                    count ++ ;
                } catch( Exception e ) {
                    LOG.error( "  %%-キュー[type:" + key.getQueueType() +
                        " manager:" + key.getParentManagerName() +
                        " name:" + key.getName() +
                        "]の保存に失敗",e ) ;
                }
                
            }
            
        }
        else {
            SaveAndLoad.deleteHeaderByQueue() ;
        }
        
        return count ;
        
    }
    
    /**
     * キューマネージャ情報を保存.
     */
    private static final int saveQueueManager()
        throws Exception {
        
        int len ;
        int count = 0 ;
        
        QueueManager man = null ;
        QueueManagerBean bean = null ;
        String[] names = QueueManagerFactory.getNames() ;
        
        if( names != null && ( len = names.length ) > 0 ) {
            
            SerializeUtil.putSerialize(
                SaveAndLoad.getManagerFileName( HEAD_MANAGER ),names ) ;
            
            for( int i = 0 ; i < len ; i ++ ) {
                
                if( ( man = QueueManagerFactory.get( names[ i ] ) ) == null ) {
                    continue ;
                }
                
                if( ( bean = man.getQueueManagerBean() ) == null ) {
                    continue ;
                }
                
                try {
                    SerializeUtil.putCb32Serialize(
                        false,null,SaveAndLoad.getFileName( HEAD_MANAGER,names[ i ] ),
                        bean ) ;
                    count ++ ;
                } catch( Exception e ) {
                    LOG.error( "  %%-キューマネージャ[" + names[ i ] +
                        "]の保存に失敗",e ) ;
                }
                
            }
            
        }
        else {
            SaveAndLoad.deleteHeaderByQueueManager() ;
        }
        
        return count ;
        
    }
    
    /**
     * キャッシュ情報をロード.
     */
    private static final int loadCache()
        throws Exception {
        
        int count = 0 ;
        String[] names = null ;
        
        try {
            names = ( String[] )SerializeUtil.getSerialize(
                SaveAndLoad.getManagerFileName( HEAD_CACHE ) ) ;
        } catch( Exception ee ) {
            return 0 ;
        }
        
        if( names == null ) {
            return 0 ;
        }
        
        int len = names.length ;
        MqCacheBean bean = null ;
        
        for( int i = 0 ; i < len ; i ++ ) {
            
            if( names[ i ] == null ) {
                continue ;
            }
            
            try {
                bean = ( MqCacheBean )SerializeUtil.getCb32Serialize(
                    false,null,SaveAndLoad.getFileName( HEAD_CACHE,names[ i ] ) ) ;
                
                if( bean != null ) {
                    MqCacheFactory.create( bean ) ;
                    count ++ ;
                    LOG.info( "  $$-キャッシュ[" + names[ i ] +
                        "]の読み込みに成功" ) ;
                }
            } catch( Exception e ) {
                LOG.error( "  %%-キャッシュ[" + names[ i ] +
                    "]の読み込みに失敗",e ) ;
                throw e ;
            }
            
        }
        
        return count ;
        
    }
    
    /**
     * コネクション情報をロード.
     */
    private static final int loadConnect()
        throws Exception {
        
        int count = 0 ;
        String[] names = null ;
        
        try {
            names = ( String[] )SerializeUtil.getSerialize(
                SaveAndLoad.getManagerFileName( HEAD_CONNECT ) ) ;
        } catch( Exception ee ) {
            return 0 ;
        }
        
        if( names == null ) {
            return 0 ;
        }
        
        int len = names.length ;
        ConnectBean bean = null ;
        
        for( int i = 0 ; i < len ; i ++ ) {
            
            if( names[ i ] == null ) {
                continue ;
            }
            
            try {
                bean = ( ConnectBean )SerializeUtil.getCb32Serialize(
                    false,null,SaveAndLoad.getFileName( HEAD_CONNECT,names[ i ] ) ) ;
                
                if( bean != null ) {
                    ConnectFactory.create( bean ) ;
                    count ++ ;
                    LOG.info( "  $$-コネクション[" + names[ i ] +
                        "]の読み込みに成功" ) ;
                }
            } catch( Exception e ) {
                LOG.error( "  %%-コネクション[" + names[ i ] +
                    "]の読み込みに失敗",e ) ;
                throw e ;
            }
            
        }
        
        return count ;
        
    }
    
    /**
     * チャネル情報をロード.
     */
    private static final int loadChannel()
        throws Exception {
        
        int count = 0 ;
        ChannelKey[] keys = null ;
        
        try {
            keys = ( ChannelKey[] )SerializeUtil.getSerialize(
                SaveAndLoad.getManagerFileName( HEAD_CHANNEL ) ) ;
        } catch( Exception ee ) {
            return 0 ;
        }
        
        if( keys == null ) {
            return 0 ;
        }
        
        int len = keys.length ;
        ChannelBean bean = null ;
        ChannelKey key = null ;
        
        for( int i = 0 ; i < len ; i ++ ) {
            
            if( ( key = keys[ i ] ) == null ) {
                continue ;
            }
            
            try {
                bean = ( ChannelBean )SerializeUtil.getCb32Serialize(
                    false,null,SaveAndLoad.getFileName(
                        HEAD_CHANNEL,key.getChannelType()+CUT+key.getName() ) ) ;
                
                if( bean != null ) {
                    ChannelFactory.create( bean ) ;
                    count ++ ;
                    LOG.info( "  $$-キューマネージャ[type:" + key.getChannelType() +
                        " name:" + key.getName() +
                        "]の読み込みに成功" ) ;
                }
            } catch( Exception e ) {
                LOG.error( "  %%-チャネル[type:" + key.getChannelType() +
                    " name:" + key.getName() +
                    "]の読み込みに失敗",e ) ;
                throw e ;
            }
        }
        
        return count ;
        
    }
    
    /**
     * キュー情報をロード.
     */
    private static final int loadQueue()
        throws Exception {
        
        int count = 0 ;
        BaseQueueKey[] keys = null ;
        
        try {
            keys = ( BaseQueueKey[] )SerializeUtil.getSerialize(
                SaveAndLoad.getManagerFileName( HEAD_QUEUE ) ) ;
        } catch( Exception ee ) {
            return 0 ;
        }
        
        if( keys == null ) {
            return 0 ;
        }
        
        int len = keys.length ;
        BaseQueueBean bean = null ;
        BaseQueueKey key = null ;
        
        for( int i = 0 ; i < len ; i ++ ) {
            
            if( ( key = keys[ i ] ) == null ) {
                continue ;
            }
            
            try {
                bean = ( BaseQueueBean )SerializeUtil.getCb32Serialize(
                    false,null,SaveAndLoad.getFileName( HEAD_QUEUE,
                        key.getQueueType()+CUT+
                        key.getParentManagerName()+CUT+
                        key.getName() ) ) ;
                
                if( bean != null ) {
                    BaseQueueFactory.create( bean ) ;
                    count ++ ;
                    LOG.info( "  $$-キュー[type:" + key.getQueueType() +
                        " manager:" + key.getParentManagerName() +
                        " name:" + key.getName() +
                        "]の読み込みに成功" ) ;
                }
            } catch( Exception e ) {
                LOG.error( "  %%-キュー[type:" + key.getQueueType() +
                    " manager:" + key.getParentManagerName() +
                    " name:" + key.getName() +
                    "]の読み込みに失敗",e ) ;
                throw e ;
            }
            
        }
        
        return count ;
        
    }
    
    /**
     * キューマネージャ情報をロード.
     */
    private static final int loadQueueManager()
        throws Exception {
        
        int count = 0 ;
        String[] names = null ;
        
        try {
            names = ( String[] )SerializeUtil.getSerialize(
               SaveAndLoad.getManagerFileName( HEAD_MANAGER ) ) ;
        } catch( Exception ee ) {
            return 0 ;
        }
        
        if( names == null ) {
            return 0 ;
        }
        
        int len = names.length ;
        QueueManagerBean bean = null ;
        
        for( int i = 0 ; i < len ; i ++ ) {
            
            if( names[ i ] == null ) {
                continue ;
            }
            
            try {
                bean = ( QueueManagerBean )SerializeUtil.getCb32Serialize(
                    false,null,SaveAndLoad.getFileName( HEAD_MANAGER,names[ i ] ) ) ;
                
                if( bean != null ) {
                    QueueManagerFactory.create( bean ) ;
                    count ++ ;
                    LOG.info( "  $$-キューマネージャ[" + names[ i ] +
                        "]の読み込みに成功" ) ;
                }
            } catch( Exception e ) {
                LOG.error( "  %%-キューマネージャ[" + names[ i ] +
                    "]の読み込みに失敗",e ) ;
                throw e ;
            }
            
        }
        
        return count ;
        
    }
    
    /**
     * 構成キャッシュヘッダファイルを削除.
     */
    private static final void deleteHeaderByCache() {
        try {
            IOCom.deleteFile(
                SaveAndLoad.getManagerFileName( HEAD_CACHE ) ) ;
        } catch( Exception e ) {
        }
    }
    
    /**
     * 構成コネクションヘッダファイルを削除.
     */
    private static final void deleteHeaderByConnect() {
        try {
            IOCom.deleteFile(
                SaveAndLoad.getManagerFileName( HEAD_CONNECT ) ) ;
        } catch( Exception e ) {
        }
    }
    
    /**
     * 構成チャネルヘッダファイルを削除.
     */
    private static final void deleteHeaderByChannel() {
        try {
            IOCom.deleteFile(
                SaveAndLoad.getManagerFileName( HEAD_CHANNEL ) ) ;
        } catch( Exception e ) {
        }
    }
    
    /**
     * 構成キューヘッダファイルを削除.
     */
    private static final void deleteHeaderByQueue() {
        try {
            IOCom.deleteFile(
                SaveAndLoad.getManagerFileName( HEAD_QUEUE ) ) ;
        } catch( Exception e ) {
        }
    }
    
    /**
     * 構成キューマネージャヘッダファイルを削除.
     */
    private static final void deleteHeaderByQueueManager() {
        try {
            IOCom.deleteFile(
                SaveAndLoad.getManagerFileName( HEAD_MANAGER ) ) ;
        } catch( Exception e ) {
        }
    }
    
    /**
     * ファイル名を生成.
     */
    private static final String getFileName( String head,String name ) {
        
        String dir = SaveAndLoad.createDir() ;
        
        return new StringBuffer().
            append( dir ).
            append( "/" ).
            append( head ).
            append( "_" ).
            append( name ).
            append( MqDefine.SAVE_PLUS ).
            toString() ;
        
    }
    
    /**
     * 構成管理ファイル名を生成.
     */
    private static final String getManagerFileName( String head ) {
        
        String dir = SaveAndLoad.createDir() ;
        
        return new StringBuffer().
            append( dir ).
            append( "/" ).
            append( head ).
            append( MqDefine.SAVE_MANAGE_PLUS ).
            toString() ;
        
    }
    
    /**
     * 構成ディレクトリ生成処理.
     */
    private static final String createDir() {
        
        String dir = ConfigService.getMqConfig().getSaveAndLoadDirectory() ;
        
        if( IOCom.isDirExists( dir ) == false ) {
            IOCom.mkdirs( dir ) ;
        }
        
        return dir ;
    }
}

