/*
 * @(#)ConnectMqImple.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access ;

import org.maachang.queue.access.net.ConnectObject;
import org.maachang.queue.access.net.ReceiveTimeoutException;

/**
 * キューコネクション.
 * <BR><BR>
 * 接続中のキューを表すオブジェクト.
 *  
 * @version 2007/01/02
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
abstract class ConnectMqImple implements ConnectMq {
    
    /**
     * トランザクション待機.
     * 0.5sec.
     */
    protected static final long WAIT_TRANSACTION = 500L ;
    
    /**
     * キューマネージャコネクションオブジェクト.
     */
    protected MaachangQManager maachangQManager = null ;
    
    /**
     * キューマネージャ名.
     */
    protected String manager = null ;
    
    /**
     * キュー名.
     */
    protected String queue = null ;
    
    /**
     * キュータイプ.
     */
    protected int queueType = -1 ;
    
    /**
     * コネクションオブジェクト.
     */
    protected ConnectObject connect = null ;
    
    /**
     * オートコミット.
     */
    protected boolean autoCommit = false ;
    
    /**
     * トランザクション確立フラグ..
     */
    protected boolean transactionFlag = false ;
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     * <BR>
     * @exception Exception 例外.
     */
    protected void finalize() throws Exception {
        this.disconnect() ;
    }
    
    /**
     * コネクションを解放.
     * <BR><BR>
     * コネクションを解放します.
     */
    public void disconnect() {
        
        maachangQManager = null ;
        connect = null ;
        manager = null ;
        queue = null ;
        queueType = -1 ;
        autoCommit = false ;
        transactionFlag = false ;
        
    }
    
    /**
     * トランザクション開始.
     * <BR><BR>
     * トランザクションを開始します.
     * <BR>
     * @exception MaachangQException MaachangQ例外.
     */
    public synchronized void transaction() throws MaachangQException {
        this.transaction( 0L ) ;
    }
    
    /**
     * トランザクション開始.
     * <BR><BR>
     * トランザクションを開始します.
     * <BR>
     * @param timeout 対象のタイムアウト値を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public synchronized void transaction( long timeout )
        throws MaachangQException {
        
        long tm = 0L ;
        if( timeout > 0L ) {
            tm = System.currentTimeMillis() + timeout ;
        }
        else {
            timeout = MaachangQAccessDefine.DEFAULT_TIMEOUT ;
        }
        
        // トランザクションが行われていない場合.
        if( this.transactionFlag == false && autoCommit == false ) {
            
            // トランザクション確立まで、待機.
            for( ;; ) {
                
                if( this.isOpen() == false ) {
                    throw new MaachangQException(
                        "対象キューは既にクローズしています",
                        MaachangQErrorCode.ERROR_CLOSE_QUEUE ) ;
                }
                
                try {
                    
                    // トランザクション送信.
                    ExecutionClientMq.transactionQueue(
                        connect,manager,queue,queueType,timeout ) ;
                    
                    // トランザクションが確立された場合.
                    this.transactionFlag = true ;
                    break ;
                    
                } catch( ReceiveTimeoutException rt ) {
                    throw rt ;
                } catch( MaachangQException fa ) {
                    // 他のコネクションがトランザクションを保持している.
                    if( fa.getErrorCode() == MaachangQErrorCode.ERROR_USE_TRANSACTION ) {
                        if( tm > 0L && System.currentTimeMillis() >= tm ) {
                            throw new ReceiveTimeoutException(
                                "タイムアウトが発生しました" ) ;
                        }
                        this.waitTime() ;
                    }
                    // 上記以外のエラー.
                    else {
                        throw fa ;
                    }
                } catch( Exception e ) {
                    throw new MaachangQException( e ) ;
                }
            }
            
        }
        
    }
    
    /**
     * コミット処理.
     * <BR><BR>
     * コミット処理を行います.
     * <BR>
     * @exception MaachangQException MaachangQ例外.
     */
    public synchronized void commit() throws MaachangQException {
        
        this.commitOrRollback( true ) ;
        
    }
    
    /**
     * ロールバック処理.
     * <BR><BR>
     * ロールバック処理を行います.
     * <BR>
     * @exception MaachangQException MaachangQ例外.
     */
    public synchronized void rollback() throws MaachangQException {
        
        this.commitOrRollback( false ) ;
        
    }
    
    /**
     * 対象キュータイプを取得.
     * <BR><BR>
     * 対象のキュータイプを取得します.
     * <BR>
     * @return int キュータイプが返されます.<BR>
     *             [MaachangQAccessDefine.TYPE_SEND]が返された場合、送信キューです.<BR>
     *             [MaachangQAccessDefine.TYPE_RECEIVE]が返された場合、受信キューです.
     */
    public synchronized int getQueueType() {
        
        return queueType ;
        
    }
    
    /**
     * 対象キュー名を取得.
     * <BR><BR>
     * 接続中の対象キュー名を取得します.
     * <BR>
     * @return String 対象キュー名が返されます.
     */
    public synchronized String getName() {
        
        return queue ;
        
    }
    
    /**
     * 対象キューマネージャ名を取得.
     * <BR><BR>
     * 接続中のキューマネージャ名を取得します.
     * <BR>
     * @return String 対象キューマネージャ名が返されます.
     */
    public synchronized String getManagerName() {
        return manager ;
    }
    
    /**
     * MaachangQManagerオブジェクトを取得.
     * <BR><BR>
     * MaachangQManagerオブジェクトを取得します.
     * <BR>
     * @return MaachangQManager 対象のマネージャオブジェクトが返されます.
     */
    public synchronized MaachangQManager getMaachangQManager() {
        return maachangQManager ;
    }
    
    /**
     * オートコミットモード取得.
     * <BR><BR>
     * このキューがオートコミットモードであるか取得します.
     * <BR>
     * @return boolean オートコミットモードを取得します.<BR>
     *                 [true]が返された場合、オートコミットです.<BR>
     *                 [false]が返された場合、オートコミットではありません.
     */
    public synchronized boolean isAutoCommit() {
        return autoCommit ;
    }
    
    /**
     * トランザクション中であるかチェック.
     * <BR><BR>
     * トランザクション中であるかチェックします.
     * <BR>
     * @return boolean トランザクション中であるかチェックします.<BR>
     *                 [true]が返された場合、トランザクション中です.<BR>
     *                 [false]が返された場合、トランザクション中ではありません.
     */
    public synchronized boolean isTransaction() {
        return this.transactionFlag ;
    }
    
    /**
     * キューマネージャがオープン中かチェック.
     * <BR><RB>
     * 対象のキューマネージャがオープン中であるかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、オープン中です.<BR>
     *                 [false]が返された場合、クローズしています.
     */
    public synchronized boolean isOpen() {
        if( connect != null ) {
            return ( connect.isClose() == false ) ? true : false ;
        }
        return false ;
    }
    
    /**
     * 待機処理.
     * <BR><BR>
     * 一定期間処理待ちを行います.
     */
    protected final void waitTime()
        throws MaachangQException {
        try {
            Thread.sleep( WAIT_TRANSACTION ) ;
        } catch( Exception ee ) {
            throw new MaachangQException( ee ) ;
        }
    }
    
    /**
     * コミット or ロールバック.
     */
    private final void commitOrRollback( boolean mode )
        throws MaachangQException {
        
        if( this.isOpen() == false ) {
            throw new MaachangQException(
                "対象キューは既にクローズしています",
                MaachangQErrorCode.ERROR_CLOSE_QUEUE ) ;
        }
        
        // トランザクションが行われている場合.
        if( this.transactionFlag == true && autoCommit == false ) {
            
            // コミット or ロールバック処理.
            ExecutionClientMq.commitOrRollbackQueue(
                mode,connect,manager,queue,queueType ) ;
            this.transactionFlag = false ;
            
        }
        
    }
    
}

