/*
 * @(#)ChannelTelegram.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access.admin ;

import java.net.InetAddress;

import org.maachang.queue.access.MaachangQAccessDefine;
import org.maachang.queue.access.admin.info.CreateChannelInfo;
import org.maachang.queue.access.admin.info.CreateSendChannelInfo;
import org.maachang.queue.access.admin.info.SendChannelInfo;
import org.maachang.queue.access.protocol.CommonProtocol;
import org.maachang.queue.access.protocol.admin.AdminChannelBean;
import org.maachang.queue.access.protocol.admin.AdminChannelProtocol;


/**
 * 管理者チャネル用電文.
 *  
 * @version 2007/01/12
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
class ChannelTelegram {
    
    /**
     * 新しいチャネルを追加する電文を作成.
     * <BR><BR>
     * 新しいチャネルを追加する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param info 対象チャネル情報を設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createAddChannel(
        int[] out,boolean loginUserOwner,int userId,CreateChannelInfo info )
        throws Exception {
        
        if( info == null || info.getChannelName() == null ) {
            throw new IllegalArgumentException( "チャネル生成用条件は不正です" ) ;
        }
        
        int channelType = -1 ;
        InetAddress addr = null ;
        int port = -1 ;
        String connect = null ;
        String queueManager = null ;
        String useCb32Word = null ;
        
        if( info instanceof CreateSendChannelInfo ) {
            channelType = MaachangQAccessDefine.TYPE_SEND ;
            queueManager = ( ( CreateSendChannelInfo )info ).getQueueManager() ;
            addr = ( ( CreateSendChannelInfo )info ).getAddress() ;
            port = ( ( CreateSendChannelInfo )info ).getPort() ;
            connect = ( ( CreateSendChannelInfo )info ).getConnectName() ;
            useCb32Word = ( ( CreateSendChannelInfo )info ).getUseCb32Word() ;
            
            if( connect == null || connect.length() <= 0 ||
                queueManager == null || queueManager.length() <= 0 ||
                addr == null || port < 0 || port > 65535 ) {
                throw new IllegalArgumentException(
                    "送信チャネル生成用条件は不正です" ) ;
            }
        }
        else {
            channelType = MaachangQAccessDefine.TYPE_RECEIVE ;
        }
        
        if( out != null && out.length > 0 ) {
            out[ 0 ] = CommonProtocol.getSequenceId() ;
            return AdminChannelProtocol.createProtocol(
                out[ 0 ],                                       // 電文ID.
                userId,                                         // ログインユーザID.
                loginUserOwner,                                 // 生成ユーザオーナー.
                AdminChannelProtocol.TYPE_ADD_CHANNEL,          // 処理タイプ.
                info.getChannelName(),                          // チャネル名
                channelType,                                    // チャネルタイプ.
                queueManager,                                   // 送信キューマネージャ名.
                addr,                                           // 送信チャネルアドレス.
                port,                                           // 送信チャネルポート.
                connect,                                        // コネクションオブジェクト名.
                useCb32Word,                                    // 利用暗号ワード.
                -1,                                             // ステータスなし.
                null,                                           // 数値パラメータなし.
                -1,                                             // サイズなし.
                null                                            // パラメータなし.
                ) ;
        }
        
        throw new IllegalArgumentException( "out引数は不正です" ) ;
        
    }
    
    /**
     * 送信チャネルを変更する電文を作成.
     * <BR><BR>
     * 送信チャネルを変更する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param info 対象チャネル情報を設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createSetSendChannel(
        int[] out,boolean loginUserOwner,int userId,SendChannelInfo info )
        throws Exception {
        
        if( info == null || info.getChannelName() == null ) {
            throw new IllegalArgumentException( "チャネル条件は不正です" ) ;
        }
        
        int channelType = -1 ;
        InetAddress addr = null ;
        int port = -1 ;
        String connect = null ;
        String queueManager = null ;
        String useCb32Word = null ;
        
        channelType = MaachangQAccessDefine.TYPE_SEND ;
        queueManager = info.getQueueManager() ;
        addr = info.getAddress() ;
        port = info.getPort() ;
        connect = info.getConnectName() ;
        useCb32Word = info.getUseCb32Word() ;
        
        if( connect == null || connect.length() <= 0 ||
            queueManager == null || queueManager.length() <= 0 ||
            addr == null || port < 0 || port > 65535 ) {
            throw new IllegalArgumentException(
                "送信チャネル条件は不正です" ) ;
        }
        
        if( out != null && out.length > 0 ) {
            out[ 0 ] = CommonProtocol.getSequenceId() ;
            return AdminChannelProtocol.createProtocol(
                out[ 0 ],                                       // 電文ID.
                userId,                                         // ログインユーザID.
                loginUserOwner,                                 // 生成ユーザオーナー.
                AdminChannelProtocol.TYPE_SET_CHANNEL,          // 処理タイプ.
                info.getChannelName(),                          // チャネル名
                channelType,                                    // チャネルタイプ.
                queueManager,                                   // 接続キューマネージャ名.
                addr,                                           // 送信チャネルアドレス.
                port,                                           // 送信チャネルポート.
                connect,                                        // コネクションオブジェクト名.
                useCb32Word,                                    // 利用暗号ワード.
                -1,                                             // ステータスなし.
                null,                                           // 数値パラメータなし.
                -1,                                             // サイズなし.
                null                                            // パラメータなし.
                ) ;
        }
        
        throw new IllegalArgumentException( "out引数は不正です" ) ;
        
    }
    
    /**
     * チャネルを削除する電文を作成.
     * <BR><BR>
     * チャネルを削除する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param name 対象チャネル名を設定します.
     * @param type 対象チャネルタイプを設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createRemoveChannel(
        int[] out,boolean loginUserOwner,int userId,String name,int type )
        throws Exception {
        
        if( name == null ||
            ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            throw new IllegalArgumentException( "チャネル名は不正です" ) ;
        }
        
        return ChannelTelegram.createNameByChannel(
            out,loginUserOwner,userId,name,type,
            AdminChannelProtocol.TYPE_REMOVE_CHANNEL ) ;
        
    }
    
    /**
     * チャネルを停止する電文を作成.
     * <BR><BR>
     * チャネルを停止する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param name 対象チャネル名を設定します.
     * @param type 対象チャネルタイプを設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createStopChannel(
        int[] out,boolean loginUserOwner,int userId,String name,int type )
        throws Exception {
        
        if( name == null ||
            ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            throw new IllegalArgumentException( "チャネル名は不正です" ) ;
        }
        
        return ChannelTelegram.createNameByChannel(
            out,loginUserOwner,userId,name,type,
            AdminChannelProtocol.TYPE_STOP_CHANNEL ) ;
        
    }
    
    /**
     * チャネルを開始する電文を作成.
     * <BR><BR>
     * チャネルを開始する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param name 対象チャネル名を設定します.
     * @param type 対象チャネルタイプを設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createStartChannel(
        int[] out,boolean loginUserOwner,int userId,String name,int type )
        throws Exception {
        
        if( name == null ||
            ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            throw new IllegalArgumentException( "チャネル名は不正です" ) ;
        }
        
        return ChannelTelegram.createNameByChannel(
            out,loginUserOwner,userId,name,type,
            AdminChannelProtocol.TYPE_START_CHANNEL ) ;
        
    }
    
    /**
     * チャネルを取得する電文を作成.
     * <BR><BR>
     * チャネルを取得する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param name 対象チャネル名を設定します.
     * @param type 対象チャネルタイプを設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createGetChannel(
        int[] out,boolean loginUserOwner,int userId,String name,int type )
        throws Exception {
        
        if( name == null ||
            ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            throw new IllegalArgumentException( "チャネル名は不正です" ) ;
        }
        
        return ChannelTelegram.createNameByChannel(
            out,loginUserOwner,userId,name,type,
            AdminChannelProtocol.TYPE_GET_CHANNEL ) ;
        
    }
    
    /**
     * 全てのチャネル名一覧を取得する電文を作成.
     * <BR><BR>
     * 全てのチャネル名一覧を取得する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createGetNamesChannel( int[] out,boolean loginUserOwner,int userId )
        throws Exception {
        
        return ChannelTelegram.createNameByChannel(
            out,loginUserOwner,userId,CommonProtocol.DUMMY,
            MaachangQAccessDefine.TYPE_SEND,
            AdminChannelProtocol.TYPE_GET_CHANNEL_NAMES ) ;
        
    }
    
    /**
     * 全てのチャネル数を取得する電文を作成.
     * <BR><BR>
     * 全てのチャネル数を取得する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createGetSizeChannel( int[] out,boolean loginUserOwner,int userId )
        throws Exception {
        
        return ChannelTelegram.createNameByChannel(
            out,loginUserOwner,userId,CommonProtocol.DUMMY,
            MaachangQAccessDefine.TYPE_SEND,
            AdminChannelProtocol.TYPE_GET_CHANNEL_SIZE ) ;
        
    }
    
    /**
     * チャネルが存在するか確認する電文を作成.
     * <BR><BR>
     * チャネルが存在するか確認する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param name 対象チャネル名を設定します.
     * @param type 対象チャネルタイプを設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createUseChannel(
        int[] out,boolean loginUserOwner,int userId,String name,int type )
        throws Exception {
        
        if( name == null ||
            ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            throw new IllegalArgumentException( "チャネル名は不正です" ) ;
        }
        
        return ChannelTelegram.createNameByChannel(
            out,loginUserOwner,userId,name,type,
            AdminChannelProtocol.TYPE_IS_CHANNEL ) ;
        
    }
    
    /**
     * チャネル問い合わせ結果を解析.
     * <BR><BR>
     * チャネル問い合わせ結果を解析します.
     * <BR>
     * @param out データ受け取り用Beanを設定します.
     * @param telegram 対象の電文を設定します.
     * @exception Exception 例外.
     */
    public static final void analysisTelegram( AdminChannelBean out,byte[] telegram )
        throws Exception {
        
        if( out == null || telegram == null || telegram.length <= 0 ) {
            throw new IllegalArgumentException( "out引数は不正です" ) ;
        }
        
        int[] outId = new int[ 1 ] ;
        byte[] bin = CommonProtocol.analysisProtocol( null,outId,telegram ) ;
        AdminChannelProtocol.analysisProtocol( out,bin ) ;
        
        out.setId( outId[ 0 ] ) ;
        
    }
    
    /**
     * 名前とタイプを設定してチャネル電文を生成.
     */
    private static final byte[] createNameByChannel(
        int[] out,boolean loginUserOwner,int userId,String name,int channelType,int type )
        throws Exception {
        
        if( out != null && out.length > 0 ) {
            out[ 0 ] = CommonProtocol.getSequenceId() ;
            return AdminChannelProtocol.createProtocol(
                out[ 0 ],                                       // 電文ID.
                userId,                                         // ログインユーザID.
                loginUserOwner,                                 // 生成ユーザオーナー.
                type,                                           // 処理タイプ.
                name,                                           // チャネル名
                channelType,                                    // チャネルタイプ.
                null,                                           // 接続キューマネージャ名なし.
                ( String )null,                                 // 送信チャネルアドレスなし.
                -1,                                             // 送信チャネルポートなし.
                null,                                           // コネクションオブジェクト名なし.
                null,                                           // 利用暗号ワードなし.
                -1,                                             // ステータスなし.
                null,                                           // 数値パラメータなし.
                -1,                                             // サイズなし.
                null                                            // パラメータなし.
                ) ;
        }
        
        throw new IllegalArgumentException( "out引数は不正です" ) ;
        
    }

}

