/*
 * @(#)ExecutionQueue.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access.admin ;

import org.maachang.queue.access.MaachangQAccessDefine;
import org.maachang.queue.access.MaachangQErrorCode;
import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.access.admin.info.CreateQueueInfo;
import org.maachang.queue.access.admin.info.QueueInfo;
import org.maachang.queue.access.admin.info.QueueKeyInfo;
import org.maachang.queue.access.admin.info.SendQueueInfo;
import org.maachang.queue.access.net.ConnectObject;
import org.maachang.queue.access.protocol.CommonProtocol;
import org.maachang.queue.access.protocol.ResultProtocol;
import org.maachang.queue.access.protocol.ResultTelegram;
import org.maachang.queue.access.protocol.admin.AdminQueueBean;
import org.maachang.queue.access.protocol.admin.AdminQueueProtocol;


/**
 * 管理キュー実行処理.
 *  
 * @version 2007/01/12
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
class ExecutionQueue {
    
    /**
     * デフォルトタイムアウト.
     */
    public static final long DEFAULT_TIMEOUT = MaachangQAccessDefine.DEFAULT_TIMEOUT ;
    
    /**
     * キュー追加処理.
     * <BR><BR>
     * キュー追加処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param manager 対象のキューマネージャ名を設定します.
     * @param info 対象のキュー情報を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final void addQueue(
        ConnectObject connect,int userId,boolean userOwner,
        String manager,CreateQueueInfo info )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = QueueTelegram.createAddQueue(
                outId,userOwner,userId,manager,info ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminQueueProtocol.TYPE_ADD_QUEUE,
                    telegram ) ;
                return ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * キュー変更処理.
     * <BR><BR>
     * キュー内容の変更処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param manager 対象のキューマネージャ名を設定します.
     * @param info 対象のキュー情報を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final void setQueue(
        ConnectObject connect,int userId,boolean userOwner,
        String manager,QueueInfo info )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = QueueTelegram.createSetQueue(
                outId,userOwner,userId,manager,info ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminQueueProtocol.TYPE_SET_QUEUE,
                    telegram ) ;
                return ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * キュー削除処理.
     * <BR><BR>
     * キュー削除処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param manager 対象のキューマネージャ名を設定します.
     * @param name 対象のキューマネージャ名を設定します.
     * @param type 対象のキュータイプを設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final void removeQueue(
        ConnectObject connect,int userId,boolean userOwner,String manager,String name,int type )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = QueueTelegram.createRemoveQueue(
                outId,userOwner,userId,manager,name,type ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminQueueProtocol.TYPE_REMOVE_QUEUE,
                    telegram ) ;
                return ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * キュー取得処理.
     * <BR><BR>
     * キュー取得処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param manager 対象のキューマネージャ名を設定します.
     * @param name 対象のキューマネージャ名を設定します.
     * @param type 対象のキュータイプを設定します.
     * @return QueueInfo 取得されたキュー情報が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final QueueInfo getQueue(
        ConnectObject connect,int userId,boolean userOwner,String manager,String name,int type )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = QueueTelegram.createGetQueue(
                outId,userOwner,userId,manager,name,type ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminQueueProtocol.TYPE_GET_QUEUE,
                    telegram ) ;
                
            }
            // 受信情報が、キュー情報の場合.
            else if( result == AdminQueueProtocol.CATEGORY_TYPE_ADMIN_QUEUE ) {
                
                // キュー情報を解析.
                AdminQueueBean bean = new AdminQueueBean() ;
                QueueTelegram.analysisTelegram( bean,telegram ) ;
                
                // IDが一致しない場合.
                if( bean.getId() != outId[ 0 ] ) {
                    throw new MaachangQException( "受信電文は不正です" ) ;
                }
                if( bean.getName() == null || bean.getName().length() <= 0 ) {
                    throw new MaachangQException(
                        "対象名(" + name + ")のキュー名は存在しません",
                        MaachangQErrorCode.ERROR_NOT_QUEUE ) ;
                }
                
                QueueInfo ret = null ;
                
                // 送信キューの場合.
                if( bean.getQueueType() == MaachangQAccessDefine.TYPE_SEND ) {
                    
                    SendQueueInfo send = new SendQueueInfo() ;
                    send.setChannels( bean.getParams() ) ;
                    ret = send ;
                    
                }
                // 受信キューの場合.
                else {
                    ret = new QueueInfo() ;
                }
                
                ret.setQueueManager( bean.getManager() ) ;
                ret.setState( bean.getState() ) ;
                ret.setUseQueueSize( bean.getUseQueue() ) ;
                
                ret.setQueue( bean.getName() ) ;
                ret.setMaxMessageLength( bean.getMaxMessageLength() ) ;
                ret.setMaxQueueSize( bean.getMaxQueue() ) ;
                ret.setWarningPersend( bean.getWarningPersend() ) ;
                ret.setGzipFlag( bean.isGzipFlag() ) ;
                ret.setAutoCommit( bean.isAutoCommit() ) ;
                
                return ret ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * マネージャ内のキュー情報一覧取得処理.
     * <BR><BR>
     * マネージャ内のキュー情報一覧取得処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param manager 対象のキューマネージャ名を設定します.
     * @return QueueKeyInfo[] マネージャ内のキュー情報一覧が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final QueueKeyInfo[] getKeys(
        ConnectObject connect,int userId,boolean userOwner,String manager )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = QueueTelegram.createGetNamesQueue(
                outId,userOwner,userId,manager ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminQueueProtocol.TYPE_GET_QUEUE_NAMES,
                    telegram ) ;
                
            }
            // 受信情報が、キュー情報の場合.
            else if( result == AdminQueueProtocol.CATEGORY_TYPE_ADMIN_QUEUE ) {
                
                // キュー情報を解析.
                AdminQueueBean bean = new AdminQueueBean() ;
                QueueTelegram.analysisTelegram( bean,telegram ) ;
                
                // IDが一致しない場合.
                if( bean.getId() != outId[ 0 ] ) {
                    throw new MaachangQException( "受信電文は不正です" ) ;
                }
                
                int len = 0 ;
                if( bean.getParams() == null || bean.getNumbers() == null ||
                    bean.getParams().length != bean.getNumbers().length ) {
                    return null ;
                }
                
                len = bean.getParams().length ;
                
                if( len > 0 ) {
                    QueueKeyInfo[] ret = new QueueKeyInfo[ len ] ;
                    for( int i = 0 ; i < len ; i ++ ) {
                        
                        ret[ i ] = new QueueKeyInfo() ;
                        ret[ i ].setQueue( bean.getParams()[ i ] ) ;
                        ret[ i ].setType( bean.getNumbers()[ i ] ) ;
                        
                    }
                    
                    return ret ;
                }
                
                return null ;
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * マネージャ内のキューサイズ取得処理.
     * <BR><BR>
     * マネージャ内のキューサイズ取得処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param manager 対象のキューマネージャ名を設定します.
     * @return int マネージャ内のキュー数が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final int getSize(
        ConnectObject connect,int userId,boolean userOwner,String manager )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = QueueTelegram.createGetSizeQueue(
                outId,userOwner,userId,manager ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminQueueProtocol.TYPE_GET_QUEUE_SIZE,
                    telegram ) ;
                
            }
            // 受信情報が、キュー情報の場合.
            else if( result == AdminQueueProtocol.CATEGORY_TYPE_ADMIN_QUEUE ) {
                
                // キュー情報を解析.
                AdminQueueBean bean = new AdminQueueBean() ;
                QueueTelegram.analysisTelegram( bean,telegram ) ;
                
                // IDが一致しない場合.
                if( bean.getId() != outId[ 0 ] ) {
                    throw new MaachangQException( "受信電文は不正です" ) ;
                }
                
                return bean.getSize() ;
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * キュー存在チェック処理.
     * <BR><BR>
     * キュー存在チェック処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param manager 対象のキューマネージャ名を設定します.
     * @param name 対象のキューマネージャ名を設定します.
     * @param type 対象のキュータイプを設定します.
     * @return boolean 存在結果が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final boolean isQueue(
        ConnectObject connect,int userId,boolean userOwner,String manager,String name,int type )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = QueueTelegram.createUseQueue(
                outId,userOwner,userId,manager,name,type ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                try {
                    // 処理戻り条件をチェック.
                    ResultTelegram.checkResult(
                        outId[ 0 ],AdminQueueProtocol.TYPE_IS_QUEUE,
                        telegram ) ;
                    
                    return true ;
                } catch( Exception e ) {
                    return false ;
                }
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    
}

