/*
 * @(#)QueueTelegram.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access.admin ;

import org.maachang.queue.access.MaachangQAccessDefine;
import org.maachang.queue.access.admin.info.CreateQueueInfo;
import org.maachang.queue.access.admin.info.CreateSendQueueInfo;
import org.maachang.queue.access.admin.info.QueueInfo;
import org.maachang.queue.access.admin.info.SendQueueInfo;
import org.maachang.queue.access.protocol.CommonProtocol;
import org.maachang.queue.access.protocol.admin.AdminQueueBean;
import org.maachang.queue.access.protocol.admin.AdminQueueProtocol;


/**
 * 管理者キュー用電文.
 *  
 * @version 2007/01/11
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
class QueueTelegram {
    
    /**
     * 新しいキューを追加する電文を作成.
     * <BR><BR>
     * 新しいキューを追加する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param manager 対象のキューマネージャ名を設定します.
     * @param info 対象キュー情報を設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createAddQueue(
        int[] out,boolean loginUserOwner,int userId,String manager,CreateQueueInfo info )
        throws Exception {
        
        if( info == null ||
            manager == null ||
            ( manager = manager.trim().toLowerCase() ).length() <= 0 ||
            info.getQueue() == null ) {
            throw new IllegalArgumentException( "キューマネージャ条件は不正です" ) ;
        }
        
        int queueType = -1 ;
        String[] channelNames = null ;
        
        if( info instanceof CreateSendQueueInfo ) {
            queueType = MaachangQAccessDefine.TYPE_SEND ;
            channelNames = ( ( CreateSendQueueInfo )info ).getChannelNames() ;
        }
        else {
            queueType = MaachangQAccessDefine.TYPE_RECEIVE ;
        }
        
        if( out != null && out.length > 0 ) {
            out[ 0 ] = CommonProtocol.getSequenceId() ;
            return AdminQueueProtocol.createProtocol(
                out[ 0 ],                                       // 電文ID.
                userId,                                         // ログインユーザID.
                loginUserOwner,                                 // 生成ユーザオーナー.
                AdminQueueProtocol.TYPE_ADD_QUEUE,              // 処理タイプ.
                info.getQueue(),                                // キュー名
                manager,                                        // キューマネージャ名.
                queueType,                                      // キュータイプ.
                info.getMaxMessageLength(),                     // 最大メッセージ制限値.
                info.getMaxQueueSize(),                         // 最大キューサイズ.
                -1,                                             // 現在キューサイズなし.
                info.getWarningPersend(),                       // 警告パーセント.
                info.isGzipFlag(),                              // 圧縮フラグ.
                info.isAutoCommit(),                            // オートコミット.
                info.getCacheName(),                            // キャッシュ名.
                -1,                                             // ステータスなし.
                null,                                           // 数値パラメータなし.
                -1,                                             // サイズなし.
                channelNames                                    // 送信キューチャネル名群.
                ) ;
        }
        
        throw new IllegalArgumentException( "out引数は不正です" ) ;
        
    }
    
    /**
     * 対象キューの内容を変更する電文を作成.
     * <BR><BR>
     * 対象キューの内容を変更する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param manager 対象のキューマネージャ名を設定します.
     * @param info 対象キュー情報を設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createSetQueue(
        int[] out,boolean loginUserOwner,int userId,String manager,QueueInfo info )
        throws Exception {
        
        if( info == null ||
            manager == null ||
            ( manager = manager.trim().toLowerCase() ).length() <= 0 ||
            info.getQueue() == null ) {
            throw new IllegalArgumentException( "キューマネージャ条件は不正です" ) ;
        }
        
        int queueType = -1 ;
        String[] channelNames = null ;
        
        if( info instanceof SendQueueInfo ) {
            queueType = MaachangQAccessDefine.TYPE_SEND ;
            channelNames = ( ( SendQueueInfo )info ).getChannelNames() ;
        }
        else {
            queueType = MaachangQAccessDefine.TYPE_RECEIVE ;
        }
        
        if( out != null && out.length > 0 ) {
            out[ 0 ] = CommonProtocol.getSequenceId() ;
            return AdminQueueProtocol.createProtocol(
                out[ 0 ],                                       // 電文ID.
                userId,                                         // ログインユーザID.
                loginUserOwner,                                 // 生成ユーザオーナー.
                AdminQueueProtocol.TYPE_SET_QUEUE,              // 処理タイプ.
                info.getQueue(),                                // キュー名
                manager,                                        // キューマネージャ名.
                queueType,                                      // キュータイプ.
                -1,                                             // 最大メッセージ制限値なし.
                info.getMaxQueueSize(),                         // 最大キューサイズ.
                -1,                                             // 現在キューサイズなし.
                info.getWarningPersend(),                       // 警告パーセント.
                false,                                          // 圧縮フラグなし.
                false,                                          // オートコミットなし.
                null,                                           // キャッシュ名なし.
                -1,                                             // ステータスなし.
                null,                                           // 数値パラメータなし.
                -1,                                             // サイズなし.
                channelNames                                    // 送信キューチャネル名群.
                ) ;
        }
        
        throw new IllegalArgumentException( "out引数は不正です" ) ;
        
    }
    
    /**
     * キューを削除する電文を作成.
     * <BR><BR>
     * キューを削除する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param manager 対象のキューマネージャ名を設定します.
     * @param name 対象キュー名を設定します.
     * @param type 対象キュータイプを設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createRemoveQueue(
        int[] out,boolean loginUserOwner,int userId,String manager,String name,int type )
        throws Exception {
        
        if( manager == null ||
            ( manager = manager.trim().toLowerCase() ).length() <= 0 ) {
            throw new IllegalArgumentException( "キューマネージャ名は不正です" ) ;
        }
        if( name == null ||
            ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            throw new IllegalArgumentException( "キュー名は不正です" ) ;
        }
        
        return QueueTelegram.createNameByQueue(
            out,loginUserOwner,userId,manager,name,type,
            AdminQueueProtocol.TYPE_REMOVE_QUEUE ) ;
        
    }
    
    /**
     * キューを取得する電文を作成.
     * <BR><BR>
     * キューを取得する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param manager 対象のキューマネージャ名を設定します.
     * @param name 対象キュー名を設定します.
     * @param type 対象キュータイプを設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createGetQueue(
        int[] out,boolean loginUserOwner,int userId,String manager,String name,int type )
        throws Exception {
        
        if( manager == null ||
            ( manager = manager.trim().toLowerCase() ).length() <= 0 ) {
            throw new IllegalArgumentException( "キューマネージャ名は不正です" ) ;
        }
        if( name == null ||
            ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            throw new IllegalArgumentException( "キュー名は不正です" ) ;
        }
        
        return QueueTelegram.createNameByQueue(
            out,loginUserOwner,userId,manager,name,type,
            AdminQueueProtocol.TYPE_GET_QUEUE ) ;
        
    }
    
    /**
     * キューマネージャ内の全てのキュー名一覧を取得する電文を作成.
     * <BR><BR>
     * キューマネージャ内の全てのキュー名一覧を取得する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param manager 対象のキューマネージャ名を設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createGetNamesQueue(
        int[] out,boolean loginUserOwner,int userId,String manager )
        throws Exception {
        
        if( manager == null ||
            ( manager = manager.trim().toLowerCase() ).length() <= 0 ) {
            throw new IllegalArgumentException( "キューマネージャ名は不正です" ) ;
        }
        
        return QueueTelegram.createNameByQueue(
            out,loginUserOwner,userId,manager,CommonProtocol.DUMMY,
            MaachangQAccessDefine.TYPE_SEND,
            AdminQueueProtocol.TYPE_GET_QUEUE_NAMES ) ;
        
    }
    
    /**
     * キューマネージャ内の全てのキュー数を取得する電文を作成.
     * <BR><BR>
     * キューマネージャ内の全てのキュー数を取得する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param manager 対象のキューマネージャ名を設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createGetSizeQueue(
        int[] out,boolean loginUserOwner,int userId,String manager )
        throws Exception {
        
        if( manager == null ||
            ( manager = manager.trim().toLowerCase() ).length() <= 0 ) {
            throw new IllegalArgumentException( "キューマネージャ名は不正です" ) ;
        }
        
        return QueueTelegram.createNameByQueue(
            out,loginUserOwner,userId,manager,CommonProtocol.DUMMY,
            MaachangQAccessDefine.TYPE_SEND,
            AdminQueueProtocol.TYPE_GET_QUEUE_SIZE ) ;
        
    }
    
    /**
     * キューが存在するか確認する電文を作成.
     * <BR><BR>
     * キューが存在するか確認する電文を作成します.
     * <BR>
     * @param out 対象の電文IDが格納されます.
     * @param loginRootOwner ログインオーナーを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param manager 対象のキューマネージャ名を設定します.
     * @param name 対象キュー名を設定します.
     * @param type 対象キュータイプを設定します.
     * @return byte[] 対象の電文情報が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] createUseQueue(
        int[] out,boolean loginUserOwner,int userId,String manager,String name,int type )
        throws Exception {
        
        if( manager == null ||
            ( manager = manager.trim().toLowerCase() ).length() <= 0 ) {
            throw new IllegalArgumentException( "キューマネージャ名は不正です" ) ;
        }
        if( name == null ||
            ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            throw new IllegalArgumentException( "キュー名は不正です" ) ;
        }
        
        return QueueTelegram.createNameByQueue(
            out,loginUserOwner,userId,manager,name,type,
            AdminQueueProtocol.TYPE_IS_QUEUE ) ;
        
    }
    
    /**
     * キュー問い合わせ結果を解析.
     * <BR><BR>
     * キュー問い合わせ結果を解析します.
     * <BR>
     * @param out データ受け取り用Beanを設定します.
     * @param telegram 対象の電文を設定します.
     * @exception Exception 例外.
     */
    public static final void analysisTelegram( AdminQueueBean out,byte[] telegram )
        throws Exception {
        
        if( out == null || telegram == null || telegram.length <= 0 ) {
            throw new IllegalArgumentException( "out引数は不正です" ) ;
        }
        
        int[] outId = new int[ 1 ] ;
        byte[] bin = CommonProtocol.analysisProtocol( null,outId,telegram ) ;
        AdminQueueProtocol.analysisProtocol( out,bin ) ;
        
        out.setId( outId[ 0 ] ) ;
        
    }
    
    /**
     * 名前とタイプを設定してキュー電文を生成.
     */
    private static final byte[] createNameByQueue(
        int[] out,boolean loginUserOwner,int userId,String manager,String name,
        int queueType,int type )
        throws Exception {
        
        if( out != null && out.length > 0 ) {
            out[ 0 ] = CommonProtocol.getSequenceId() ;
            return AdminQueueProtocol.createProtocol(
                out[ 0 ],                                       // 電文ID.
                userId,                                         // ログインユーザID.
                loginUserOwner,                                 // 生成ユーザオーナー.
                type,                                           // 処理タイプ.
                name,                                           // キュー名
                manager,                                        // キューマネージャ名.
                queueType,                                      // キュータイプ.
                -1,                                             // 最大メッセージ制限値なし.
                -1,                                             // 最大キューサイズなし.
                -1,                                             // 現在キューサイズなし.
                -1,                                             // 警告パーセントなし.
                false,                                          // 圧縮フラグなし.
                false,                                          // オートコミットなし.
                null,                                           // キャッシュ名なし.
                -1,                                             // ステータスなし.
                null,                                           // 数値パラメータなし.
                -1,                                             // サイズなし.
                null                                            // パラメータなし.
                ) ;
        }
        
        throw new IllegalArgumentException( "out引数は不正です" ) ;
        
    }

}

