/*
 * @(#)BaseAdminProtocol.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access.protocol.admin ;

import org.maachang.queue.access.protocol.CommonProtocol;
import org.maachang.queue.access.util.ConvertBinary;


/**
 * 基本管理者プロトコル.
 *  
 * @version 2007/01/04
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
public class BaseAdminProtocol {
    
    /**
     * 電文基本情報設定.
     * <BR><BR>
     * 電文基本情報を設定します.
     * <BR>
     * @param addPoint 対象電文を追加するためのポイントを格納されます.
     * @param binaryLength バイナリ長を設定します.
     * @param id プロトコルIDを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param rootOwner ルートオーナを設定します.
     * @param type 処理タイプを設定します.
     * @param name 対象名を設定します.
     * @param size 存在する内容データ数を設定します.
     * @param params 処理パラメータを設定します.
     * @return byte[] 対象のバイナリを設定します.
     * @exception Exception 例外.
     */
    public static final byte[] createCommonTelegram(
        int[] addPoint,int binaryLength,int id,int userId,boolean rootOwner,
        int type,String name,int size,String[] params )
        throws Exception {
        
        // 引数が不正.
        if( addPoint == null || addPoint.length <= 0 || binaryLength <= 0 ||
            id < 0 || userId < 0 || name == null ||
            ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        
        // ルート権限が必要な処理タイプの場合.
        CommonProtocol.checkRootOwnerByType( rootOwner,type ) ;
        
        // 名前をバイナリ変換.
        byte[] nameBin = name.getBytes( CommonProtocol.CHARSET ) ;
        int nameLen = nameBin.length ;
        
        // 全体電文情報長を計算.
        int allBinaryLength = 
            4 +                 // 全体電文長.
            4 +                 // ユーザID.
            4 +                 // 処理タイプ.
            4 +                 // 名前長.
            nameLen +           // 名前.
            binaryLength +      // 指定電文長.
            4 +                 // パラメータ数.
            4 +                 // 存在する内容数.
            4 ;                 // チェックコード.
        
        // 存在する内容長、パラメータを追加する位置.
        int nextPnt = allBinaryLength - ( 4 + 4 + 4 ) ;
        
        // パラメータ長を設定.
        int i ;
        int paramLen = 0 ;
        Object[] objs = null ;
        if( params != null && ( paramLen = params.length ) > 0 ) {
            objs = new Object[ paramLen ] ;
            for( i = 0 ; i < paramLen ; i ++ ) {
                
                if( params[ i ] == null ||
                    ( params[ i ] = params[ i ].trim() ).length() <= 0 ) {
                    objs[ i ] = new byte[ 0 ] ;
                }
                else {
                    objs[ i ] = params[ i ].getBytes( CommonProtocol.CHARSET ) ;
                }
                
                allBinaryLength += ( ( byte[] )objs[ i ] ).length + 4 ;
            }
        }
        
        // 電文情報を生成.
        int pnt = 0 ;
        byte[] telegram = new byte[ allBinaryLength ] ;
        
        // 全体電文長.
        ConvertBinary.convertInt( telegram,pnt,allBinaryLength ) ;
        pnt += 4 ;
        
        // ユーザID.
        ConvertBinary.convertInt( telegram,pnt,userId ) ;
        pnt += 4 ;
        
        // 処理タイプ
        ConvertBinary.convertInt( telegram,pnt,type ) ;
        pnt += 4 ;
        
        // 名前長.
        ConvertBinary.convertInt( telegram,pnt,nameLen ) ;
        pnt += 4 ;
        
        // 名前.
        System.arraycopy( nameBin,0,telegram,pnt,nameLen ) ;
        pnt += nameLen ;
        nameBin = null ;
        
        // この情報の間に、データを設定する必要がある.
        addPoint[ 0 ] = pnt ;
        
        // 存在する内容長、パラメータを追加する位置を設定.
        pnt = nextPnt ;
        
        // パラメータを設定.
        if( paramLen > 0 ) {
            for( i = paramLen-1 ; i >= 0 ; i -- ) {
                
                // パラメータ内の文字列数.
                int objLen = ( ( byte[] )objs[ i ] ).length ;
                
                // パラメータ内容が存在する場合.
                if( objLen > 0 ) {
                    // パラメータ内容を設定.
                    System.arraycopy( ( byte[] )objs[ i ],0,telegram,pnt,objLen ) ;
                    pnt += objLen ;
                }
                
                // １つのパラメータ長を設定.
                ConvertBinary.convertInt( telegram,pnt,objLen ) ;
                pnt += 4 ;
                
            }
        }
        
        // パラメータ数.
        ConvertBinary.convertInt( telegram,pnt,paramLen ) ;
        pnt += 4 ;
        
        // 存在する内容長.
        ConvertBinary.convertInt( telegram,pnt,size ) ;
        pnt += 4 ;
        
        return telegram ;
        
    }
    
    /**
     * 電文基本情報を取得.
     * <BR><BR>
     * 電文基本情報を取得します.
     * <BR>
     * @param readPoint 対象電文読み込み開始位置が格納されます.
     * @param out 電文基本情報を格納されます.
     * @param telegram 対象の電文情報を設定します.
     * @exception Exception 例外.
     */
    public static final void analysisCommonTelegram(
        int[] readPoint,BaseAdminProtocolBean out,byte[] telegram )
        throws Exception {
        
        if( readPoint == null || readPoint.length <= 0 ||
            out == null || telegram == null || telegram.length <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        
        // 電文長をチェック.
        CommonProtocol.checkTelegramLength( telegram ) ;
        int pnt = 4 ;
        
        // チェックコードを判別.
        CommonProtocol.checkCheckCode( telegram ) ;
        
        // ユーザIDを取得.
        int userId = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // 処理タイプを取得.
        int type = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // 名前長を取得.
        int nameLen = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // 名前を取得.
        byte[] nameBin = new byte[ nameLen ] ;
        System.arraycopy( telegram,pnt,nameBin,0,nameLen ) ;
        String name = new String( nameBin,CommonProtocol.CHARSET ) ;
        nameBin = null ;
        pnt += nameLen ;
        
        // 読み込み開始位置のポイントを設定.
        readPoint[ 0 ] = pnt ;
        
        // 存在する内容長、パラメータを追加する位置を取得.
        pnt = telegram.length - ( 4 + 4 ) ;
        
        // 存在する内容長を取得.
        int size = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt -= 4 ;
        
        // パラメータ数を取得.
        int paramLen = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt -= 4 ;
        
        int i ;
        
        // パラメータ情報群を取得.
        String[] params = null ;
        if( paramLen > 0 ) {
            int len ;
            
            byte[] bin = null ;
            params = new String[ paramLen ] ;
            
            for( i = 0 ; i < paramLen ; i ++ ) {
                
                // １つのパラメータ長を取得.
                len = ConvertBinary.convertInt( pnt,telegram ) ;
                pnt -= len ;
                
                // パラメータ長が存在しない場合.
                if( len <= 0 ) {
                    params[ i ] = "" ;
                }
                // パラメータ長が存在する場合.
                else {
                    bin = new byte[ len ] ;
                    System.arraycopy( telegram,pnt,bin,0,len ) ;
                    params[ i ] = new String( bin,CommonProtocol.CHARSET ) ;
                    pnt -= 4 ;
                }
                
            }
        }
        
        // Beanに設定.
        out.setUserId( userId ) ;
        out.setType( type ) ;
        out.setName( name ) ;
        out.setSize( size ) ;
        out.setParams( params ) ;
        
    }
    
}

