package org.maachang.dbm ;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.Enumeration;

import org.maachang.util.ConvertParam;

/**
 * MaachangDbm操作用オブジェクト.
 * 
 * @version 2008/01/18
 * @author masahito suzuki
 * @since MaachangDBM 1.02
 */
public abstract class AbstractMDbmOp implements MDbmOp {
    
    /**
     * MDbm.
     */
    protected MDbm dbm = null ;
    
    /**
     * MaachangDbmオブジェクトを取得.
     * <BR>
     * @return MDbm MaachangDbmオブジェクトが返されます.
     */
    protected MDbm maachangDbm() {
        return this.dbm ;
    }
    
    /**
     * クリアー処理.
     * <BR>
     * @exception Exception 例外.
     */
    public void close() throws Exception {
        this.dbm.close() ;
    }
    
    /**
     * コミット処理.
     * <BR>
     * @exception Exception 例外.
     */
    public void commit() throws Exception {
        this.dbm.commit() ;
    }
    
    /**
     * ロールバック処理.
     * <BR>
     * @exception Exception 例外.
     */
    public void rollback() throws Exception {
        this.dbm.rollback() ;
    }
    
    /**
     * データ登録が可能かチェック.
     * <BR>
     * @exception Exception 例外.
     */
    public void check() throws Exception {
        this.dbm.check() ;
    }
    
    /**
     * 情報を設定.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @param value 対象の情報を設定します.
     * @exception Exception 例外.
     */
    public void put( byte[] key,byte[] value ) throws Exception {
        this.dbm.put( key,value ) ;
    }
    
    /**
     * 情報を設定.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @param value 対象の情報を設定します.
     * @exception Exception 例外.
     */
    public void put( String key,byte[] value ) throws Exception {
        this.dbm.put( convertString( key ),value ) ;
    }
    
    /**
     * 情報を設定.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @param value 対象の情報を設定します.
     * @exception Exception 例外.
     */
    public void put( short key,byte[] value ) throws Exception {
        this.dbm.put( convertShort( key ),value ) ;
    }
    
    /**
     * 情報を設定.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @param value 対象の情報を設定します.
     * @exception Exception 例外.
     */
    public void put( int key,byte[] value ) throws Exception {
        this.dbm.put( convertInt( key ),value ) ;
    }
    
    /**
     * 情報を設定.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @param value 対象の情報を設定します.
     * @exception Exception 例外.
     */
    public void put( long key,byte[] value ) throws Exception {
        this.dbm.put( convertLong( key ),value ) ;
    }
    
    /**
     * 情報を設定.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @param value 対象の情報を設定します.
     * @exception Exception 例外.
     */
    public void put( byte[] key,String value ) throws Exception {
        this.dbm.put( key,convertString( value ) ) ;
    }
    
    /**
     * 情報を設定.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @param value 対象の情報を設定します.
     * @exception Exception 例外.
     */
    public void put( String key,String value ) throws Exception {
        this.dbm.put( convertString( key ),convertString( value ) ) ;
    }
    
    /**
     * 情報を設定.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @param value 対象の情報を設定します.
     * @exception Exception 例外.
     */
    public void put( short key,String value ) throws Exception {
        this.dbm.put( convertShort( key ),convertString( value ) ) ;
    }
    
    /**
     * 情報を設定.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @param value 対象の情報を設定します.
     * @exception Exception 例外.
     */
    public void put( int key,String value ) throws Exception {
        this.dbm.put( convertInt( key ),convertString( value ) ) ;
    }
    
    /**
     * 情報を設定.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @param value 対象の情報を設定します.
     * @exception Exception 例外.
     */
    public void put( long key,String value ) throws Exception {
        this.dbm.put( convertLong( key ),convertString( value ) ) ;
    }
    
    /**
     * 情報を設定.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @param value 対象の情報を設定します.
     * @exception Exception 例外.
     */
    public void put( byte[] key,Serializable value ) throws Exception {
        this.dbm.put( key,convertBinary( value ) ) ;
    }
    
    /**
     * 情報を設定.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @param value 対象の情報を設定します.
     * @exception Exception 例外.
     */
    public void put( String key,Serializable value ) throws Exception {
        this.dbm.put( convertString( key ),convertBinary( value ) ) ;
    }
    
    /**
     * 情報を設定.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @param value 対象の情報を設定します.
     * @exception Exception 例外.
     */
    public void put( short key,Serializable value ) throws Exception {
        this.dbm.put( convertShort( key ),convertBinary( value ) ) ;
    }
    
    /**
     * 情報を設定.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @param value 対象の情報を設定します.
     * @exception Exception 例外.
     */
    public void put( int key,Serializable value ) throws Exception {
        this.dbm.put( convertInt( key ),convertBinary( value ) ) ;
    }
    
    /**
     * 情報を設定.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @param value 対象の情報を設定します.
     * @exception Exception 例外.
     */
    public void put( long key,Serializable value ) throws Exception {
        this.dbm.put( convertLong( key ),convertBinary( value ) ) ;
    }
    
    /**
     * 情報を削除.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @exception Exception 例外.
     */
    public void remove( byte[] key ) throws Exception {
        this.dbm.remove( key ) ;
    }
    
    /**
     * 情報を削除.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @exception Exception 例外.
     */
    public void remove( String key ) throws Exception {
        this.dbm.remove( convertString( key ) ) ;
    }
    
    /**
     * 情報を削除.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @exception Exception 例外.
     */
    public void remove( short key ) throws Exception {
        this.dbm.remove( convertShort( key ) ) ;
    }
    
    /**
     * 情報を削除.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @exception Exception 例外.
     */
    public void remove( int key ) throws Exception {
        this.dbm.remove( convertInt( key ) ) ;
    }
    
    /**
     * 情報を削除.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @exception Exception 例外.
     */
    public void remove( long key ) throws Exception {
        this.dbm.remove( convertLong( key ) ) ;
    }
    
    /**
     * 情報を取得.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @return byte[] 対象の情報が返されます.
     * @exception Exception 例外.
     */
    public byte[] get( byte[] key ) throws Exception {
        return this.dbm.get( key ) ;
    }
    
    /**
     * 情報を取得.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @return byte[] 対象の情報が返されます.
     * @exception Exception 例外.
     */
    public byte[] get( String key ) throws Exception {
        return this.dbm.get( convertString( key ) ) ;
    }
    
    /**
     * 情報を取得.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @return byte[] 対象の情報が返されます.
     * @exception Exception 例外.
     */
    public byte[] get( short key ) throws Exception {
        return this.dbm.get( convertShort( key ) ) ;
    }
    
    /**
     * 情報を取得.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @return byte[] 対象の情報が返されます.
     * @exception Exception 例外.
     */
    public byte[] get( int key ) throws Exception {
        return this.dbm.get( convertInt( key ) ) ;
    }
    
    /**
     * 情報を取得.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @return byte[] 対象の情報が返されます.
     * @exception Exception 例外.
     */
    public byte[] get( long key ) throws Exception {
        return this.dbm.get( convertLong( key ) ) ;
    }
    
    /**
     * 情報を取得.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @return String 対象の情報が返されます.
     * @exception Exception 例外.
     */
    public String getString( byte[] key ) throws Exception {
        byte[] b = this.dbm.get( key ) ;
        if( b == null ) {
            return null ;
        }
        return new String( b,"UTF8" ) ;
    }
    
    /**
     * 情報を取得.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @return String 対象の情報が返されます.
     * @exception Exception 例外.
     */
    public String getString( String key ) throws Exception {
        byte[] b = this.dbm.get( convertString( key ) ) ;
        if( b == null ) {
            return null ;
        }
        return new String( b,"UTF8" ) ;
    }
    
    /**
     * 情報を取得.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @return String 対象の情報が返されます.
     * @exception Exception 例外.
     */
    public String getString( short key ) throws Exception {
        byte[] b = this.dbm.get( convertShort( key ) ) ;
        if( b == null ) {
            return null ;
        }
        return new String( b,"UTF8" ) ;
    }
    
    /**
     * 情報を取得.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @return String 対象の情報が返されます.
     * @exception Exception 例外.
     */
    public String getString( int key ) throws Exception {
        byte[] b = this.dbm.get( convertInt( key ) ) ;
        if( b == null ) {
            return null ;
        }
        return new String( b,"UTF8" ) ;
    }
    
    /**
     * 情報を取得.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @return String 対象の情報が返されます.
     * @exception Exception 例外.
     */
    public String getString( long key ) throws Exception {
        byte[] b = this.dbm.get( convertLong( key ) ) ;
        if( b == null ) {
            return null ;
        }
        return new String( b,"UTF8" ) ;
    }
    
    /**
     * 情報を取得.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @return Serializable 対象の情報が返されます.
     * @exception Exception 例外.
     */
    public Serializable getObject( byte[] key ) throws Exception {
        byte[] b = this.dbm.get( key ) ;
        if( b == null ) {
            return null ;
        }
        return convertObject( b ) ;
    }
    
    /**
     * 情報を取得.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @return Serializable 対象の情報が返されます.
     * @exception Exception 例外.
     */
    public Serializable getObject( String key ) throws Exception {
        byte[] b = this.dbm.get( convertString( key ) ) ;
        if( b == null ) {
            return null ;
        }
        return convertObject( b ) ;
    }
    
    /**
     * 情報を取得.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @return Serializable 対象の情報が返されます.
     * @exception Exception 例外.
     */
    public Serializable getObject( short key ) throws Exception {
        byte[] b = this.dbm.get( convertShort( key ) ) ;
        if( b == null ) {
            return null ;
        }
        return convertObject( b ) ;
    }
    
    /**
     * 情報を取得.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @return Serializable 対象の情報が返されます.
     * @exception Exception 例外.
     */
    public Serializable getObject( int key ) throws Exception {
        byte[] b = this.dbm.get( convertInt( key ) ) ;
        if( b == null ) {
            return null ;
        }
        return convertObject( b ) ;
    }
    
    /**
     * 情報を取得.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @return Serializable 対象の情報が返されます.
     * @exception Exception 例外.
     */
    public Serializable getObject( long key ) throws Exception {
        byte[] b = this.dbm.get( convertLong( key ) ) ;
        if( b == null ) {
            return null ;
        }
        return convertObject( b ) ;
    }
    
    /**
     * 指定キーが存在するかチェック.
     * <BR>
     * @param key チェック対象のキー内容を設定します.
     * @return boolean [true]の場合、情報が存在します.
     * @exception Exception 例外.
     */
    public boolean containsKey( byte[] key) throws Exception {
        return this.dbm.containsKey( key ) ;
    }
    
    /**
     * 指定キーが存在するかチェック.
     * <BR>
     * @param key チェック対象のキー内容を設定します.
     * @return boolean [true]の場合、情報が存在します.
     * @exception Exception 例外.
     */
    public boolean containsKey( String key) throws Exception {
        return this.dbm.containsKey( convertString( key ) ) ;
    }
    
    /**
     * 指定キーが存在するかチェック.
     * <BR>
     * @param key チェック対象のキー内容を設定します.
     * @return boolean [true]の場合、情報が存在します.
     * @exception Exception 例外.
     */
    public boolean containsKey( short key) throws Exception {
        return this.dbm.containsKey( convertShort( key ) ) ;
    }
    
    /**
     * 指定キーが存在するかチェック.
     * <BR>
     * @param key チェック対象のキー内容を設定します.
     * @return boolean [true]の場合、情報が存在します.
     * @exception Exception 例外.
     */
    public boolean containsKey( int key) throws Exception {
        return this.dbm.containsKey( convertInt( key ) ) ;
    }
    
    /**
     * 指定キーが存在するかチェック.
     * <BR>
     * @param key チェック対象のキー内容を設定します.
     * @return boolean [true]の場合、情報が存在します.
     * @exception Exception 例外.
     */
    public boolean containsKey( long key) throws Exception {
        return this.dbm.containsKey( convertLong( key ) ) ;
    }
    
    /**
     * キー内容を列挙.
     * <BR><BR>
     * @return  Enumeration<byte[]> 列挙オブジェクトが返されます.
     */
    public Enumeration<byte[]> elements() {
        return this.dbm.elements() ;
    }
    
    /**
     * 新しいシーケンスIDを取得.
     * <BR><BR>
     * 新しいシーケンスIDを取得します.<BR>
     * このメソッドはトランザクションに対応しません.
     * <BR>
     * @param no シーケンスNoを設定します.<BR>
     *           [0-63]まで利用可能です.
     * @return long 新しいシーケンスIDが返されます.
     * @exception Exception 例外.
     */
    public long sequenceId( int no ) throws Exception {
        return this.dbm.sequenceId( no ) ;
    }
    
    /**
     * 格納情報数を取得.
     * <BR>
     * @return int 格納情報数が返されます.<BR>
     *             [-1]が返された場合、オブジェクトは既に破棄されています.
     */
    public int size() {
        return this.dbm.size() ;
    }
    
    /**
     * MaachangDbm展開ディレクトリを取得.
     * <BR>
     * @return String MaachangDbm展開ディレクトリ名が返されます.
     */
    public String getDirectory() {
        return this.dbm.getDirectory() ;
    }
    
    /**
     * このオブジェクトが有効かチェック.
     * <BR>
     * @return boolean [true]の場合、有効です.
     */
    public boolean isUse() {
        return this.dbm.isUse() ;
    }
    
    /**
     * このオブジェクトがトランザクション対応かチェック.
     * <BR>
     * @return boolean [true]の場合、トランザクションに対応しています.
     */
    public boolean isTransaction() {
        return this.dbm.isTransaction() ;
    }
    
    private static final byte[] convertString( String key ) throws Exception {
        if( key == null || key.length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        return key.getBytes( "UTF8" ) ;
    }
    
    private static final byte[] convertShort( short key ) throws Exception {
        return ConvertParam.convertShort( key ) ;
    }
    
    private static final byte[] convertInt( int key ) throws Exception {
        return ConvertParam.convertInt( key ) ;
    }
    
    private static final byte[] convertLong( long key ) throws Exception {
        return ConvertParam.convertLong( key ) ;
    }
    
    private static final byte[] convertBinary( Serializable value )
        throws Exception {
        if( value == null ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        byte[] ret = null ;
        ObjectOutputStream o = null ;
        try{
            ByteArrayOutputStream b = new ByteArrayOutputStream() ;
            o = new ObjectOutputStream( b ) ;
            o.writeObject( value ) ;
            o.flush() ;
            ret = b.toByteArray() ;
        } catch( Exception e ){
            throw e ;
        } finally {
            try {
                o.close() ;
            } catch( Exception e ) {
            }
        }
        return ret ;
    }
    
    private static final Serializable convertObject( byte[] bin )
        throws Exception {
        if( bin == null || bin.length <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        ObjectInputStream in = null;
        Serializable ret = null ;
        try{
            in = new ObjectInputStream( new ByteArrayInputStream( bin ) ) ;
            ret = ( Serializable )in.readObject() ;
        }catch( Exception e ){
            throw e ;
        }finally{
            try{
                in.close() ;
            }catch( Exception e ){
            }
            in = null ;
        }
        return ret ;
    }

}
