package org.maachang.dbm.engine ;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;

import org.maachang.util.FileUtil;

/**
 * セクター群管理オブジェクト.
 * 
 * @version 2008/01/17
 * @author masahito suzuki
 * @since MaachangDBM 1.00
 */
public class MSctArray {
    
    /**
     * セクターファイル名拡張子.
     */
    private static final String SECTOR_PLUS = ".sector" ;
    
    /**
     * セクター管理.
     */
    private ArrayList<MSector> sectors = null ;
    
    /**
     * セクター管理ファイル格納ディレクトリ.
     */
    private String directory = null ;
    
    /**
     * 空きセクター数.
     */
    private int useSector = 0 ;
    
    /**
     * 同期オブジェクト
     */
    private final Object sync = new Object() ;
    
    /**
     * コンストラクタ.
     */
    private MSctArray() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 条件を設定してオブジェクトを生成します.
     * <BR>
     * @param directory 対象のディレクトリを設定します.
     * @exception Exception 例外.
     */
    public MSctArray( String directory ) throws Exception {
        if( directory == null || ( directory = directory.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        directory = FileUtil.getFullPath( directory ) ;
        if( directory.endsWith( "\\" ) || directory.endsWith( "/" ) ) {
            directory = directory.substring( 0,directory.length() - 1 ) ;
        }
        if( FileUtil.isDirExists( directory ) == false ) {
            FileUtil.mkdirs( directory ) ;
        }
        this.sectors = loadSectors( directory ) ;
        this.useSector = calcUseSector() ;
        this.directory = directory ;
    }
    
    /**
     * デストラクタ.
     * <BR>
     * @exception Exception 例外.
     */
    protected void finalize() throws Exception {
        this.destroy() ;
    }
    
    /**
     * オブジェクト破棄.
     */
    public void destroy() {
        synchronized( sync ) {
            if( sectors != null ) {
                destroySectors() ;
            }
            sectors = null ;
            useSector = 0 ;
            directory = null ;
        }
    }
    
    /**
     * オブジェクト更新.
     * <BR><BR>
     * オブジェクトを更新します.
     * <BR>
     * @exception Exception 例外.
     */
    public void flush() throws Exception {
        synchronized( sync ) {
            if( check() == false ) {
                throw new IOException( "オブジェクトは既に破棄されています" ) ;
            }
            saveSectors() ;
        }
    }
    
    private static final ArrayList<MSector> loadSectors( String directory )
        throws Exception {
        ArrayList<MSector> ret = new ArrayList<MSector>() ;
        String[] lst = FileUtil.getFileList( directory ) ;
        if( lst != null ) {
            int len = lst.length ;
            int cnt = 0 ;
            for( int i = 0 ; i < len ; i ++ ) {
                if( lst[ i ] != null && ( lst[ i ] = lst[ i ].trim() ).length() > 0 &&
                    lst[ i ].endsWith( SECTOR_PLUS ) ) {
                    int no ;
                    try {
                        no = Integer.parseInt(
                            lst[ i ].substring(
                                0,lst[ i ].length() - SECTOR_PLUS.length() ) ) ;
                    } catch( Exception e ) {
                        no = -1 ;
                    }
                    if( no != -1 && no < MKey.MAX_VALUE_FILE_SIZE ) {
                        cnt ++ ;
                    }
                }
            }
            for( int i = 0 ; i < cnt ; i ++ ) {
                MSector sector = new MSector( i,new StringBuilder().
                    append( directory ).append( FileUtil.FILE_SPACE ).
                    append( String.valueOf( i ) ).append( SECTOR_PLUS ).toString() ) ;
                ret.add( sector ) ;
            }
        }
        return ret ;
    }
    
    private void saveSectors() throws Exception {
        int len = sectors.size() ;
        for( int i = 0 ; i < len ; i ++ ) {
            sectors.get( i ).flush() ;
        }
    }
    
    private void destroySectors() {
        int len = sectors.size() ;
        for( int i = 0 ; i < len ; i ++ ) {
            sectors.get( i ).destroy() ;
        }
    }
    
    /**
     * 新しいセクターを追加.
     * <BR><BR>
     * 新しいセクターを追加します.
     * <BR>
     * @exception Exception 例外.
     */
    public void add() throws Exception {
        synchronized( sync ) {
            if( check() == false ) {
                throw new IOException( "オブジェクトは既に破棄されています" ) ;
            }
            int no = sectors.size() ;
            if( no + 1 >= MKey.MAX_VALUE_FILE_SIZE ) {
                throw new IOException( "セクターファイルは["+MKey.MAX_VALUE_FILE_SIZE+
                    "]以上追加できません" ) ;
            }
            long sz = new File( directory ).getFreeSpace() ;
            if( MSector.oneSectorFileSize() >= sz ) {
                throw new IOException( "ファイル作成に対して、容量[" + sz + "]が足りません" ) ;
            }
            MSector sector = new MSector( no,new StringBuilder().
                append( directory ).append( FileUtil.FILE_SPACE ).
                append( String.valueOf( no ) ).append( SECTOR_PLUS ).toString() ) ;
            sectors.add( sector ) ;
            useSector += MSector.MAX_SECTOR ;
        }
    }
    
    /**
     * 指定項番のセクターを取得.
     * <BR><BR>
     * 指定項番のセクターを取得します.
     * <BR>
     * @param no 対象のセクター項番を設定します.
     * @return MSector 対象のセクター情報が返されます.
     * @exception Exception 例外.
     */
    public MSector get( int no ) throws Exception {
        MSector ret = null ;
        synchronized( sync ) {
            if( check() == false ) {
                throw new IOException( "オブジェクトは既に破棄されています" ) ;
            }
            if( no <= -1 || no >= sectors.size() ) {
                ret = null ;
            }
            else {
                ret = sectors.get( no ) ;
            }
        }
        return ret ;
    }
    
    /**
     * 一番空き領域の大きいセクターオブジェクトを取得.
     * <BR><BR>
     * 一番空き領域の大きいセクターオブジェクトを取得します.
     * <BR>
     * @param targetUse 指定空き容量を設定します.
     * @return MSector 対象のセクター情報が返されます.
     * @exception Exception 例外.
     */
    public MSector getMaxUseObject( int targetUse ) throws Exception {
        if( targetUse <= 0 ) {
            targetUse = -1 ;
        }
        MSector ret = null ;
        synchronized( sync ) {
            if( check() == false ) {
                throw new IOException( "オブジェクトは既に破棄されています" ) ;
            }
            int maxUse = 0 ;
            int len = sectors.size() ;
            if( len > 1 && sectors.get( 0 ).useSector() < sectors.get( len-1 ).useSector() ) {
                for( int i = len-1 ; i >= 0 ; i -- ) {
                    MSector s = sectors.get( i ) ;
                    int u = s.useSector() ;
                    if( targetUse != -1 && targetUse < u ) {
                        ret = s ;
                        break ;
                    }
                    if( maxUse < u ) {
                        ret = s ;
                        maxUse = u ;
                    }
                }
            }
            else {
                for( int i = 0 ; i < len ; i ++ ) {
                    MSector s = sectors.get( i ) ;
                    int u = s.useSector() ;
                    if( targetUse != -1 && targetUse < u ) {
                        ret = s ;
                        break ;
                    }
                    if( maxUse < u ) {
                        ret = s ;
                        maxUse = u ;
                    }
                }
            }
        }
        return ret ;
    }
    
    /**
     * 利用可能セクターを１デクリメント.
     * <BR><BR>
     * 利用可能セクターを１デクリメントします.
     */
    protected void removeUseSector() {
        synchronized( sync ) {
            if( check() == true ) {
                useSector -- ;
            }
        }
    }
    
    /**
     * 利用可能セクターを１インクリメント.
     * <BR><BR>
     * 利用可能セクターを１インクリメントします.
     */
    protected void addUseSector() {
        synchronized( sync ) {
            if( check() == true ) {
                useSector ++ ;
            }
        }
    }
    
    /**
     * 全セクター数の利用可能セクター数を取得.
     * <BR><BR>
     * 全セクター数の利用可能セクター数を取得します.
     * <BR>
     * @return int 全セクター数の利用可能セクター数が返されます.
     */
    public int useSector() {
        int ret = 0 ;
        synchronized( sync ) {
            if( check() == false ) {
                ret = -1 ;
            }
            else {
                ret = useSector ;
            }
        }
        return ret ;
    }
    
    /**
     * 全セクター数を取得.
     * <BR><BR>
     * 全セクター数を取得します.
     * <BR>
     * @return int 全セクター数が返されます.
     */
    public int maxSector() {
        int ret = 0 ;
        synchronized( sync ) {
            if( check() == false ) {
                ret = -1 ;
            }
            else {
                ret = size() * MSector.MAX_SECTOR ;
            }
        }
        return ret ;
    }
    
    /**
     * セクター管理数を取得.
     * <BR><BR>
     * 現在のセクター管理数が返されます.
     * <BR>
     * @return int セクター管理数が返されます.
     */
    public int size() {
        int ret = 0 ;
        synchronized( sync ) {
            if( check() == false ) {
                ret = -1 ;
            }
            else {
                ret = sectors.size() ;
            }
        }
        return ret ;
    }
    
    /**
     * セクター管理ディレクトリを取得.
     * <BR><BR>
     * セクター管理ディレクトリを取得します.
     * <BR>
     * @return String セクター管理ディレクトリ名が返されます.
     */
    public String getDirectory() {
        String ret = null ;
        synchronized( sync ) {
            if( check() == false ) {
                ret = null ;
            }
            else {
                ret = directory ;
            }
        }
        return ret ;
    }
    
    /**
     * このオブジェクトが有効かチェック.
     * <BR><BR>
     * このオブジェクトが有効であるかチェックします.
     * <BR>
     * @return boolean [true]の場合、有効です.
     */
    public boolean isUse() {
        boolean ret = false ;
        synchronized( sync ) {
            ret = check() ;
        }
        return ret ;
    }
    
    private static final int SECTOR = 0x00001fff ;
    private static final int MASK = ~SECTOR ;
    private static final int SHIFT = 13 ;
    
    /**
     * 指定データ長をセクター長に変換.
     */
    protected static final int sectorLength( int len ) {
        return ( ( len & MASK ) >> SHIFT ) + ( ( ( len & SECTOR ) != 0 ) ? 1 : 0 ) ;
    }
    
    private boolean check() {
        if( sectors == null || directory == null ) {
            return false ;
        }
        return true ;
    }
    
    private int calcUseSector() {
        int ret = 0 ;
        int len = sectors.size() ;
        for( int i = 0 ; i < len ; i ++ ) {
            int use = sectors.get( i ).useSector() ;
            if( use == -1 ) {
                return -1 ;
            }
            ret += use ;
        }
        return ret ;
    }
}
