package org.maachang.dbm.service ;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.Socket;
import java.util.Enumeration;

import org.maachang.dbm.MDbm;
import org.maachang.dbm.MDbmManager;
import org.maachang.util.ConvertParam;

/**
 * DbmSession.
 * 
 * @version 2008/01/18
 * @author masahito suzuki
 * @since MaachangDBM 1.03
 */
class DbmSession {
    public static final int RECEIVE_TIMEOUT = 30000 ;
    private static final long HEAT_BEAT_TIMING = 5000L ;
    private static final int MAX_HEART_BEAT = 3 ;
    private static final int BUFFER = 131072 ;
    private Socket socket = null ;
    private InputStream inputStream = null ;
    private OutputStream outputStream = null ;
    private MDbm mdbm = null ;
    private String ipPort = null ;
    private Enumeration<byte[]> keys = null ;
    private long heartBeat = -1L ;
    private boolean heartBeatFlag = false ;
    private int heartBeatTiming = 0 ;
    
    private DbmSession() {
        
    }
    
    public DbmSession( Socket socket ) throws Exception {
        if( socket == null || socket.isClosed() == true ) {
            throw new IllegalArgumentException( "接続は不正です" ) ;
        }
        try {
            this.socket = socket ;
            this.socket.setSendBufferSize( BUFFER ) ;
            this.socket.setReceiveBufferSize( BUFFER ) ;
            this.socket.setKeepAlive( true ) ;
            this.socket.setTcpNoDelay( true ) ;
            this.socket.setReuseAddress( true ) ;
            this.socket.setSoLinger( false,0 ) ;
            this.socket.setSoTimeout( RECEIVE_TIMEOUT ) ;
            this.inputStream = new BufferedInputStream( this.socket.getInputStream() ) ;
            this.outputStream = new BufferedOutputStream( this.socket.getOutputStream() ) ;
            this.mdbm = MDbmManager.getInstance().getMDbm( true ) ;
            this.ipPort = new StringBuilder().append( this.socket.getInetAddress().getHostName() ).
                append( "/" ).append( this.socket.getPort() ).toString() ;
            this.heartBeat = System.currentTimeMillis() + HEAT_BEAT_TIMING ;
            this.heartBeatFlag = false ;
            this.heartBeatTiming = 0 ;
        } catch( Exception e ) {
            if( socket != null ) {
                try {
                    socket.close() ;
                } catch( Exception ee ) {
                }
            }
            throw e ;
        }
    }
    
    protected void finalize() throws Exception {
        this.destroy() ;
    }
    
    public synchronized void destroy() {
        if( socket != null ) {
            System.out.println( new StringBuilder().append( "<<[" ).
                append( System.currentTimeMillis() ).append( "]:_close:" ).
                append( ipPort ).toString() ) ;
        }
        if( this.mdbm != null ) {
            try {
                mdbm.commit() ;
            } catch( Exception e ) {
            }
        }
        if( this.inputStream != null ) {
            try {
                this.inputStream.close() ;
            } catch( Exception e ) {
            }
        }
        if( this.outputStream != null ) {
            try {
                this.outputStream.flush() ;
            } catch( Exception e ) {
            }
            try {
                this.outputStream.close() ;
            } catch( Exception e ) {
            }
        }
        if( this.socket != null ) {
            try {
                this.socket.close() ;
            } catch( Exception e ) {
            }
        }
        this.socket = null ;
        this.inputStream = null ;
        this.outputStream = null ;
        this.mdbm = null ;
        this.keys = null ;
        this.heartBeat = -1L ;
        this.heartBeatFlag = true ;
        this.heartBeatTiming = 0 ;
    }
    
    public synchronized boolean isClosed() {
        boolean ret = false ;
        try {
            if( this.socket == null || this.socket.isBound() == false ||
                this.socket.isClosed() == true ||
                this.mdbm == null || this.mdbm.isUse() == false ) {
                this.destroy() ;
                ret = true ;
            }
            ret = false ;
        } catch( Exception e ) {
            this.destroy() ;
            ret = true ;
        }
        return ret ;
    }
    
    public synchronized void heartBeat() throws Exception {
        try {
            if( heartBeat >= 0L ) {
                long nowTime = System.currentTimeMillis() ;
                if( heartBeatFlag == false && heartBeat <= nowTime ) {
                    sendHeartBeat() ;
                    this.heartBeat = nowTime + HEAT_BEAT_TIMING ;
                    this.heartBeatTiming ++ ;
                    if( this.heartBeatTiming >= MAX_HEART_BEAT ) {
                        this.heartBeatFlag = true ;
                    }
                }
                else if( heartBeatFlag == true && heartBeat <= nowTime ) {
                    this.destroy() ;
                }
            }
        } catch( Exception e ) {
            this.destroy() ;
        }
    }
    
    public synchronized boolean isReceive() throws Exception {
        if( isClosed() == true ) {
            throw new IOException( "既にクローズされています" ) ;
        }
        boolean ret = false ;
        if( this.inputStream().available() > 0 ) {
            ret = true ;
        }
        return ret ;
    }
    
    public synchronized Socket socket() {
        if( isClosed() == true ) {
            return null ;
        }
        return this.socket ;
    }
    
    public synchronized InputStream inputStream() {
        if( isClosed() == true ) {
            return null ;
        }
        return this.inputStream ;
    }
    
    public synchronized OutputStream outputStream() {
        if( isClosed() == true ) {
            return null ;
        }
        return this.outputStream ;
    }
    
    public synchronized MDbm mdbm() {
        return mdbm ;
    }
    
    public synchronized void clearKeys() {
        this.keys = null ;
    }
    
    public synchronized Enumeration<byte[]> getKeys() {
        return keys ;
    }
    
    public synchronized void setKeys( Enumeration<byte[]> keys ) {
        this.keys = keys ;
    }
    
    public synchronized String ipPort() {
        return ipPort ;
    }
    
    //public synchronized void stopHeartBeat() {
    //    this.heartBeat = -1L ;
    //}
    
    //public synchronized void startHeartBeat() {
    //    this.heartBeat = System.currentTimeMillis() + HEAT_BEAT_TIMING ;
    //    this.heartBeatTiming = 0 ;
    //}
    
    public synchronized void successHeartBeat() {
        this.heartBeatFlag = false ;
        this.heartBeat = System.currentTimeMillis() + HEAT_BEAT_TIMING ;
        this.heartBeatTiming = 0 ;
    }
    
    private void sendHeartBeat() throws Exception {
        if( outputStream != null ) {
            synchronized( outputStream ) {
                outputStream.write( ConvertParam.convertInt( 0 ) ) ;
                outputStream.flush() ;
            }
        }
    }
}

