package org.maachang.comet.httpd.engine.script.image;

import org.maachang.util.ConvertParam;


/**
 * イメージデータオブジェクト.
 * 
 * @version 2008/01/27
 * @author masahito suzuki
 * @since MaachangComet 1.07
 */
public class ImageData {
    
    /**
     * 画像Mimeタイプ.
     */
    protected static final String[] IMAGE_MIME = {
        "","image/jpeg","image/gif","image/x-png","image/x-bmp",
        "application/x-shockwave-flash" } ;
    
    private int type = -1 ;
    private long lastTime = -1L ;
    private byte[] image = null ;
    
    /**
     * コンストラクタ.
     */
    public ImageData() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 指定内容の情報を設定します.
     * <BR>
     * @param type 対象の画像タイプを設定します.
     * @param mime 対象のMimeタイプを設定します.
     * @param lastTime 対象の最終更新時間を設定します.
     * @param length 対象のイメージ長を設定します.
     * @param image 対象のイメージ情報を設定します.
     */
    public ImageData( int type,long lastTime,byte[] image ) {
        this.type = type ;
        this.lastTime = lastTime ;
        this.image = image ;
    }
    
    protected void finalize() throws Exception {
        this.destroy() ;
    }
    
    /**
     * データ破棄.
     */
    public void destroy() {
        this.type = -1 ;
        this.lastTime = -1L ;
        this.image = null ;
    }
    
    /**
     * データタイプを設定.
     * <BR>
     * @param type データタイプを設定します.
     */
    public void setType( int type ) {
        this.type = type ;
    }
    
    /**
     * データタイプを取得.
     * <BR>
     * @return int データタイプが返されます.
     */
    public int getType() {
        return this.type ;
    }
    
    /**
     * Mimeタイプを取得.
     * <BR>
     * @return String mimeタイプが返されます.
     */
    public String getMime() {
        if( type < 0 || type >= IMAGE_MIME.length ) {
            return null ;
        }
        return IMAGE_MIME[ type ] ;
    }
    
    /**
     * 最終更新時間を設定.
     * <BR>
     * @param lastTime 最終更新時間を設定します.
     */
    public void setLastTime( long lastTime ) {
        this.lastTime = lastTime ;
    }
    
    /**
     * 最終更新時間を取得.
     * <BR>
     * @return long 最終更新時間が返されます.
     */
    public long getLastTime() {
        return this.lastTime ;
    }
    
    /**
     * イメージ長を取得.
     * <BR><BR>
     * @return image 対象のイメージ長が返されます.
     */
    public int getLength() {
        if( image == null ) {
            return -1 ;
        }
        return image.length ;
    }
    
    /**
     * イメージ情報を設定.
     * <BR>
     * @param image イメージ情報を設定します.
     */
    public void setImage( byte[] image ) {
        this.image = image ;
    }
    
    /**
     * イメージ情報を取得.
     * <BR>
     * @return byte[] イメージ情報が返されます.
     */
    public byte[] getImage() {
        return this.image ;
    }
    
    /**
     * データ保存.
     */
    protected byte[] save() {
        return save( type,lastTime,image ) ;
    }
    
    /**
     * データ保存.
     */
    protected static final byte[] save( int type,long lastTime,byte[] image ) {
        byte[] ret = new byte[ 12 + image.length ] ;
        int pnt = 0 ;
        ConvertParam.convertInt( ret,pnt,type ) ;
        pnt += 4 ;
        ConvertParam.convertLong( ret,pnt,lastTime ) ;
        pnt += 8 ;
        System.arraycopy( image,0,ret,pnt,image.length ) ;
        return ret ;
    }
    
    /**
     * データロード.
     */
    protected static final ImageData load( byte[] binary ) throws Exception {
        ImageData ret = new ImageData() ;
        int pnt = 0 ;
        ret.type = ConvertParam.convertInt( pnt,binary ) ;
        pnt += 4 ;
        ret.lastTime = ConvertParam.convertInt( pnt,binary ) ;
        pnt += 8 ;
        int len = binary.length - pnt ;
        ret.image = new byte[ len ] ;
        System.arraycopy( binary,pnt,ret.image,0,len ) ;
        return ret ;
    }
}
